package fr.irisa.cairn.model.polymodel.prdg.util;

import static fr.irisa.cairn.model.polymodel.prdg.factory.PRDGUserFactory.createDependenceFunction;
import static fr.irisa.cairn.model.polymodel.prdg.factory.PRDGUserFactory.createPRDG;
import static fr.irisa.cairn.model.polymodel.prdg.factory.PRDGUserFactory.createPRDGEdge;
import static fr.irisa.cairn.model.polymodel.prdg.factory.PRDGUserFactory.createPRDGNode;
import static fr.irisa.cairn.model.integerLinearAlgebra.factory.IntegerExpressionUserFactory.var;
import static fr.irisa.cairn.model.integerLinearAlgebra.factory.IntegerExpressionUserFactory.term;
import static fr.irisa.cairn.model.integerLinearAlgebra.factory.IntegerExpressionUserFactory.linexp;
import static fr.irisa.cairn.model.integerLinearAlgebra.factory.IntegerExpressionUserFactory.linConstraint;
import static fr.irisa.cairn.model.integerLinearAlgebra.factory.IntegerExpressionUserFactory.linConstraintSystem;

import java.util.ArrayList;
import java.util.List;

import fr.irisa.cairn.model.integerLinearAlgebra.IVariable;
import fr.irisa.cairn.model.integerLinearAlgebra.IntExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.IntLinearConstraint;
import fr.irisa.cairn.model.integerLinearAlgebra.IntLinearConstraintSystem;
import fr.irisa.cairn.model.integerLinearAlgebra.IntLinearExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.Operator;
import fr.irisa.cairn.model.integerLinearAlgebra.Scope;
import fr.irisa.cairn.model.integerLinearAlgebra.factory.IntegerExpressionUserFactory;
import fr.irisa.cairn.model.polymodel.AffineMapping;
import fr.irisa.cairn.model.polymodel.PolyhedralDomain;
import fr.irisa.cairn.model.polymodel.factory.PolyModelDefaultFactory;
import fr.irisa.cairn.model.polymodel.prdg.DependenceFunction;
import fr.irisa.cairn.model.polymodel.prdg.PRDG;
import fr.irisa.cairn.model.polymodel.prdg.PRDGEdge;
import fr.irisa.cairn.model.polymodel.prdg.PRDGNode;

/**
 * A random PRDG generator for test purpose. May generate inconsistent PRDG.
 * Generated dependencies are uniform.
 * 
 * @author Antoine Floc'h - Initial contribution and API
 * 
 */
public class PrdgGenerator {
	private PolyModelDefaultFactory factory;
	private int maxDependencyDepth;

	public PrdgGenerator(PolyModelDefaultFactory factory, int maxDependencyDepth) {
		this.factory = factory;
		this.maxDependencyDepth = maxDependencyDepth;
	}

	/**
	 * Generate a PRDG
	 * @param n number of nodes
	 * @param e number of edges
	 * @param dims number indices
	 * @param parameters number of parameters
	 * @return
	 */
	public PRDG generate(int n, int e, int dims, int parameters) {
		Scope scope = IntegerExpressionUserFactory.scope();
		PRDG prdg = createPRDG();
		prdg.setScope(scope);
		
		List<IVariable> dimsVariables = new ArrayList<IVariable>();
		List<IVariable> paramVariables = new ArrayList<IVariable>();
		for (int i = 0; i < dims; i++) {
			IVariable idx = var("i" + i);
			scope.getSymbols().add(idx);
			dimsVariables.add(idx);
		}
		for (int i = 0; i < parameters; i++) {
			IVariable p = var("P" + i);
			scope.getSymbols().add(p);
			paramVariables.add(p);
		}

		for (int i = 0; i < n; i++) {
			PolyhedralDomain domain = generateStatementDomain(dimsVariables,
					paramVariables);
			PRDGNode node = createPRDGNode("s" + i, domain);
			prdg.getNodes().add(node);
		}

		for (int i = 0; i < e; i++) {
			int source = (int) (Math.random() * n);
			int ref = (int) (Math.random() * n);
			PRDGEdge dep = generateUniformDependency(prdg.getNodes()
					.get(source), prdg.getNodes().get(ref));
			prdg.getEdges().add(dep);
		}
		return prdg;
	}

	private PRDGEdge generateUniformDependency(PRDGNode source,
			PRDGNode reference) {
		// Create a random uniform dependence function
		List<IntLinearExpression> expressions = new ArrayList<IntLinearExpression>();
		for (IVariable idx : source.getDomain().getIndices()) {
			int d = (int) (Math.random() * maxDependencyDepth);
			int sign = (int) (Math.random() * 2);
			d = sign < 2 ? -d : d;
			IntLinearExpression uniformDepExp = linexp(term(idx), term(d));
			expressions.add(uniformDepExp);
		}
		AffineMapping mapping = factory.affineMappingFromList(source
				.getDomain().getParams(), source.getDomain().getIndices(),
				expressions);
		DependenceFunction function = createDependenceFunction(mapping);

		// Create a random dependence domain
		PolyhedralDomain dependenceDomain = generateRandomDependencyDomain(
				source.getDomain().getIndices(), source.getDomain().getParams());
		return createPRDGEdge(source, reference, dependenceDomain, function);
	}

	private PolyhedralDomain generateStatementDomain(List<IVariable> indices,
			List<IVariable> parameters) {
		IntLinearConstraintSystem polyhedron = PolyModelDefaultFactory
				.linConstraintSystem();
		int dim = (int) (Math.random()*indices.size());
		for (int i = 0; i < dim; i++) {
			IVariable index = indices.get(i);
			// lower bound
			IntLinearExpression lbe = linexp(term(index));
			polyhedron.getConstraints().add(linConstraint(lbe, Operator.GE));

			// upper bound
			IVariable parameter = random(parameters);
			IntLinearExpression ube = linexp(term(index), term(-1, parameter));
			polyhedron.getConstraints().add(linConstraint(ube, Operator.LE));
		}
		PolyhedralDomain domain = factory.domain(polyhedron, indices,
				parameters);
		return domain;
	}

	private PolyhedralDomain generateRandomDependencyDomain(
			List<IVariable> indices, List<IVariable> parameters) {
		IntLinearConstraintSystem polyhedron = PolyModelDefaultFactory
				.linConstraintSystem();

		PolyhedralDomain domain = factory.domain(polyhedron, indices,
				parameters);
		int nbConstraints = (int) (Math.random() * (indices.size()));

		IntLinearConstraintSystem system = linConstraintSystem();
		for (int i = 0; i < nbConstraints; ++i) {
			IntLinearConstraint randomConstraint = randomConstraint(indices,
					parameters);
			system.getConstraints().add(randomConstraint);
		}

		return domain;
	}

	private IntLinearConstraint randomConstraint(List<IVariable> vars,
			List<IVariable> params) {
		List<IVariable> allVars = new ArrayList<IVariable>(vars);
		allVars.addAll(params);
		int nbTerms = (int) (Math.random() * allVars.size());

		// terms
		List<IntExpression> terms = new ArrayList<IntExpression>();
		for (int i = 0; i < nbTerms; ++i) {
			int var = (int) (Math.random() * allVars.size());
			int coef = (int) (Math.random() * maxDependencyDepth);
			terms.add(term(coef, allVars.get(var)));
		}
		IntLinearExpression expression =  linexp(terms);

		int op = (int) (Math.random() * 6);
		Operator comparator = Operator.get(op);
		return linConstraint(expression, comparator);

	}

	private IVariable random(List<IVariable> vars) {
		int k = (int) (Math.random() * vars.size());
		return vars.get(k);
	}
}
