package fr.irisa.cairn.model.polymodel.ada.factory;

import java.util.HashMap;
import java.util.Map;

import fr.irisa.cairn.model.polymodel.AffineMapping;
import fr.irisa.cairn.model.polymodel.PolyhedralDomain;
import fr.irisa.cairn.model.polymodel.ada.ADAInput;
import fr.irisa.cairn.model.polymodel.ada.AdaFactory;
import fr.irisa.cairn.model.polymodel.ada.CandidateStatement;
import fr.irisa.cairn.model.polymodel.ada.ReadAccess;
import fr.irisa.cairn.model.polymodel.ada.Variable;
import fr.irisa.cairn.model.polymodel.ada.WriteAccess;
import fr.irisa.cairn.model.polymodel.isl.factory.ISLDefaultFactory;

public class ADAUserFactory {
	
	private static final AdaFactory factory = AdaFactory.eINSTANCE;

	public static ADAInput createADAInput() {
		return factory.createADAInput();
	}
	
	/**
	 * Creates candidates statements with a domain and a schedule.
	 * For loop programs, domains correspond to the iteration space surrounding a statement,
	 * and the schedule is used to specify relative ordering between statement instances.
	 * 
	 * @param id
	 * @param domain
	 * @param schedule
	 * @return
	 */
	public static CandidateStatement createCandidateStatement(String id, PolyhedralDomain domain, AffineMapping schedule) {
		CandidateStatement stmt = factory.createCandidateStatement();
		
		stmt.setID(id);
		stmt.setDomain(domain);
		stmt.setSchedule(schedule);
		
		return stmt;
	}
	
	/**
	 * Creates a read access. Accesses are characterized by the variable being accessed, and the index function.
	 * 
	 * @param var
	 * @param accessFunc
	 * @return
	 */
	public static ReadAccess createReadAccess(Variable var, AffineMapping accessFunc) {
		ReadAccess read = factory.createReadAccess();
		
		read.setVariable(var);
		read.setAccessFunction(accessFunc);
		
		return read;
	}

	/**
	 * Creates a write access. Accesses are characterized by the variable being accessed, and the index function.
	 * 
	 * @param var
	 * @param accessFunc
	 * @return
	 */
	public static WriteAccess createWriteAccess(Variable var, AffineMapping accessFunc) {
		WriteAccess write = factory.createWriteAccess();
		
		write.setVariable(var);
		write.setAccessFunction(accessFunc);
		
		return write;
	}
	
	/**
	 * Creates an instance of Variable object, used to distinguish array variables.
	 * 
	 * @param name
	 * @return
	 */
	public static Variable createVariable(String name) {
		Variable var = factory.createVariable();
		
		var.setName(name);
		
		return var;
	}

	/**
	 * Overload to provide String interface.
	 * @see createCandidateStatement(String id, PolyhedralDomain domain, AffineMapping schedule)
	 * 
	 * @param id
	 * @param domainIslStr
	 * @param scheduleIslStr
	 * @return
	 */
	public static CandidateStatement createCandidateStatement(String id, String domainIslStr, String scheduleIslStr) {
		CandidateStatement stmt = factory.createCandidateStatement();
		
		stmt.setID(id);
		stmt.setDomain(ISLDefaultFactory.INSTANCE.polyhedralDomainFromISLString(domainIslStr));
		stmt.setSchedule(ISLDefaultFactory.INSTANCE.affineMappingFromISLString(scheduleIslStr));
		
		return stmt;
	}
	

	/**
	 * Overload to provide String interface.
	 * @see createReadAccess(Variable var, AffineMapping accessFunc)
	 * 
	 * @param var
	 * @param accessFuncIslStr
	 * @return
	 */
	public static ReadAccess createReadAccess(Variable var, String accessFuncIslStr) {
		ReadAccess read = factory.createReadAccess();
		
		read.setVariable(var);
		read.setAccessFunction(ISLDefaultFactory.INSTANCE.affineMappingFromISLString(accessFuncIslStr));
		
		return read;
	}

	/**
	 * Overload to provide String interface.
	 * @see createWriteAccess(Variable var, AffineMapping accessFunc)
	 * 
	 * @param var
	 * @param accessFuncIslStr
	 * @return
	 */
	public static WriteAccess createWriteAccess(Variable var, String accessFuncIslStr) {
		WriteAccess write = factory.createWriteAccess();
		
		write.setVariable(var);
		write.setAccessFunction(ISLDefaultFactory.INSTANCE.affineMappingFromISLString(accessFuncIslStr));
		
		return write;
	}
	
	/**
	 * Creates ADAInput from collection of strings, given as String[][].
	 * Main use of this method should come from test cases and examples.
	 * 
	 * 
	 * @param statements Array of String[] with statementID, domain, schedule in ISL syntax 
	 * @param reads Array of String[] with statementID, variableName, access function in ISL syntax 
	 * @param writes Array of String[] with statementID, variableName, access function in ISL syntax
	 * @return
	 */
	public static ADAInput createADAInputFromStrings(String[][] statements, String[][] reads, String[][] writes) {

		Map<String, Variable> variables = new HashMap<String, Variable>();
		
		ADAInput input = ADAUserFactory.createADAInput();
		
		for (String[] stmt : statements) {
			input.getStatements().add(ADAUserFactory.createCandidateStatement(stmt[0], stmt[1], stmt[2]));
		}
		
		for (String[] read : reads) {
			//add variables when first seen
			if (!variables.containsKey(read[1])) {
				Variable var = ADAUserFactory.createVariable(read[1]);
				variables.put(read[1], var);
				input.getVariables().add(var);
			}
			ReadAccess readAccess = ADAUserFactory.createReadAccess(input.getVariable(read[1]), read[2]);
			input.getStatement(read[0]).getReads().add(readAccess);
			
		}
		
		for (String[] write : writes) {
			//add variables when first seen
			if (!variables.containsKey(write[1])) {
				Variable var = ADAUserFactory.createVariable(write[1]);
				variables.put(write[1], var);
				input.getVariables().add(var);
			}
			

			WriteAccess writeAccess = ADAUserFactory.createWriteAccess(input.getVariable(write[1]), write[2]);
			input.getStatement(write[0]).setWrite(writeAccess);
		}
		
		return input;
	}
	
}
