package fr.irisa.cairn.model.integerLinearAlgebra.adapter;

import java.util.List;

import org.eclipse.emf.common.util.EList;

import tom.library.sl.VisitFailure;
import tom.library.sl.Visitable;
import fr.irisa.cairn.model.integerLinearAlgebra.IntCeilDExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.IntDivExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.IntExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.IntFloorDExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.IntMaxExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.IntMinExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.IntModExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.IntMulExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.IntProdExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.IntReductionExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.IntSumExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.IntTermExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.util.IntegerLinearAlgebraSwitch;

/**
 * Instruction adapter allow use of TOM strategies for iintegerExpression. An adapter
 * contain an instruction and implements {@link Visitable}. When setting
 * children adapters, instructions are copied in order to avoid modification of
 * original instructions.
 * 
 * @author sderrien adapted from afloch
 * 
 */
public class IntExpressionAdapter implements Visitable {
	
	private IntExpression expression;
	private Object[] childrenCache;
	private Visitable[] childrenAdapter;

	public IntExpressionAdapter(IntExpression expr) {
		if(expr==null) throw new RuntimeException("Null pointer");
		this.expression = expr;
	}

	public IntExpression expr() {
		return expression;
	}

	private Object[] getChildrenCache() {
		if (childrenCache == null) {
			childrenCache = ChildrenGetter.INSTANCE.doSwitch(expression);
		}
		return childrenCache;
	}

	private Visitable[] getChildrenAdapters() {
		if (childrenAdapter == null) {
			Object[] children = getChildrenCache();
			if(children!=null) {
				childrenAdapter = new Visitable[children.length];
				for (int i = 0; i < children.length; ++i) {
					if (children[i] instanceof IntExpression) {
						childrenAdapter[i] = new IntExpressionAdapter((IntExpression)children[i]);
					}else {
						childrenAdapter[i] = new IntExpressionListAdapter((EList<IntExpression>)children[i]);
					}
				}
			}
		}
		return childrenAdapter;
	}

	public Visitable getChildAt(int i) {
		return getChildrenAdapters()[i];
	}

	public int getChildCount() {
		return getChildrenCache().length;
	}

	public Visitable[] getChildren() {
		return getChildrenAdapters();
	}

	public Visitable setChildAt(int i, Visitable child) {
		throw new RuntimeException("IntExpressionAdapter.setChildAt("+i+","+child+" for "+this.expression);
	}

	public Visitable setChildren(Visitable[] children) {
		ChildrenSetter setter = new ChildrenSetter(children,this);
		return setter.doSwitch(expression);

	}

	public IntExpressionAdapter accept(IntExpressionStrategy strategy)
			throws VisitFailure {
		return strategy.visit_Instr(this);
	}

	@Override
	public String toString() {
		return "A:" + expression.toString();
	}

	private static class ChildrenGetter extends IntegerLinearAlgebraSwitch<Object[]> {
		private static ChildrenGetter INSTANCE = new ChildrenGetter();

		protected ChildrenGetter() {

		}

		@Override
		public Object[] caseIntReductionExpression(IntReductionExpression object) {
			Object[] childrenCache = new Object[1];
			childrenCache[0] = object.getTerms();
			return childrenCache;
		}

		@Override
		public Object[] caseIntExpression(IntExpression object) {
			return new IntExpression[0];
		}

		@Override
		public Object[] caseIntDivExpression(IntDivExpression object) {
			IntExpression[] childrenCache = new IntExpression[1];
			childrenCache[0] = object.getExpr();
			return childrenCache;
		}

		@Override
		public Object[] caseIntMulExpression(IntMulExpression object) {
			IntExpression[] childrenCache = new IntExpression[1];
			childrenCache[0] = object.getExpr();
			return childrenCache;
		}
		
		
	}

	private static class ChildrenSetter extends IntegerLinearAlgebraSwitch<Visitable> {
		private Visitable[] children;
		private Visitable parent;

		public ChildrenSetter(Visitable[] children,Visitable parent) {
			this.children = children;
			
			this.parent= parent;
		}

		@Override
		public Visitable caseIntReductionExpression(IntReductionExpression instr) {
			//IntReductionExpression copy = (IntReductionExpression) instr.copy();
			if(children.length==0 || children.length>1) throw new UnsupportedOperationException("Not yet implemented");
			if(children[0]!=null) {
				IntExpressionListAdapter listAdapter= (IntExpressionListAdapter) children[0];
				/// FIXME
//				if(parent instanceof IntExpressionAdapter) {
//					listAdapter.setContainer(((IntExpressionAdapter)parent).expression);
//					System.err.println("Setting container "+parent+ " for "+listAdapter);
//				}
				if(listAdapter.list()!=null) {
					List<IntExpression> list = listAdapter.list();
					instr.getTerms().clear();
					for (IntExpression intExpression : list) {
						instr.getTerms().add(intExpression.copy());
					}
				} 
			}
			Visitable adapter = new IntExpressionAdapter(instr);
			return adapter;
		}

		@Override
		public Visitable caseIntDivExpression(IntDivExpression instr) {
			IntDivExpression copy = (IntDivExpression) instr.copy();
			copy.setDenum(instr.getDenum());
			copy.setExpr(instr.getExpr().copy());
			Visitable adapter = new IntExpressionAdapter(copy);
			return adapter;
		}

		@Override
		public Visitable caseIntMulExpression(IntMulExpression instr) {
			IntMulExpression copy = (IntMulExpression) instr.copy();
			copy.setFactor(instr.getFactor());
			copy.setExpr(instr.getExpr().copy());
			Visitable adapter = new IntExpressionAdapter(copy);
			return adapter;
		}
	}

	public boolean isBinaryAdd() {
		if (expr() instanceof IntSumExpression) {
			IntSumExpression sum = (IntSumExpression) expr();
			return sum.getTerms().size()==2;
		}
		return false;
	}


	public boolean isSumReduction() {
		if (expr() instanceof IntSumExpression) {
			return true;
		}
		return false;
	}

	public boolean isProdReduction() {
		if (expr() instanceof IntProdExpression) {
			return true;
		}
		return false;
	}

	public boolean isMaxReduction() {
		if (expr() instanceof IntMaxExpression) {
			return true;
		}
		return false;
	}

	public boolean isMinReduction() {
		if (expr() instanceof IntMinExpression) {
			return true;
		}
		return false;
	}

	public boolean isConstantTerm() {
		if (expr() instanceof IntTermExpression) {
			IntTermExpression sum = (IntTermExpression) expr();
			return sum.getVar()==null;
		}
		return false;
	}

	public boolean isTerm() {
		if (expr() instanceof IntTermExpression) {
			IntTermExpression sum = (IntTermExpression) expr();
			return sum.getVar()!=null;
		}
		return false;
	}

	public boolean isIntMulExpression() {
		if (expr() instanceof IntMulExpression) {
			return true;
		}
		return false;
	}

	public boolean isIntDivExpression() {
		if (expr() instanceof IntDivExpression) {
			return true;
		}
		return false;
	}

	public boolean isIntFloorDExpression() {
		return (expr() instanceof IntFloorDExpression);
	}

	public boolean isIntCeilDExpression() {
		return (expr() instanceof IntCeilDExpression);
	}

	public boolean isIntModExpression() {
		return (expr() instanceof IntModExpression);
	}

	//	public boolean isBinaryAdd() {
//		if (expr() instanceof IntSumExpression) {
//			IntSumExpression sum = (IntSumExpression) expr();
//			return sum.getTerms().size()==2;
//		}
//		return false;
//	}

}
