package fr.irisa.cairn.model.integerLinearAlgebra.factory;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import fr.irisa.cairn.model.integerLinearAlgebra.IVariable;
import fr.irisa.cairn.model.integerLinearAlgebra.IntExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.IntLinearConstraint;
import fr.irisa.cairn.model.integerLinearAlgebra.IntLinearExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.Operator;
import fr.irisa.cairn.model.integerLinearAlgebra.SymbolVariable;

import static fr.irisa.cairn.model.integerLinearAlgebra.factory.IntegerExpressionUserFactory.*;

public class IntExpressionParser {

	private static final int VAR = 100;
	private static final int NUMBER = 101;
	private static final int MUL = 102;
	private static final int ADD = 103;
	private static final int SUB = 104;
	// private static final int GT = 5;
	// private static final int GE = 6;
	// private static final int LT = 7;
	// private static final int LE = 8;
	// private static final int EQ = 9;
	private static final int EOF = 105;

	private Reader in;
	private int lastChar;
	private StringBuffer token;
	private int currentToken;
	private Map<String, IVariable> scope;

	public IntExpressionParser() {
		token = new StringBuffer();
		scope = new HashMap<String, IVariable>();
	}

	private String format(String s) {
		if (s.startsWith("+"))
			s = s.substring(1); 
		return s;
	}
	
	public IntLinearConstraint parseIntLinearConstraint(
			String exprString, SymbolVariable... vars) throws RuntimeException {
		exprString = format(exprString);
		checkUnicity(vars);
		initializeFrom(exprString);
		nextToken();
		return equation();
	}
	
	public IntLinearConstraint parseIntLinearConstraint(String exprString, Map<String, IVariable> varMap) {
		exprString = format(exprString);
		scope = varMap;
		initializeFrom(exprString);
		nextToken();
		return equation();	
	}


	public IntLinearConstraint parseIntLinearConstraint(
			String exprString, List<SymbolVariable> vars)
			throws RuntimeException {
		exprString = format(exprString);
		checkUnicity(vars);
		initializeFrom(exprString);
		nextToken();
		return equation();
	}

	public IntExpression parseIntExpression(String exprString, SymbolVariable... vars) throws RuntimeException {
		exprString = format(exprString);
		checkUnicity(vars);
		initializeFrom(exprString);
		nextToken();
		IntExpression expr = parseExpr();
		return expr;
	}
	
	public IntExpression parseIntExpression(String exprString, Map<String, IVariable> varMap) throws RuntimeException {
		exprString = format(exprString);
		scope = varMap;
		initializeFrom(exprString);
		nextToken();
		IntExpression expr = parseExpr();
		return expr;
	}
	
	public IntLinearExpression parseIntLinearExpression(String exprString, SymbolVariable... vars) throws RuntimeException {
		exprString = format(exprString);
		checkUnicity(vars);
		initializeFrom(exprString);
		nextToken();
		return linexp(parseExpr());
	}
	
	public IntLinearExpression parseIntLinearExpression(String exprString, Map<String, IVariable> varMap) throws RuntimeException {
		exprString = format(exprString);
		scope = varMap;
		initializeFrom(exprString);
		nextToken();
		return linexp(parseExpr());
	}
	

	private void checkUnicity(List<SymbolVariable> vars) {
		for (SymbolVariable sym1 : vars) {
			scope.put(sym1.getName(), sym1);
			for (SymbolVariable sym2 : vars) {
				if (sym2.getName().equals(sym1.getName())) {
					if (sym2 != sym1)
						throw new RuntimeException(
								"Two symbols with the same name in varibale set");
				}
			}
		}
	}

	private void checkUnicity(SymbolVariable[] vars) {
		checkUnicity(Arrays.asList(vars));
	}

	/*
	 * equation -> expr op expr
	 */
	private IntLinearConstraint equation() throws RuntimeException {
		IntExpression lhs = parseExpr();
		int op = currentToken;
		nextToken();
		IntExpression rhs = parseExpr();

		if (op == 2) {
			op = 3;
			rhs=add(rhs,1);
		}
		if (op == 4) {
			op = 3;
			rhs=add(rhs,-1);
			IntExpression temp = lhs;
			lhs = rhs;
			rhs = temp;
		}
		if (op == 5) {
			op = 3;
			IntExpression temp = lhs;
			lhs = rhs;
			rhs = temp;
		}
		lhs=IntegerExpressionUserFactory.sub(lhs,rhs);
		
		IntLinearExpression simplify = linexp(lhs.simplify());
		return IntegerExpressionUserFactory.linConstraint(simplify, Operator.get(op));
	}



	/*
	 * expr -> term { ('+' | '-') term }
	 */
	private IntExpression parseExpr() throws RuntimeException {
		IntExpression expr = parseTerm();
		while (currentToken == ADD || currentToken == SUB) {
			int op = currentToken;
			nextToken();
			IntExpression term = parseTerm();
			if (op == SUB) {
				term=mul(-1,term);
			}
			expr=add(expr,term);
		}
		return (IntExpression) expr.simplify();
	}

	/*
	 * term -> atom { ('*'|%EMPTY) atom } atom -> number | value
	 */
	private IntExpression parseTerm() throws RuntimeException {
		IVariable var = null;
		int value = 1;
		boolean negative = false;
		do {
			if (currentToken == MUL)
				nextToken();
			if (currentToken == SUB) {
				negative = !negative;
				nextToken();
			}
			if (currentToken == VAR) {

				if (scope.containsKey(token.toString()))
					var = scope.get(token.toString());
				else
					throw new RuntimeException("Unknown variable "+ token.toString());
				nextToken();
			} else if (currentToken == NUMBER) {
				value = Integer.parseInt(token.toString());
				nextToken();
			}
		} while (currentToken == MUL || currentToken == VAR
				|| currentToken == NUMBER);

		if(negative) value=-value;
		IntExpression term = IntegerExpressionUserFactory.term(value,var);
		return term;
	}

	private void initializeFrom(String exprString) {
		in = new StringReader(exprString);
		lastChar = -1;
	}

	private void nextToken() throws RuntimeException {
		currentToken = readNextToken();
	}

	private int readNextToken() throws RuntimeException {
		while (true) {
			int c = getChar();
			switch (c) {
			case -1:
				return EOF;
			case '*':
				return MUL;
			case '+':
				return ADD;
			case '-':
				return SUB;
			case '=':
				return Operator.EQ.ordinal();
			case '>': {
				int r = getChar();
				if (r == '=')
					return Operator.GE.ordinal();
				lastChar = r;
				return Operator.GT.ordinal();
			}
			case '<': {
				int r = getChar();
				if (r == '=')
					return Operator.LE.ordinal();
				lastChar = r;
				return Operator.LT.ordinal();
			}
			default:
				if (Character.isDigit(c))
					return handleNumber(c);
				if (Character.isLetter(c))
					return handleVar(c);
			}
		}
	}

	private int handleVar(int c) throws RuntimeException {
		token.setLength(0);
		token.append((char) c);
		c = getChar();
		while (Character.isLetter(c) || Character.isJavaIdentifierPart(c) || Character.isDigit(c)) {
			token.append((char) c);
			c = getChar();
		}
		lastChar = c;
		return VAR;
	}

	private int handleNumber(int c) throws RuntimeException {
		token.setLength(0);
		token.append((char) c);
		while (Character.isDigit(c = getChar())) {
			token.append((char) c);
		}
		lastChar = c;
		return NUMBER;
	}

	private int getChar() throws RuntimeException {
		if (lastChar >= 0) {
			int c = lastChar;
			lastChar = -1;
			return c;
		}
		try {
			return in.read();
		} catch (IOException e) {
			throw new RuntimeException(e.getMessage());
		}
	}

}
