package fr.irisa.cairn.model.polymodel.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import fr.irisa.cairn.model.integerLinearAlgebra.IVariable;
import fr.irisa.cairn.model.integerLinearAlgebra.IntConstraint;
import fr.irisa.cairn.model.integerLinearAlgebra.IntConstraintSystem;
import fr.irisa.cairn.model.integerLinearAlgebra.IntLinearConstraint;
import fr.irisa.cairn.model.integerLinearAlgebra.IntLinearConstraintSystem;
import fr.irisa.cairn.model.integerLinearAlgebra.IntLinearExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.IntTermExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.Scope;
import fr.irisa.cairn.model.integerLinearAlgebra.factory.IntExpressionParser;
import fr.irisa.cairn.model.integerLinearAlgebra.factory.IntegerExpressionUserFactory;
import fr.irisa.cairn.model.polymodel.AffineMapping;
import fr.irisa.cairn.model.polymodel.AffineMappingWithScope;
import fr.irisa.cairn.model.polymodel.IndexDimension;
import fr.irisa.cairn.model.polymodel.Matrix;
import fr.irisa.cairn.model.polymodel.MatrixRow;
import fr.irisa.cairn.model.polymodel.PolyhedralDomain;
import fr.irisa.cairn.model.polymodel.PolyhedralDomainWithScope;
import fr.irisa.cairn.model.polymodel.PolymodelFactory;

/**
 * Default user polymodel factory. Provide user friendly functions to create
 * polyhedral objects.
 * 
 * Convention : 
 * polyhedralDomain creates domains with scopes,
 * domain creates domains without scopes
 * 
 * @author steven, antoine
 * 
 */
public class PolyModelDefaultFactory extends IntegerExpressionUserFactory {

	protected PolymodelFactory factory;

	/**
	 * Default constructor: use EMF factory.
	 */
	public PolyModelDefaultFactory() {
		this(PolymodelFactory.eINSTANCE);
	}

	/**
	 * Constructor using a given factory to create polyhedral objects.
	 * 
	 * @param factory
	 */
	public PolyModelDefaultFactory(PolymodelFactory factory) {
		this.factory = factory;
	}

	/**
	 * Helper function that constructs the varmap used to parse constraints.
	 * List of ids given as String is mapped to IVariable, and new dimensions
	 * are created if it does not match any of the params/indices in the
	 * outerDomain
	 * 
	 * @param ids
	 * @param outerDomain
	 * @param newvars
	 * @return
	 */
	protected Map<String, IVariable> constructVarMapFromString(List<String> params, List<String> ids) {
		Map<String, IVariable> varMap = new HashMap<String, IVariable>();

		for (int i = 0; i < params.size(); i++) {
			IndexDimension var = createIndexDimension(i, params.get(i));
			if (varMap.containsKey(params.get(i))) {
				throw new RuntimeException("Duplicate names in domain parameters/indices");
			}
			varMap.put(params.get(i), var);
		}
		for (int i = 0; i < ids.size(); i++) {
			IndexDimension var = createIndexDimension(i+params.size(), ids.get(i));
			if (varMap.containsKey(ids.get(i))) {
				throw new RuntimeException("Duplicate names in domain parameters/indices");
			}
			varMap.put(ids.get(i), var);
		}


		return varMap;
	}

	protected Map<String, IVariable> constructVarMapFromIVariable(List<? extends IVariable> params, List<? extends IVariable> indices) {
		Map<String, IVariable> varMap = new HashMap<String, IVariable>();

		for (IVariable iv : params) {
			if (varMap.containsKey(iv.toString())) {
				throw new RuntimeException("Duplicate names in domain parameters/indices");
			}
			if (iv instanceof IndexDimension) {
				varMap.put(iv.toString(), iv.copy());
			} else {
				varMap.put(iv.toString(), iv);
			}
		}
		for (IVariable iv : indices) {
			if (varMap.containsKey(iv.toString())) {
				throw new RuntimeException("Duplicate names in domain parameters/indices : " + iv);
			}
			if (iv instanceof IndexDimension) {
				varMap.put(iv.toString(), iv.copy());
			} else {
				varMap.put(iv.toString(), iv);
			}
		}

		return varMap;
	}
	
	/**
	 * Helper for result of constructVarMapFromIVariable to ensure IVariables are within Scope.
	 * The given list of IVariable is the IVariable used to construct the varMap.
	 * 
	 * @param indices
	 * @param varMap
	 * @return
	 */
	protected List<IVariable> retrieveSelectedIndices(List<? extends IVariable> indices, Map<String, ? extends IVariable> varMap) {
		List<IVariable> list = new LinkedList<IVariable>();
		
		for (IVariable id : indices) {
			if (varMap.containsKey(id.toString())) {
				list.add(varMap.get(id.toString()));
			}
		}
		
		return list;
	}

//	/**
//	 * Returns list of variables used sorted from first to last dimension.
//	 * 
//	 * @param varMap
//	 * @return
//	 */
//	protected List<? extends IVariable> getSortedVariableListFromVarMap(Map<String, IVariable> varMap) {
//		// Get sorted list of dimensions
//		List<IndexDimension> vars = new LinkedList<IndexDimension>();
//		for (IVariable val : varMap.values()) {
//			// Currently assumes IndexDimension
//			vars.add((IndexDimension) val);
//		}
//		Collections.sort(vars);
//
//		return vars;
//	}

	/**
	 * Helper function to parse constraints
	 * 
	 * @param constraints
	 * @param varMap
	 * @return
	 */
	protected List<IntLinearConstraint> parseConstraints(List<String> constraints, Map<String, IVariable> varMap) {
		List<IntLinearConstraint> consts = new LinkedList<IntLinearConstraint>();
		for (String constraint : constraints) {
			IntLinearConstraint constr = IntegerExpressionUserFactory.constraint(constraint, varMap);
			consts.add(constr);
		}
		return consts;
	}

	protected List<IntLinearConstraint> parseConstraints(String[] constraints, Map<String, IVariable> varMap) {
		List<IntLinearConstraint> consts = new LinkedList<IntLinearConstraint>();
		for (String constraint : constraints) {
			IntLinearConstraint constr = IntegerExpressionUserFactory.constraint(constraint, varMap);
			consts.add(constr);
		}
		return consts;
	}
	
	
	/**
	 * Constructs a polyhedral domain using the list of indices and constraints given.
	 * Assumes that the first numParams ids are parameters. 
	 * 
	 * @param ids
	 * @param numParams
	 * @param constraints
	 * @return
	 */
	public PolyhedralDomain polyhedralDomainFromString(List<String> ids, int numParams, List<String> constraints) {
		return polyhedralDomainFromString(ids.subList(0, numParams), ids.subList(numParams, ids.size()), constraints);
	}

	/**
	 * Constructs a domain with given params/indices/constraints.
	 * 
	 * @param params
	 * @param ids
	 * @param constraints
	 * @return
	 */
	public PolyhedralDomain polyhedralDomainFromString(List<String> params, List<String> ids, List<String> constraints) {
		// Create indexDimension for each id
		Map<String, IVariable> varMap = constructVarMapFromString(params, ids);
		// Parse each constraint
		List<IntLinearConstraint> consts = parseConstraints(constraints, varMap);

		//Initialize parameter and index list
		List<IVariable> ps = new LinkedList<IVariable>();
		List<IVariable> is = new LinkedList<IVariable>();
		for (String p : params) {
			ps.add(varMap.get(p));
		}
		for (String i : ids) {
			is.add(varMap.get(i));
		}

		PolyhedralDomainWithScope domain = factory.createPolyhedralDomainWithScope();
		domain.getParams().addAll(ps);
		domain.getIndices().addAll(is);
		//Scope
		domain.setScope(IntegerExpressionUserFactory.scope());
		domain.getScope().getSymbols().addAll(ps);
		domain.getScope().getSymbols().addAll(is);
		
		domain.getPolyhedra().add(IntegerExpressionUserFactory.linearConstraintSystem(consts));

		return domain;
	}
	
	public PolyhedralDomain polyhedralDomainUnionFromString(List<String> ids, int numParams, List<List<String>> constraintsList) {
		return polyhedralDomainUnionFromString(ids.subList(0, numParams), ids.subList(numParams, ids.size()), constraintsList);
	}
	
	public PolyhedralDomain polyhedralDomainUnionFromString(List<String> params, List<String> ids, List<List<String>> constraintsList) {
		// Create indexDimension for each id
		Map<String, IVariable> varMap = constructVarMapFromString(params, ids);

		//Initialize parameter and index list
		List<IVariable> ps = new LinkedList<IVariable>();
		List<IVariable> is = new LinkedList<IVariable>();
		for (String p : params) {
			ps.add(varMap.get(p));
		}
		for (String i : ids) {
			is.add(varMap.get(i));
		}

		PolyhedralDomainWithScope domain = factory.createPolyhedralDomainWithScope();
		domain.getParams().addAll(ps);
		domain.getIndices().addAll(is);
		//Scope
		domain.setScope(IntegerExpressionUserFactory.scope());
		domain.getScope().getSymbols().addAll(varMap.values());
		
		// Parse each constraint
		for (List<String> constraints : constraintsList) {
			List<IntLinearConstraint> consts = parseConstraints(constraints, varMap);
			domain.getPolyhedra().add(IntegerExpressionUserFactory.linearConstraintSystem(consts));
			
		}
		
		return domain;
	}
	
	public PolyhedralDomain polyhedralDomain(List<? extends IVariable> params, List<? extends IVariable> indices, List<List<String>> constraintList) {
		// Create indexDimension for each id
		Map<String, IVariable> varMap = constructVarMapFromIVariable(params, indices);

		PolyhedralDomainWithScope domain = factory.createPolyhedralDomainWithScope();
		domain.getParams().addAll(retrieveSelectedIndices(params, varMap));
		domain.getIndices().addAll(retrieveSelectedIndices(indices, varMap));
		//Scope
		domain.setScope(IntegerExpressionUserFactory.scope());
		domain.getScope().getSymbols().addAll(varMap.values());
		

		// Parse each constraint
		if (constraintList != null) {
			for (List<String> constraints : constraintList) {
				domain.getPolyhedra().add(IntegerExpressionUserFactory.linearConstraintSystem(parseConstraints(constraints, varMap)));
			}
		}

		return domain;
		
	}

	// public PolyhedralDomain polyhedralDomain(List<String> ids, List<String>
	// constraints, PolyhedralDomain outerDomain) {
	// //Create indexDimension for each id
	// Map<String, IVariable> varMap = constructVarMap(ids, outerDomain);
	// //Parse each constraint
	// List<IntLinearConstraint> consts = parseConstraints(constraints, varMap);
	// //Get list of indices
	// List<? extends IVariable> vars = getSortedVariableListFromVarMap(varMap);
	// //Remove parameters
	// List<IVariable> indices = new LinkedList<IVariable>();
	// for (int i = outerDomain.getParams().size(); i < vars.size(); i++) {
	// indices.add(vars.get(i));
	// }
	// //Get new indices
	// List<IVariable> newIndices = new LinkedList<IVariable>();
	// for (int i =
	// outerDomain.getParams().size()+outerDomain.getIndices().size(); i <
	// vars.size(); i++) {
	// newIndices.add(vars.get(i));
	// }
	//
	// //Create domain object and set all values
	// PolyhedralDomainWithScope domain =
	// factory.createPolyhedralDomainWithScope();
	// domain.setScope(IntegerExpressionUserFactory.scope());
	// domain.getScope().getSymbols().addAll(newIndices);
	// domain.getParams().addAll(outerDomain.getParams());
	// domain.getIndices().addAll(indices);
	// //Constraints of this polyhedron
	// IntLinearConstraintSystem polyhedron =
	// IntegerExpressionUserFactory.linearConstraintSystem(consts);
	// //Constraints in outer domain is contained by outer domain, so need to
	// copy
	// for (IntLinearConstraintSystem ilcs : outerDomain.getPolyhedra()) {
	// //One for each polyhedron in outer domain
	// IntLinearConstraintSystem poly = polyhedron.copy();
	// List<IntConstraint> addList = new LinkedList<IntConstraint>();
	// //Add all constraints that are not obviously redundant
	// for (IntConstraint ilc : ilcs.getConstraints()) {
	// boolean redundant = false;
	// for (IntConstraint exilc : poly.getConstraints()) {
	// //TODO should use more precise equivalence check than string comparison
	// if (ilc.toString().compareTo(exilc.toString()) == 0) {
	// redundant = true;
	// break;
	// }
	//
	// }
	// if (!redundant) {
	// addList.add(ilc.copy());
	// }
	// }
	// poly.getConstraints().addAll(addList);
	// domain.getPolyhedra().add(poly);
	// }
	//
	// return domain;
	// }

	public PolyhedralDomain polyhedralDomain() {
		PolyhedralDomainWithScope res = factory.createPolyhedralDomainWithScope();
		Scope s = IntegerExpressionUserFactory.factory.createScope();
		//init lists with EMF getter.
		res.getIndices();
		res.getParams();
		res.getPolyhedra();
		s.getSymbols();
		res.setScope(s);
		return res;
	}
	public PolyhedralDomain polyhedralDomain(IntLinearConstraintSystem... systems) {
		PolyhedralDomainWithScope domain = factory.createPolyhedralDomainWithScope();
		//Scope
		domain.setScope(IntegerExpressionUserFactory.scope());
		for (IntLinearConstraintSystem polyhedron : systems) {
			domain.getPolyhedra().add(polyhedron);
		}
		return domain;
	}

	public PolyhedralDomain domain() {
		PolyhedralDomain createPolyhedralDomain = factory.createPolyhedralDomain();
		return createPolyhedralDomain;
	}

	public Matrix matrix(long[][] array) {
		Matrix mat = factory.createMatrix();
		int width = -1;
		for (long[] row : array) {
			MatrixRow res = factory.createMatrixRow();
			if (width == -1) {
				width = row.length;
			} else {
				if (width != row.length) {
					throw new UnsupportedOperationException("Cannot buil Matrix from variable size rows");
				}
			}
			for (long elt : row) {
				res.getValues().add((long) elt);
			}
			mat.getRows().add(res);
		}
		return mat;
	}

	public Matrix matrix(int nbRows, int nbCols) {
		Matrix mat = factory.createMatrix();
		for (int row = 0; row < nbRows; row++) {
			MatrixRow res = factory.createMatrixRow();
			for (int col = 0; col < nbCols; col++) {
				res.getValues().add((long) 0);
			}
			mat.getRows().add(res);
		}
		return mat;
	}

	public PolyhedralDomain domain(List<IntLinearConstraintSystem> systems) {
		PolyhedralDomain union = factory.createPolyhedralDomain();
		for (IntLinearConstraintSystem system : systems) {
			union.getPolyhedra().add(system);
		}
		return union;
	}

	public PolyhedralDomain domain(List<IntLinearConstraintSystem> systems, List<? extends IVariable> vars, List<? extends IVariable> params) {
		PolyhedralDomain union = factory.createPolyhedralDomain();
		for (IntLinearConstraintSystem system : systems) {
			union.getPolyhedra().add(system);
		}
		union.getIndices().addAll(vars);
		union.getParams().addAll(params);
		return union;
	}

	public PolyhedralDomain domain(List<IntLinearConstraintSystem> systems, List<? extends IVariable> vars, List<? extends IVariable> exists,List<? extends IVariable> params) {
		PolyhedralDomain union = factory.createPolyhedralDomain();
		for (IntLinearConstraintSystem system : systems) {
			union.getPolyhedra().add(system);
		}
		union.getIndices().addAll(vars);
		union.getExistential().addAll(exists);
		union.getParams().addAll(params);
		return union;
	}

	public PolyhedralDomain domain(List<? extends IVariable> vars, List<? extends IVariable> params) {
		PolyhedralDomain union = factory.createPolyhedralDomain();
		union.getIndices().addAll(vars);
		union.getParams().addAll(params);
		return union;
	}

	public PolyhedralDomain domain(IntLinearConstraintSystem system, List<? extends IVariable> vars, List<? extends IVariable> params) {
		PolyhedralDomain union = factory.createPolyhedralDomain();
		union.getPolyhedra().add(system);
		union.getIndices().addAll(vars);
		union.getParams().addAll(params);
		return union;
	}

	public PolyhedralDomain domain(IntLinearConstraintSystem system, List<? extends IVariable> vars, int nbparam) {
		int toIndex = vars.size() - nbparam;
		List<? extends IVariable> indices = vars.subList(0, toIndex);
		if (nbparam == 0) {
			return domain(system, indices, new ArrayList<IVariable>());
		} else {
			List<? extends IVariable> params = vars.subList(toIndex, vars.size());
			return domain(system, indices, params);
		}
	}

	public PolyhedralDomain domainFromMatrix(List<Matrix> matrices, List<? extends IVariable> vars, List<? extends IVariable> params) {
		List<IVariable> merged = new ArrayList<IVariable>();
		merged.addAll(params);
		merged.addAll(vars);
		
		PolyhedralDomain union = factory.createPolyhedralDomain();
		for (Matrix system : matrices) {
			IntLinearConstraintSystem constraintsSystem = system.toConstraintsSystem(merged);
			union.getPolyhedra().add(constraintsSystem);
		}
		union.getIndices().addAll(vars);
		union.getParams().addAll(params);
		return union;
	}

	public PolyhedralDomain domainFromMatrix(List<Matrix> matrices, List<? extends IVariable> vars, int nbparam) {
		int toIndex = vars.size() - nbparam;
		List<? extends IVariable> indices = vars.subList(0, toIndex);
		if (nbparam == 0) {
			return domainFromMatrix(matrices, indices, new ArrayList<IVariable>());
		} else {
			List<? extends IVariable> params = vars.subList(toIndex, vars.size());
			return domainFromMatrix(matrices, indices, params);
		}
	}

	public IndexDimension createIndexDimension(int dim, String name) {
		IndexDimension id = factory.createIndexDimension();

		id.setName(name);
		id.setDimension(dim);

		return id;
	}
	
	public AffineMapping createIdentityMapping(PolyhedralDomain domain) {
		List<String> exprs = new LinkedList<String>();
		for (IVariable iv : domain.getIndices()) {
			exprs.add(iv.toString());
		}
		return affineMapping(domain.getParams(), domain.getIndices(),exprs);
	}
	
	public AffineMapping affineMappingFromString(List<String> ids, int numParams, List<String> exprs) {
		return affineMappingFromString(ids.subList(0, numParams), ids.subList(numParams, ids.size()), exprs);
	}
	
	public AffineMapping affineMappingFromString(List<String> params, List<String> indices, List<String> exprs) {
		IntExpressionParser parser = new IntExpressionParser();
		// Create indexDimension for each id
		Map<String, IVariable> varMap = constructVarMapFromString(params, indices);

		//Initialize parameter and index list
		List<IVariable> ps = new LinkedList<IVariable>();
		List<IVariable> is = new LinkedList<IVariable>();
		for (String p : params) {
			ps.add(varMap.get(p));
		}
		for (String i : indices) {
			is.add(varMap.get(i));
		}

		// Create and initialize scope
		AffineMappingWithScope map = factory.createAffineMappingWithScope();
		map.setScope(IntegerExpressionUserFactory.scope());
		map.getScope().getSymbols().addAll(ps);
		map.getScope().getSymbols().addAll(is);
		// Initialize map with params/indices
		map.getParams().addAll(ps);
		map.getIndices().addAll(is);

		// Then parse and add expressions
		// Parse and add expressions for each dimension of the RHS
		for (String expr : exprs) {
			// skip null dimension
			if (expr.length() == 0)
				continue;
			// System.out.println(expr + " : " + func);
			map.getFunctions().add(parser.parseIntLinearExpression(expr, varMap));
		}
		return map;
	}
	
	public AffineMapping affineMappingFromList(List<IVariable> params, List<IVariable> indices, List<IntLinearExpression> exprs) {
		Map<String, IVariable> varMap = constructVarMapFromIVariable(params, indices);

		// Create and initialize scope
		AffineMapping map = factory.createAffineMapping();
		// Initialize map with params/indices
		map.getParams().addAll(retrieveSelectedIndices(params, varMap));
		map.getIndices().addAll(retrieveSelectedIndices(indices, varMap));

		// Then parse and add expressions
		// Parse and add expressions for each dimension of the RHS
		for (IntLinearExpression expr : exprs) {
			// System.out.println(expr + " : " + func);
			map.getFunctions().add((IntLinearExpression)expr.copy());
		}
		return map;
	}

	public AffineMapping affineMapping(List<IVariable> params, List<IVariable> indices, List<String> exprs) {
		IntExpressionParser parser = new IntExpressionParser();
		Map<String, IVariable> varMap = constructVarMapFromIVariable(params, indices);

		// Create and initialize scope
		AffineMappingWithScope map = factory.createAffineMappingWithScope();
		map.setScope(IntegerExpressionUserFactory.scope());
		map.getScope().getSymbols().addAll(varMap.values());
		// Initialize map with params/indices
		map.getParams().addAll(retrieveSelectedIndices(params, varMap));
		map.getIndices().addAll(retrieveSelectedIndices(indices, varMap));

		// Then parse and add expressions
		// Parse and add expressions for each dimension of the RHS
		for (String expr : exprs) {
			// skip null dimension
			if (expr.length() == 0)
				continue;
			// System.out.println(expr + " : " + func);
			map.getFunctions().add(parser.parseIntLinearExpression(expr, varMap));
		}
		return map;
	}
	
	public AffineMapping affineMappingFromMatrix(Matrix matrix, List<IVariable> params, List<IVariable> names) {
		//Parameter names are unchanged
		List<String> indices = new LinkedList<String>();
		for (IVariable iv : params) {
			indices.add(iv.toString());
		}
		//Use given names for new indices
		if (names != null) {
			for (IVariable name : names) {
				indices.add(name.toString());
			}
		}	
		while (indices.size() < matrix.getNbCols()-1) {
			indices.add("inv"+indices.size());
		}
		
		List<String> exprs = new LinkedList<String>();
		//Skip the row for parameters TODO
		for (MatrixRow row : matrix.getRows().subList(params.size(), matrix.getNbRows())) {
			StringBuffer expr = new StringBuffer(row.get(row.getSize()-1)+""); //constant
			//Stop before constant
			for (int c = 0; c < row.getSize()-1; c++) {
				//Positive
				if (row.get(c) > 0) {
					expr.append("+");
				//Negative
				} else if (row.get(c) < 0) {
					expr.append("-");
				//Skip 0
				} else {
					continue;
				}
				if (Math.abs(row.get(c))>1) {
					expr.append(row.get(c)+"");
				}
				expr.append(indices.get(c));
			}
			exprs.add(expr.toString());
		}
		
		return affineMappingFromString(indices, params.size(), exprs);
	}

	public PolyhedralDomain polyhedralDomainWithScope(List<IVariable> indexes, List<IVariable> params) {
		Scope s = IntegerExpressionUserFactory.factory.createScope();
		s.getSymbols().addAll(indexes);
		s.getSymbols().addAll(params);
		
		PolyhedralDomainWithScope p = factory.createPolyhedralDomainWithScope();
		p.setScope(s);
		p.getIndices().addAll(indexes);
		p.getParams().addAll(params);
		p.getPolyhedra().add(linConstraintSystem());
		return p;
	}

	/**
	 * create a new polyhedron (create new variables, copy constraints & replace references to constraints).
	 * @param resSys
	 * @param vars
	 * @param params
	 * @return
	 */
	public PolyhedralDomain polyhedralDomainFromNonAffineWithScope(List<IntConstraintSystem> resSys, List<IVariable> vars,	List<IVariable> params) {
		
		return null;
	}
	/**
	 * create a new polyhedron (create new variables, copy constraints & replace references to constraints).
	 * @param resSys
	 * @param vars
	 * @param params
	 * @return
	 */
	public PolyhedralDomain polyhedralDomainWithScope(List<IntConstraintSystem> resSys, List<IVariable> vars,	List<IVariable> params) {

		Map<String, IVariable> substitution = new HashMap<String, IVariable>(vars.size()+params.size());
		
		List<IVariable> indexes = new ArrayList<IVariable>(vars.size());
		for (IVariable v : vars) {
			IVariable copy = v.copy();
			indexes.add(copy);
			substitution.put(copy.toString(), copy);
		}
		List<IVariable> newParams = new ArrayList<IVariable>(params.size());
		for (IVariable v : params) {
			IVariable copy = v.copy();
			newParams.add(copy);
			substitution.put(copy.toString(), copy);
		}
		
		Scope s = IntegerExpressionUserFactory.factory.createScope();
		s.getSymbols().addAll(indexes);
		s.getSymbols().addAll(newParams);
		
		PolyhedralDomainWithScope p = factory.createPolyhedralDomainWithScope();
		p.setScope(s);
		p.getIndices().addAll(indexes);
		p.getParams().addAll(newParams);
		
		for (IntConstraintSystem non_affine_sys : resSys) {
			IntLinearConstraintSystem newSys = IntegerExpressionUserFactory.linConstraintSystem();

			IntLinearConstraintSystem sys = non_affine_sys.normalize();
			
			for (IntConstraint constraint : sys.getConstraints()) {
				if(!constraint.isAffine()) {
					throw new UnsupportedOperationException(
							"Not yet implemented");
				}
				IntLinearConstraint nc = (IntLinearConstraint) constraint.copy();
				for (IntTermExpression term : nc.getLinexpr().getSimpleTerms()) {
					IVariable var = term.getVar();
					if (var != null) {
						if (!substitution.containsKey(var.toString())) {
							IVariable copy = var.copy();
							s.getSymbols().add(copy);
							p.getIndices().add(copy);
							substitution.put(copy.toString(), copy);
						}
						term.setVar(substitution.get(var.toString()));
					}
				}
				newSys.getLinearConstraints().add(nc);
			}
			p.getPolyhedra().add(newSys);
		}
		
		return p;
	}

	public PolyhedralDomain polyhedralDomainWithScopeNoSubst(List<IntLinearConstraintSystem> resSys, Collection<? extends IVariable> indices, Collection<? extends IVariable> params) {
		PolyhedralDomainWithScope res = factory.createPolyhedralDomainWithScope();
		Scope s = IntegerExpressionUserFactory.factory.createScope();
		res.setScope(s);
		s.getSymbols().addAll(indices);
		s.getSymbols().addAll(params);
		
		res.getIndices().addAll(indices);
		res.getParams().addAll(params);
		
		res.getPolyhedra().addAll(resSys);
		return res;
		
		
	}

}
