package fr.irisa.cairn.model.polymodel.util;

import java.util.LinkedList;
import java.util.List;

import org.eclipse.emf.ecore.util.EcoreUtil;

import fr.irisa.cairn.model.integerLinearAlgebra.IVariable;
import fr.irisa.cairn.model.integerLinearAlgebra.Operator;
import fr.irisa.cairn.model.integerLinearAlgebra.factory.IntegerExpressionUserFactory;
import fr.irisa.cairn.model.polymodel.AffineMapping;
import fr.irisa.cairn.model.polymodel.PolyhedralDomain;
import fr.irisa.cairn.model.polymodel.PolyhedralDomainWithScope;
import fr.irisa.cairn.model.polymodel.factory.PolyModelDefaultFactory;

/**
 * Some methods to ease formulation of PIP problems in polyhedralDomain. originally from polyhedralIR
 * 
 * @author yuki
 *
 */
public class DomainOperations {
	
	protected final PolyModelDefaultFactory factory;
	
	public DomainOperations(PolyModelDefaultFactory factory) {
		this.factory = factory;
	}


	/**
	 * Given two domains, domA {P,z|constraints(z)} and domB{P,z'|constraints(z')}, 
	 * returns a new domain : {P,z,z'|constraints(z) + constraints(z')}
	 * 
	 * @param domA
	 * @param domB
	 * @return
	 */
	public PolyhedralDomain mergeDomains(PolyhedralDomain domA, PolyhedralDomain domB) {
		PolyhedralDomain domRes = EcoreUtil.copy(domA);
		//Copy domA add dim(z') new dimensions
		for (int i = 0; i < domB.getIndices().size(); i++) {
			IVariable newIndex = factory.createIndexDimension(domA.getNParams()+domA.getNIndices()+i, "zp"+i);
			domRes.getIndices().add(newIndex);
			//If the domain is with scope, add the new dimension to the scope as well
			if (domRes instanceof PolyhedralDomainWithScope) {
				((PolyhedralDomainWithScope)domRes).getScope().getSymbols().add(newIndex);
			}
		}

		//Copy domB add dim(z) new dimensions
		PolyhedralDomain domBex = EcoreUtil.copy(domB);
		for (int i = 0; i < domA.getIndices().size(); i++) {
			IVariable newIndex = factory.createIndexDimension(domB.getNParams()+domB.getNIndices()+i, "zp"+i);
			domBex.getIndices().add(newIndex);
			//If the domain is with scope, add the new dimension to the scope as well
			if (domBex instanceof PolyhedralDomainWithScope) {
				((PolyhedralDomainWithScope)domBex).getScope().getSymbols().add(newIndex);
			}
		}
		//Create function that takes z'+z to z+z' space
		List<String> exprs = new LinkedList<String>();
		for (int i = 0; i < domA.getIndices().size(); i++) {
			exprs.add("zp"+i);
		}
		for (IVariable iv : domB.getIndices()) {
			exprs.add(iv.toString());
		}
		
		AffineMapping map = factory.affineMapping(domBex.getParams(), domBex.getIndices(), exprs);
		domBex = domBex.image(map);

		//take intersection of the two
		return domRes.intersection(domBex);
	}
	

	/**
	 * Expects an inputDomain for two sets of indices and parameter : P, z, z'
	 * and add constraints of the following form
	 * f1(z) op f2(z')
	 * 
	 * @param inputDom
	 * @param f1
	 * @param f2
	 * @param op
	 * @return
	 */
	public void addConstraintsRelatingTwoSetsOfIndices(PolyhedralDomain inputDom, Constraint constraint) {
		//check on inputs
		if (constraint.f1.getDimRHS() != constraint.f2.getDimRHS()) {
			throw new RuntimeException("Dimensionality of RHS of the two functions given must be equal.");
		}
		if (constraint.f1.getIndices().size() + constraint.f2.getIndices().size() != inputDom.getIndices().size()) {
			throw new RuntimeException("Number of indices used in two functions given must match the number of indices in input domain.");
		}

		//create two functions to relate indices on input domain with the two functions given
		//Input domain has dimensions for P + z + z', so we need :
		//  fa(z+z') = z and fb(z+z') = z', so that it can be composed with the two functions given, which is :
		//  f1(z) = ? and f2(z') = ?

		List<String> indexNames = new LinkedList<String>();
		for (IVariable iv : inputDom.getIndices()) {
			indexNames.add(iv.toString());
		}
		//First function, sublist of 0 to numIndices(f1) -- sublist is inclusive to exclusive
		List<String> sublistA = indexNames.subList(0, constraint.f1.getIndices().size());
		AffineMapping fa = factory.affineMapping(inputDom.getParams(), inputDom.getIndices(), sublistA);
		
		//Second function, sublist of numIndices(f1) to numIndices(f2)
		List<String> sublistB = indexNames.subList(constraint.f1.getIndices().size(), constraint.f1.getIndices().size() + constraint.f2.getIndices().size());
		AffineMapping fb = factory.affineMapping(inputDom.getParams(), inputDom.getIndices(), sublistB);

		AffineMapping domToZ = constraint.f1.compose(fa);
		AffineMapping domToZp = constraint.f2.compose(fb);
		
		//Create IntLinearConstraint for each dimension on the RHS
		//relate the result of the compose functions to each other using the given operator
		for (int d = 0; d < constraint.f1.getDimRHS(); d++) {
			inputDom.addConstraint(
					IntegerExpressionUserFactory.linConstraint(
							IntegerExpressionUserFactory.linexp(
									IntegerExpressionUserFactory.sub(
											domToZ.getFunctions().get(d), domToZp.getFunctions().get(d))), constraint.op));
		}
	}
	
	/**
	 * Sub class for specifying list of constraints of the form f1(z) op f2(z')
	 * What z and z' are is different for each method using this class.
	 * 
	 * @author yuki
	 *
	 */
	public static class Constraint {
		public final AffineMapping f1;
		public final AffineMapping f2;
		public final Operator op;

		public Constraint(AffineMapping f1, AffineMapping f2, Operator op) {
			this.f1 = f1;
			this.f2 = f2;
			this.op = op;
			
			if (f1.getDimRHS() != f2.getDimRHS()) {
				throw new RuntimeException("Dimensionality of RHS of the two functions given must be equal.");
			}
		}
	}
}
