package fr.irisa.cairn.model.polymodel.util;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Vector;

import fr.irisa.cairn.model.polymodel.PolymodelException;

/**
 * Number of primitive matrix operations that work on 2D long arrays
 * Copied from original implementation in the legacy AlphaZ system.
 * 
 * @author yuki
 *
 */
public class MatrixOperations {

	/**
	 * Converts a matrix in 2D-vectors to 2D-arrays.
	 * 
	 * @param A
	 *            Matrix in 2D-vector
	 * @return Matrix in 2D-array
	 */
	public static long[][] toMatrix(Vector<Vector<Long>> A) {
		if (A.size() == 0 || A.get(0).size() == 0) {
			return null;
		}

		long[][] B = new long[A.size()][A.get(0).size()];

		for (int i = 0; i < B.length; i++) {
			for (int j = 0; j < B[0].length; j++) {
				B[i][j] = A.get(i).get(j);
			}
		}

		return B;
	}

	/**
	 * Converts a matrix in 2D-vectors to 2D-arrays.
	 * 
	 * @param A
	 *            Matrix in 2D-vector
	 * @return Matrix in 2D-array
	 */
	public static long[][] toLongMatrix(double[][] A) {
		long[][] B = new long[A.length][A[0].length];

		for (int i = 0; i < A.length; i++) {
			for (int j = 0; j < A[0].length; j++) {
				B[i][j] = Math.round(A[i][j]);
			}
		}

		return B;
	}

	/**
	 * Converts a matrix in long to double.
	 * 
	 * @param A
	 *            Matrix in long
	 * @return Matrix in double
	 */
	public static double[][] toDoubleMatrix(long[][] A) {
		double[][] B = new double[A.length][A[0].length];

		for (int i = 0; i < A.length; i++) {
			for (int j = 0; j < A[0].length; j++) {
				B[i][j] = (double) A[i][j];
			}
		}

		return B;
	}


	public static long[][] transpose(long[][] A) {
		long[][] tA = new long[A[0].length][A.length];

		for (int i = 0; i < A.length; i++) {
			for (int j = 0; j < A[i].length; j++) {
				tA[j][i] = A[i][j];
			}
		}

		return tA;
	}

	/**
	 * Swaps row i with row j.
	 * 
	 * @param A
	 *            Matrix to operate on.
	 * @param i
	 *            Index if a row to swap
	 * @param j
	 *            Index if a row to swap
	 */
	public static void swapRows(long[][] A, int i, int j) {
		int M = A[0].length;
		long temp;

		for (int c = 0; c < M; c++) {
			temp = A[i][c];
			A[i][c] = A[j][c];
			A[j][c] = temp;
		}
	}

	/**
	 * Swaps row i with row j.
	 * 
	 * @param A
	 *            Matrix to operate on.
	 * @param i
	 *            Index if a row to swap
	 * @param j
	 *            Index if a row to swap
	 */
	public static void swapRows(double[][] A, int i, int j) {
		int M = A[0].length;
		double temp;

		for (int c = 0; c < M; c++) {
			temp = A[i][c];
			A[i][c] = A[j][c];
			A[j][c] = temp;
		}
	}

	/**
	 * Appends additional rows to an existing matrix. Overload for appending
	 * rows of all zeros.
	 * 
	 * @param A
	 *            Matrix to append to
	 * @param num
	 *            Number of all zero rows to append
	 * @return New matrix with appended rows
	 */
	public static long[][] rowBind(long[][] A, int num) {
		return rowBind(A, num, null);
	}

	/**
	 * Appends additional rows to an existing matrix. Overload for appending the
	 * entire matrix to another.
	 * 
	 * @param A
	 *            Matrix to append to
	 * @param newRows
	 *            Another matrix to be appended
	 * @return New matrix with appended rows
	 */
	public static long[][] rowBind(long[][] A, long[][] newRows) {
		return rowBind(A, newRows.length, newRows);
	}

	/**
	 * Appends additional rows to an existing matrix.
	 * 
	 * @param A
	 *            Matrix to append to
	 * @param num
	 *            Number of rows to append
	 * @param newRows
	 *            Rows to append
	 * @return New matrix with appended rows
	 */
	public static long[][] rowBind(long[][] A, int num, long[][] newRows) {
		// If number negative, don't change
		if (num <= 0) {
			return A;
		}

		// If newRow is empty don't change
		if (newRows != null && (newRows.length == 0 || newRows[0].length == 0)) {
			return A;
		}

		int colSize;

		// Size > 0
		if (A == null || A.length == 0) {
			colSize = newRows[0].length;
			// Size does not match
		} else if (newRows != null && (A[0].length != newRows[0].length || num > newRows.length)) {
			return null;
		} else {
			colSize = A[0].length;
		}

		long[][] mat;

		if (A != null) {
			mat = new long[A.length + num][colSize];
		} else {
			mat = new long[num][colSize];
		}

		if (A != null) {
			for (int i = 0; i < A.length; i++) {
				for (int j = 0; j < colSize; j++) {
					mat[i][j] = A[i][j];
				}
			}
		}

		if (newRows != null) {
			for (int i = 0; i < num; i++) {
				for (int j = 0; j < colSize; j++) {
					if (A != null) {
						mat[A.length + i][j] = newRows[i][j];
					} else {
						mat[i][j] = newRows[i][j];
					}
				}
			}
		}

		return mat;
	}

	/**
	 * Appends additional rows to an existing matrix. Overload for appending
	 * rows of all zeros.
	 * 
	 * @param A
	 *            Matrix to append to
	 * @param num
	 *            Number of all zero rows to append
	 * @return New matrix with appended rows
	 */
	public static double[][] rowBind(double[][] A, int num) {
		return rowBind(A, num, null);
	}

	/**
	 * Appends additional rows to an existing matrix. Overload for appending the
	 * entire matrix to another.
	 * 
	 * @param A
	 *            Matrix to append to
	 * @param newRows
	 *            Another matrix to be appended
	 * @return New matrix with appended rows
	 */
	public static double[][] rowBind(double[][] A, double[][] newRows) {
		return rowBind(A, newRows.length, newRows);
	}

	/**
	 * Appends additional rows to an existing matrix.
	 * 
	 * @param A
	 *            Matrix to append to
	 * @param num
	 *            Number of rows to append
	 * @param newRows
	 *            Rows to append
	 * @return New matrix with appended rows
	 */
	public static double[][] rowBind(double[][] A, int num, double[][] newRows) {
		// If number negative, don't change
		if (num <= 0) {
			return A;
		}

		//
		if (newRows == null) {
			return A;
		}

		// If newRow is empty don't change
		if (newRows != null && (newRows.length == 0 || newRows[0].length == 0)) {
			return A;
		}

		int colSize;

		// Size > 0
		if (A == null || A.length == 0) {
			colSize = newRows[0].length;
			// Size does not match
		} else if (newRows != null && ((A != null && A[0].length != newRows[0].length) || num > newRows.length)) {
			return null;
		} else {
			colSize = A[0].length;
		}

		double[][] mat;

		if (A != null) {
			mat = new double[A.length + num][colSize];
		} else {
			mat = new double[num][colSize];
		}

		if (A != null) {
			for (int i = 0; i < A.length; i++) {
				for (int j = 0; j < colSize; j++) {
					mat[i][j] = A[i][j];
				}
			}
		}

		if (newRows != null) {
			for (int i = 0; i < num; i++) {
				for (int j = 0; j < colSize; j++) {
					if (A != null) {
						mat[A.length + i][j] = newRows[i][j];
					} else {
						mat[i][j] = newRows[i][j];
					}
				}
			}
		}

		return mat;
	}

	/**
	 * Appends additional columns to an existing matrix. Overload for appending
	 * columns of all zeros.
	 * 
	 * @param A
	 *            Matrix to append to
	 * @param num
	 *            Number of all zero columns to append
	 * @return New matrix with appended columns
	 */
	public static long[][] columnBind(long[][] A, int num) {
		return columnBind(A, num, null);
	}

	/**
	 * Appends additional columns to an existing matrix. Overload for appending
	 * the entire matrix to another.
	 * 
	 * @param A
	 *            Matrix to append to
	 * @param newCols
	 *            Another matrix to be appended
	 * @return New matrix with appended columns
	 */
	public static long[][] columnBind(long[][] A, long[][] newCols) {
		return columnBind(A, newCols[0].length, newCols);
	}

	/**
	 * Appends additional columns to an existing matrix.
	 * 
	 * @param A
	 *            Matrix to append to
	 * @param num
	 *            Number of columns to append
	 * @param newCols
	 *            Columns to append
	 * @return New matrix with appended columnss
	 */
	public static long[][] columnBind(long[][] A, int num, long[][] newCols) {
		// If number negative, don't change
		if (num <= 0) {
			return A;
		}
		// Size does not match
		if (newCols != null && ((A != null && A.length != newCols.length) || num > newCols[0].length)) {
			return null;
		}

		long[][] mat;

		if (A != null) {
			mat = new long[A.length][A[0].length + num];
		} else {
			mat = new long[newCols.length][num];
		}

		if (A != null) {
			for (int i = 0; i < mat.length; i++) {
				for (int j = 0; j < A[0].length; j++) {
					mat[i][j] = A[i][j];
				}
			}
		}

		if (newCols != null) {
			for (int i = 0; i < mat.length; i++) {
				for (int j = 0; j < num; j++) {
					if (A != null) {
						mat[i][A[i].length + j] = newCols[i][j];
					} else {
						mat[i][j] = newCols[i][j];
					}
				}
			}
		}

		return mat;
	}

	/**
	 * Appends additional columns to an existing matrix. Overload for appending
	 * columns of all zeros.
	 * 
	 * @param A
	 *            Matrix to append to
	 * @param num
	 *            Number of all zero columns to append
	 * @return New matrix with appended columns
	 */
	public static double[][] columnBind(double[][] A, int num) {
		return columnBind(A, num, null);
	}

	/**
	 * Appends additional columns to an existing matrix. Overload for appending
	 * the entire matrix to another.
	 * 
	 * @param A
	 *            Matrix to append to
	 * @param newCols
	 *            Another matrix to be appended
	 * @return New matrix with appended columns
	 */
	public static double[][] columnBind(double[][] A, double[][] newCols) {
		return columnBind(A, newCols[0].length, newCols);
	}

	/**
	 * Appends additional columns to an existing matrix.
	 * 
	 * @param A
	 *            Matrix to append to
	 * @param num
	 *            Number of columns to append
	 * @param newCols
	 *            Columns to append
	 * @return New matrix with appended columnss
	 */
	public static double[][] columnBind(double[][] A, int num, double[][] newCols) {
		// If number negative, don't change
		if (num <= 0) {
			return A;
		}
		// Size does not match
		if (newCols != null && ((A != null && A.length != newCols.length) || num > newCols[0].length)) {
			return null;
		}

		double[][] mat;

		if (A != null) {
			mat = new double[A.length][A[0].length + num];
		} else {
			mat = new double[newCols.length][num];
		}

		if (A != null) {
			for (int i = 0; i < mat.length; i++) {
				for (int j = 0; j < A[0].length; j++) {
					mat[i][j] = A[i][j];
				}
			}
		}

		if (newCols != null) {
			for (int i = 0; i < mat.length; i++) {
				for (int j = 0; j < num; j++) {
					if (A != null) {
						mat[i][A[i].length + j] = newCols[i][j];
					} else {
						mat[i][j] = newCols[i][j];
					}
				}
			}
		}

		return mat;
	}

	/**
	 * Returns the list of indexes for non-zero columns.
	 * 
	 * @param A
	 *            Matrix to operate on
	 * @return List of non-zero column indexes
	 */
	public static long[] nonZeroColumns(long[][] A) {
		long[] non_zeros = new long[A[0].length];

		for (long[] row : A) {
			// Increment one when non-zero
			for (int i = 0; i < non_zeros.length; i++) {
				if (row[i] != 0) {
					non_zeros[i]++;
				}
			}
		}

		return non_zeros;
	}

	/**
	 * Returns the matrix with all-zero columns removed. Overload when non-zero
	 * columns are not known yet.
	 * 
	 * @param A
	 *            Matrix to operate on
	 * @return Matrix with all-zero columns removed
	 */
	public static long[][] removeZeroColumns(long[][] A) {
		return removeZeroColumns(A, null);
	}

	/**
	 * Returns the matrix with all-zero columns removed.
	 * 
	 * @param A
	 *            Matrix to operate on
	 * @return List of non-zero column indexes
	 */
	public static long[][] removeZeroColumns(long[][] A, long[] non_zeros) {
		if (non_zeros == null) {
			non_zeros = nonZeroColumns(A);
		}

		// Count number of non zeros
		int nNZ = 0;

		for (long el : non_zeros) {
			if (el != 0) {
				nNZ++;
			}
		}

		// Reduced mat = matrix without column with just zeros.
		long[][] reduced_mat = new long[A.length][nNZ];

		// Copy matrix skipping zero columns
		for (int i = 0; i < A.length; i++) {
			int nz = 0;

			for (int j = 0; j < A[i].length; j++) {
				if (non_zeros[j] == 0) {
					nz++;
					continue;
				}
				reduced_mat[i][j - nz] = A[i][j];
			}
		}
		return reduced_mat;
	}

	/**
	 * Returns the vector with zero columns removed.
	 * 
	 * @param A
	 *            Vector to operate on
	 * @return List of non-zero column indexes
	 */
	public static long[] removeZeroColumns(long[] A, long[] non_zeros) {
		if (non_zeros == null) {
			non_zeros = nonZeroColumns(new long[][] { A });
		}

		// Count number of non zeros
		int nNZ = 0;

		for (long el : non_zeros) {
			if (el != 0) {
				nNZ++;
			}
		}

		// Reduced non zero = vector of non zeros with all elements > 0
		long[] reduced_non_zeros = new long[nNZ];
		int nz = 0;

		for (int i = 0; i < A.length; i++) {
			if (non_zeros[i] == 0) {
				nz++;
				continue;
			}
			reduced_non_zeros[i - nz] = non_zeros[i];
		}
		return reduced_non_zeros;
	}

	/**
	 * Returns if the two vectors are numerically equal.
	 * 
	 * @param A
	 *            vector
	 * @param B
	 *            vector
	 * @return true if A and B are equal.
	 */
	public static boolean isEqual(long[] A, long[] B) {

		// Check for null
		if (A == null || B == null) {
			return false;
		}

		// check if empty
		if (A.length == 0 || B.length == 0) {
			return false;
		}

		// check size
		if (A.length != B.length) {
			return false;
		}

		// Check
		for (int i = 0; i < A.length; i++) {
			if (A[i] != B[i]) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Returns if the two vectors are numerically equal.
	 * 
	 * @param A
	 *            vector
	 * @param B
	 *            vector
	 * @return true if A and B are equal.
	 */
	public static boolean isEqual(double[] A, double[] B) {

		// Check for null
		if (A == null || B == null) {
			return false;
		}

		// check if empty
		if (A.length == 0 || B.length == 0) {
			return false;
		}

		// check size
		if (A.length != B.length) {
			return false;
		}

		// Check
		for (int i = 0; i < A.length; i++) {
			if (A[i] != B[i]) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Returns if the two matrices are numerically equal.
	 * 
	 * @param A
	 *            matrix
	 * @param B
	 *            matrix
	 * @return true if A and B are equal.
	 */
	public static boolean isEqual(long[][] A, long[][] B) {
		// Check for null
		if (A == null || B == null) {
			return false;
		}

		// check if empty
		if (A.length == 0 || B.length == 0 || A[0].length == 0 || B[0].length == 0) {
			return false;
		}

		// check size
		if (A.length != B.length || A[0].length != B[0].length) {
			return false;
		}

		// Check
		for (int i = 0; i < A.length; i++) {
			if (!isEqual(A[i], B[i])) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Returns if the two matrices are numerically equal.
	 * 
	 * @param A
	 *            matrix
	 * @param B
	 *            matrix
	 * @return true if A and B are equal.
	 */
	public static boolean isEqual(double[][] A, double[][] B) {
		// Check for null
		if (A == null || B == null) {
			return false;
		}

		// check if empty
		if (A.length == 0 || B.length == 0 || A[0].length == 0 || B[0].length == 0) {
			return false;
		}

		// check size
		if (A.length != B.length || A[0].length != B[0].length) {
			return false;
		}

		// Check
		for (int i = 0; i < A.length; i++) {
			if (!isEqual(A[i], B[i])) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Returns a new vector containing alpha*X
	 * 
	 * @param X
	 *            vector to be multiplied
	 * @param alpha
	 *            scalar to be added
	 * @return alpha*X
	 */
	public static long[] scalarMultiplication(long[] X, long alpha) {
		long[] res = new long[X.length];

		for (int i = 0; i < X.length; i++) {
			res[i] = X[i] * alpha;
		}

		return res;
	}

	/**
	 * Returns a new vector containing alpha*X
	 * 
	 * @param X
	 *            vector to be multiplied
	 * @param alpha
	 *            scalar to be added
	 * @return alpha*X
	 */
	public static double[] scalarMultiplication(double[] X, double alpha) {
		double[] res = new double[X.length];

		for (int i = 0; i < X.length; i++) {
			res[i] = X[i] * alpha;
		}

		return res;
	}

	/**
	 * Returns a new matrix containing alpha*A
	 * 
	 * @param A
	 *            matrix to be multiplied
	 * @param alpha
	 *            scalar to be added
	 * @return alpha*A
	 */
	public static long[][] scalarMultiplication(long[][] A, long alpha) {
		long[][] res = new long[A.length][A[0].length];

		for (int i = 0; i < A.length; i++) {
			res[i] = scalarMultiplication(A[i], alpha);
		}

		return res;
	}

	/**
	 * Returns a new matrix containing alpha*A
	 * 
	 * @param A
	 *            matrix to be multiplied
	 * @param alpha
	 *            scalar to be added
	 * @return alpha*A
	 */
	public static double[][] scalarMultiplication(double[][] A, double alpha) {
		double[][] res = new double[A.length][A[0].length];

		for (int i = 0; i < A.length; i++) {
			res[i] = scalarMultiplication(A[i], alpha);
		}

		return res;
	}

	/**
	 * Removes columns from the vector given, and returns a new matrix
	 * 
	 * @param vec
	 *            vector to operate on
	 * @param columns
	 *            columns (indexed from 0) to remove
	 * @return new vector with columns removed
	 */
	public static long[] removeColumns(long[] vec, int[] columns) {
		// Empty vectors
		if (vec == null || vec.length == 0) {
			return null;
		}

		// When nothing to remove
		if (columns == null || columns.length == 0) {
			return vec;
		}

		// If all columns are to be removed
		if (vec.length <= columns.length) {
			return null;
		}

		// Body
		long[] res = new long[vec.length - columns.length];

		int cur = 0;

		for (int c = 0; c < vec.length; c++) {
			if (contains(c, columns)) {
				continue;
			} else {
				res[cur] = vec[c];
				cur++;
			}
		}

		return res;
	}

	/**
	 * Removes columns from the matrix given, and returns a new matrix.
	 * 
	 * @param mat
	 *            matrix to operate on
	 * @param columns
	 *            columns (indexed from 0) to remove
	 * @return new matrix with columns removed
	 */
	public static long[][] removeColumns(long[][] mat, int[] columns) {
		// Empty matrices
		if (mat == null || mat.length == 0 || mat[0].length == 0) {
			return null;
		}

		// When nothing to remove
		if (columns == null || columns.length == 0) {
			return mat;
		}

		// If all columns are to be removed
		if (mat[0].length <= columns.length) {
			return null;
		}

		// Body
		long[][] res = new long[mat.length][mat[0].length - columns.length];

		for (int r = 0; r < mat.length; r++) {
			res[r] = removeColumns(mat[r], columns);
		}

		return res;
	}
	private static boolean contains(int val, int[] col) {
		for (int v : col) {
			if (val == v) {
				return true;
			}
		}

		return false;
	}

	/* Written by Simon below */

	/**
	 * 
	 * Elements in row i add scale multiplication of scale and elements in row j
	 * 
	 * @param A
	 *            Matrix to operate on.
	 * @param i
	 *            Index if a row to be added
	 * @param j
	 *            Index if a row to add into row i
	 * @param scalar
	 *            Scalar to be multiplied
	 * 
	 * */
	public static void rowAddition(long[][] A, int i, int j, long scalar) {
		int M = A[0].length;

		for (int k = 0; k < M; k++) {
			A[i][k] = A[i][k] + A[j][k] * scalar;
		}
	}

	/**
	 * Elements in row i add scale multiplication of scale and elements in row j
	 * 
	 * @param A
	 *            Matrix to operate on.
	 * @param i
	 *            Index if a column to be added
	 * @param j
	 *            Index if a column to add into row i
	 * @param scalar
	 *            Scalar to be multiplied
	 * 
	 * */

	public static void columnAddition(long[][] A, int i, int j, long scalar) {
		int M = A.length;

		for (int k = 0; k < M; k++) {
			A[k][i] = A[k][i] + A[k][j] * scalar;
		}
	}

	/**
	 * Swaps column i with row j.
	 * 
	 * @param A
	 *            Matrix to operate on.
	 * @param i
	 *            Index if a column to swap
	 * @param j
	 *            Index if a column to swap
	 */

	public static void swapColumns(long[][] A, int i, int j) {
		for (int k = 0; k < A.length; k++) {
			long temp = A[k][i];

			A[k][i] = A[k][j];
			A[k][j] = temp;
		}
	}

	/**
	 * calculate the matrix multiplication A*B
	 * 
	 * @param A
	 *            Matrix to operate on.
	 * @param B
	 *            Matrix to operate on
	 * @return the multiplication of A*B
	 * 
	 * */
	public static long[][] matrixMultiplication(long[][] A, long[][] B) {
		if (A == null || B == null) {
			return null;
		}
		if (A[0].length != B.length) {
			return null;
		}
		long[][] C = new long[A.length][B[0].length];

		for (int i = 0; i < C.length; i++) {
			for (int j = 0; j < C[0].length; j++) {
				C[i][j] = 0;
				for (int k = 0; k < A[0].length; k++) {
					C[i][j] += A[i][k] * B[k][j];
				}
			}
		}
		return C;
	}

	/**
	 * calculate the Smith Normal Form of A
	 * 
	 * @param A
	 *            Matrix to operate on.
	 * @return A list contains 3 elements: List[0] is the Smith Normal Form of
	 *         A, List[1] is P and List[2] is Q which P and Q are unimodular
	 *         matrix and P*A*Q=SNF of A
	 * 
	 * */
	public static ArrayList<long[][]> smithNormalForm(long[][] A) {

		/* initialize */
		ArrayList<long[][]> result = new ArrayList<long[][]>();
		int M = A.length;
		int N = A[0].length;
		int bound = Math.min(M, N);
		long[][] left, right;

		left = new long[M][M];
		right = new long[N][N];
		for (int i = 0; i < M; i++) {
			for (int j = 0; j < M; j++) {
				if (i == j) {
					left[i][i] = 1;
				} else {
					left[i][j] = 0;
				}
			}
		}
		for (int i = 0; i < N; i++) {
			for (int j = 0; j < N; j++) {
				if (i == j) {
					right[i][i] = 1;
				} else {
					right[i][j] = 0;
				}
			}
		}

		/* operate row by row */
		for (int i = 0; i < bound; i++) {

			/* found gcd of the matrix */
			long gcd = A[i][i];

			for (int j = i; j < M; j++) {
				for (int k = i; k < N; k++) {
					gcd = gcd(A[j][k], gcd);
				}
			}

			/* Look for the min value */
			long Min = Long.MAX_VALUE;
			int posX = -1, posY = -1;

			for (int j = i; j < M; j++) {
				for (int k = i; k < N; k++) {
					if (Math.abs(A[j][k]) < Min && A[j][k] != 0) {
						posX = j;
						posY = k;
						Min = Math.abs(A[j][k]);
					}
				}
			}

			/* swap min value to left-up corner */
			if (posX != i) {
				if (posX == -1) {
					for (int j = 0; j < Math.min(M, N); j++) {
						if (A[j][j] < 0) {
							A[j][j] = -A[j][j];
							scalarMultiplication(left[j], -1);
						}
					}
					result.add(A);
					result.add(left);
					result.add(right);
					return result;
				}
				swapRows(A, i, posX);
				swapRows(left, i, posX);
			}
			if (posY != i) {
				if (posY == -1) {
					for (int j = 0; j < Math.min(M, N); j++) {
						if (A[j][j] < 0) {
							A[j][j] = -A[j][j];
							scalarMultiplication(left[j], -1);
						}
					}
					result.add(A);
					result.add(left);
					result.add(right);
					return result;
				}
				swapColumns(A, i, posY);
				swapColumns(right, i, posY);
			}
			while (true) {

				/* massage the row i */
				for (int j = i + 1; j < N; j++) {
					while (A[i][j] % A[i][i] != 0) {
						long scalar = -A[i][j] / A[i][i];

						columnAddition(A, j, i, scalar);
						columnAddition(right, j, i, scalar);
						swapColumns(A, i, j);
						swapColumns(right, i, j);
					}
					long scalar = -A[i][j] / A[i][i];

					columnAddition(A, j, i, scalar);
					columnAddition(right, j, i, scalar);
				}

				/* massage the column i */
				for (int j = i + 1; j < M; j++) {
					while (A[j][i] % A[i][i] != 0) {
						long scalar = -A[j][i] / A[i][i];

						rowAddition(A, j, i, scalar);
						rowAddition(left, j, i, scalar);
						swapRows(A, i, j);
						swapRows(left, i, j);
					}
					long scalar = -A[j][i] / A[i][i];

					rowAddition(A, j, i, scalar);
					rowAddition(left, j, i, scalar);
				}

				/* massage the row i */
				for (int j = i + 1; j < N; j++) {
					long scalar = -A[i][j] / A[i][i];

					columnAddition(A, j, i, scalar);
					columnAddition(right, j, i, scalar);
				}

				/* massage the column i */
				for (int j = i + 1; j < M; j++) {
					long scalar = -A[j][i] / A[i][i];

					rowAddition(A, j, i, scalar);
					rowAddition(left, j, i, scalar);
				}

				/* do the remaining */
				if (Math.abs(A[i][i]) != Math.abs(gcd)) {
					for (int j = i + 1; j < M; j++) {
						for (int k = i + 1; k < N; k++) {
							if (A[j][k] % A[i][i] != 0) {
								rowAddition(A, j, i, 1);
								rowAddition(left, j, i, 1);
								j = M + 1;
								k = N + 1;
							}
						}
					}
				} else {
					break;
				}
			}
		}
		for (int i = 0; i < Math.min(M, N); i++) {
			if (A[i][i] < 0) {
				A[i][i] = -A[i][i];
				left[i] = scalarMultiplication(left[i], -1);
			}
		}
		result.add(A);
		result.add(left);
		result.add(right);
		return result;
	}

	/**
	 * 
	 * Calculate the greatest common divider of two integers a and b
	 * 
	 * @param a
	 *            One input integer
	 * @param b
	 *            Another input integer
	 * @return Greatest common divider of a and b
	 */
	private static long gcd(long a, long b) {
		if (b == 0) {
			return a;
		}
		return gcd(b, a % b);
	}

	/**
	 * Calculate the inverse of a square unimodular integer matrix
	 * 
	 * @param Matrix
	 *            Input matrix
	 * @return The inverse matrix of input matrix
	 */

	public static long[][] intInverse(long[][] Matrix) {
		long[][] A = Matrix.clone();

		if (A.length == 1) {
			return A;
		}

		/**
		 * First deal with the left upper corner
		 */
		int M = A.length;
		long[][] left;

		left = new long[M][M];
		for (int i = 0; i < M; i++) {
			long sc = 1;

			if (A[i][0] != 0) {
				sc = A[i][0] / Math.abs(A[i][0]);
			}
			for (int j = 0; j < M; j++) {
				A[i][j] = A[i][j] * sc;
				if (i == j) {
					left[i][j] = sc;
				} else {
					left[i][j] = 0;
				}
			}
		}

		/* massage the column i */
		for (int i = 0; i < M; i++) {
			if (A[i][0] != 0) {
				swapRows(A, 0, i);
				swapRows(left, 0, i);
				break;
			}
		}
		for (int j = 1; j < M; j++) {
			while (A[j][0] % A[0][0] != 0) {
				long scalar = -A[j][0] / A[0][0];

				rowAddition(A, j, 0, scalar);
				rowAddition(left, j, 0, scalar);
				swapRows(A, 0, j);
				swapRows(left, 0, j);
			}
			long scalar = -A[j][0] / A[0][0];

			rowAddition(A, j, 0, scalar);
			rowAddition(left, j, 0, scalar);
		}

		/* massage the row i */
		long[][] smaller = new long[M - 1][M - 1];

		for (int i = 0; i < M - 1; i++) {
			for (int j = 0; j < M - 1; j++) {
				smaller[i][j] = A[i + 1][j + 1];
			}
		}
		long[][] vector = new long[1][M - 1];

		for (int i = 0; i < M - 1; i++) {
			vector[0][i] = A[0][i + 1];
		}
		smaller = intInverse(smaller);
		vector = matrixMultiplication(vector, smaller);
		long[][] left2 = new long[M][M];

		for (int i = 0; i < M; i++) {
			for (int j = 0; j < M; j++) {
				if (i == 0 && i != j) {
					left2[i][j] = -vector[0][j - 1];
				} else if (i == j) {
					left2[i][i] = 1;
				} else {
					left2[i][j] = 0;
				}
			}
		}
		long[][] left3 = new long[M][M];

		left3[0][0] = 1;
		for (int i = 1; i < M; i++) {
			left3[0][i] = 0;
			left3[i][0] = 0;
		}
		for (int i = 1; i < M; i++) {
			for (int j = 1; j < M; j++) {
				left3[i][j] = smaller[i - 1][j - 1];
			}
		}
		long[][] result = new long[M][M];

		result = matrixMultiplication(left2, left);
		result = matrixMultiplication(left3, result);
		return result;

	}

	/**
	 * Calculate inverse transformation of T with equations A (Only the
	 * equations in domain should be input) T is n*m, A is r*m, the return
	 * matrix should be m*n
	 * 
	 * @param T
	 *            Transformation matrix
	 * @param A
	 *            Equations matrix
	 * @return Inverse Matrix of T
	 */
	public static long[][] getInverseInContext(long[][] T, long[][] A) throws PolymodelException {
		long[][] com;

		// If A is not null, simplify A and combine it into T as the matrix that
		// should be operated on
		// else T is the matrix that should be operated on
		if (A != null && A.length > 0) {
			if (A[0].length != T[0].length) {
				throw new RuntimeException("Dimension of context and function does not match.");
			}
			// Find a unimodular matrix APrime to make APrime*X=0 is equivalent
			// to AX=0
			ArrayList<long[][]> aSNFList = smithNormalForm(A);
			long[][] newA = aSNFList.get(0);
			int valid = newA.length;

			for (int i = 0; i < newA.length; i++) {
				if (i >= newA[0].length) {
					valid = newA[0].length;
					break;
				}
				if (newA[i][i] == 0) {
					valid = i - 1;
					break;
				} else if (newA[i][i] != 1) {
					newA[i][i] = 1;
				}
			}
			long[][] APrime = new long[valid][A[0].length];

			for (int i = 0; i < APrime.length; i++) {
				for (int j = 0; j < APrime[i].length; j++) {
					APrime[i][j] = newA[i][j];
				}
			}
			APrime = matrixMultiplication(APrime, intInverse(aSNFList.get(2)));
			com = rowBind(T, APrime);
		} else {
			com = T.clone();
		}
		// pick out the constant column
		long[][] constant = new long[com.length][1];

		for (int i = 0; i < com.length; i++) {
			constant[i][0] = com[i][com[i].length - 1];
		}
		// remove constant column from combination matrix
		int[] index = new int[1];

		index[0] = com[0].length - 1;
		com = removeColumns(com, index);
		
		if (com.length < com[0].length) {
			throw new PolymodelException("There is no left-inverse for this function in the given context.");
		}
		// Calculate the Smith Normal Form of combination matrix: com=P*snf*Q
		ArrayList<long[][]> snf = smithNormalForm(com);

		// If diagonal elements of snf are not all 1s, there is no left-inverse
		// of it.
		for (int i = 0; i < snf.get(0)[0].length; i++) {
			if (snf.get(0)[i][i] != 1) {
				throw new PolymodelException("There is no left-inverse for this function in the given context.");
			}
		}
		// If diagonal elements of snf are all 1s, the left-inverse should be
		// Q^(-1) * [Id 0]^T * P^(-1)
		long[][] Id0 = new long[snf.get(2).length][snf.get(1).length];

		for (int i = 0; i < Id0.length; i++) {
			for (int j = 0; j < Id0[i].length; j++) {
				if (i == j) {
					Id0[i][j] = 1;
				} else {
					Id0[i][j] = 0;
				}
			}
		}
		long[][] result = matrixMultiplication(snf.get(2), Id0);

		result = matrixMultiplication(result, snf.get(1));
		// Calculate the new constant column
		long[][] newConst = matrixMultiplication(result, constant);

		newConst = scalarMultiplication(newConst, -1);
		long[][] res = new long[result.length][T.length];

		for (int i = 0; i < res.length; i++) {
			for (int j = 0; j < res[0].length; j++) {
				res[i][j] = result[i][j];
			}
		}

		// Combine the constant value back to the matrix
		return columnBind(res, newConst);
	}


	public static String MatrixToDep(long[][] Matrix, String[] Vars) {
		String result = new String();

		result = "(";
		for (int i = 0; i < Vars.length - 1; i++) {
			result = result + Vars[i] + ",";
		}
		result = result + Vars[Vars.length - 1] + "->";
		for (int i = 0; i < Matrix.length; i++) {
			String currentV = new String();

			for (int j = 0; j < Matrix[i].length - 1; j++) {
				if (Matrix[i][j] != 0) {
					String number = new Long(Matrix[i][j]).toString();

					if (number.charAt(0) == '+') {
						number = number.substring(1);
					}
					if (number.equals("1")) {
						number = "";
					}
					currentV = currentV + number + Vars[j] + "+";
				}
			}
			String number = new Long(Matrix[i][Matrix[i].length - 1]).toString();

			if (number.charAt(0) == '+') {
				number = number.substring(1);
			}
			if (currentV.length() == 0) {
				currentV = number;
			} else {
				if (Matrix[i][Matrix[i].length - 1] == 0) {
					currentV = currentV.substring(0, currentV.length() - 1);
				} else {
					currentV = currentV + number;
				}
			}
			while (currentV.indexOf("+-") != -1) {
				currentV = currentV.substring(0, currentV.indexOf("+-")) + currentV.substring(currentV.indexOf("+-") + 1);
			}
			result = result + currentV;
			if (i != Matrix.length - 1) {
				result = result + ",";
			}
		}
		return result + ")";
	}

	/**
	 * Converts the internal representation of domains to inequalities The first
	 * column of the internal representation denotes eq/ineq.
	 * 
	 * @param mat
	 * @return
	 */
	public static long[][] convertDomainToInequalities(long[][] mat) {

		Vector<Vector<Long>> res = new Vector<Vector<Long>>();

		// For each row
		for (int i = 0; i < mat.length; i++) {
			Vector<Long> eq = new Vector<Long>();
			Vector<Long> eqneg = new Vector<Long>();

			// Copy the constraints skipping the first col
			for (int j = 1; j < mat[i].length; j++) {
				eq.add(mat[i][j]);
				// If first col is 0, then it is a equality, construct the
				// negation
				if (mat[i][0] == 0) {
					eqneg.add(-mat[i][j]);
				}
			}
			res.add(eq);
			if (mat[i][0] == 0) {
				res.add(eqneg);
			}
		}

		return MatrixOperations.toMatrix(res);
	}

	/**
	 * Calculates determinant of a matrix
	 * 
	 * Took VB code from http://www.freevbcode.com/ShowCode.asp?ID=1685
	 * 
	 * @param matrix
	 * @return
	 */
	public static double determinant(long[][] matrix) {
		double deter = 1;
		double[][] mat = MatrixOperations.toDoubleMatrix(matrix);
		int N = mat.length;

		// non-square
		if (mat.length != mat[0].length) {
			return 0;
		}

		for (int k = 0; k < N; k++) {
			if (mat[k][k] == 0) {
				int j = k;
				boolean check;

				do {
					check = true;
					if (mat[k][j] == 0) {
						if (j == N - 1) {
							return 0;
						}
						check = false;
						j++;
					} else {
						for (int i = k; i < N; i++) {
							double temp = mat[i][j];

							mat[i][j] = mat[i][k];
							mat[i][k] = temp;
						}
						deter = -deter;
					}
				} while (!check);
			}
			deter = deter * mat[k][k];
			if (k < N - 1) {
				int kk = k + 1;

				for (int i = kk; i < N; i++) {
					for (int j = kk; j < N; j++) {
						mat[i][j] -= (mat[i][k] * mat[k][j] / mat[k][k]);
					}
				}
			}
		}

		return deter;
	}

	/**
	 * Returns true if the rows of the given matrix is linearly dependent. The
	 * check is done by testing if the determinant is 0 or not. If the
	 * determinant is 0, then the matrix does not have inverse, which in turn
	 * means that the trivial solution will be the only solution.
	 * 
	 * @param matrix
	 * @return
	 */
	public static boolean isLinearlyDependent(long[][] matrix) {
		// empty matrix is linearly independent
		if (matrix.length == 0) {
			return false;
		}

		// if number of vectors to test linear dependence is greater than the
		// size of the space, then it has to be dependent
		if (matrix[0].length < matrix.length + 1) {
			return true;
		}

		// transpose
		long[][] ldTest = MatrixOperations.transpose(matrix);

		// if its a square
		if (ldTest.length == ldTest[0].length) {
			// take the determinant
			double det = MatrixOperations.determinant(ldTest);

			return det == 0;
			// If it is not a square, all possible combination of reduced matrix
			// must be tested
		} else {
			// The available cols is the number of vectors given
			int availableCols = ldTest[0].length;
			int[] usedRows = new int[availableCols];

			// initialize with the top available rows
			for (int i = 0; i < availableCols; i++) {
				usedRows[i] = i;
			}

			while (true) {
				// construct a reduced matrix of mxm
				long[][] reducedMat = new long[availableCols][availableCols];

				for (int i = 0; i < usedRows.length; i++) {
					reducedMat[i] = ldTest[usedRows[i]];
				}
				// take the determinant
				double det = MatrixOperations.determinant(reducedMat);

				/*
				 * System.out.println("Linear Dependence Test");
				 * MatrixOperations.printMatrix(reducedMat);
				 * 
				 * System.out.println("Det"); System.out.println(det);
				 */

				// found linear independence
				if (det != 0) {
					return false;
				}

				// exit check
				if (usedRows[0] == ldTest.length - usedRows.length) {
					break;
				}

				// move to the next combination
				// start from the last row
				int current = 0;

				while (true) {
					// if the current row is already at the end
					if (usedRows[usedRows.length - 1 - current] == ldTest.length - 1 - current) {
						current++;
					} else {
						break;
					}
				}

				// increment to the next row
				usedRows[usedRows.length - 1 - current]++;
				// when a row above is updated, lower rows gets reset to above+1
				for (int i = current - 1; i > 0; i--) {
					usedRows[usedRows.length - 1 - i] = usedRows[usedRows.length - 1 - i + 1] + 1;
				}
			}

			return true;
		}
	}

	/**
	 * Prints out the given matrix to System.out.
	 * 
	 * @param A
	 *            matrix to print
	 */
	public static void printMatrix(long[][] A) {
		printMatrix(A, System.out);
	}

	/**
	 * Prints out the given matrix to the specified stream.
	 * 
	 * @param A
	 *            matrix to print
	 * @param out
	 *            PrintStream to output
	 */
	public static void printMatrix(long[][] A, PrintStream out) {
		if (A != null) {
			for (long[] row : A) {
				printVector(row, out);
			}
		} else {
			out.println(A);
		}
	}

	/**
	 * Prints out the given vector to the specified stream.
	 * 
	 * @param vec
	 *            Vector to output
	 * @param out
	 *            PrintStream to output
	 */
	public static void printVector(long[] vec, PrintStream out) {
		if (vec == null || vec.length == 0) {
			out.println("[]");
			return;
		}

		out.print("[");
		for (int i = 0; i < vec.length - 1; i++) {
			out.print(vec[i] + ", ");
		}
		out.println(vec[vec.length - 1] + "]");
	}
}
