package fr.irisa.cairn.model.polymodel.util;

import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;

import fr.irisa.cairn.model.integerLinearAlgebra.IVariable;
import fr.irisa.cairn.model.integerLinearAlgebra.IntExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.IntLinearConstraint;
import fr.irisa.cairn.model.integerLinearAlgebra.IntLinearConstraintSystem;
import fr.irisa.cairn.model.integerLinearAlgebra.IntLinearExpression;
import fr.irisa.cairn.model.integerLinearAlgebra.SymbolVariable;
import fr.irisa.cairn.model.polymodel.AffineMapping;
import fr.irisa.cairn.model.polymodel.IndexDimension;
import fr.irisa.cairn.model.polymodel.PolyhedralDomain;
import fr.irisa.cairn.model.polymodel.util.IntegerLinearAlgebraPrinter.FORMAT;

public class PolyModelToISLString extends PolymodelSwitch<String> {
	
	private static PolyModelToISLString _givenName;
	private static PolyModelToISLString _dimensions;
	
	public static enum NAMING_SCHEME {
		GIVEN_NAME, DIMENSIONS
	}
	
	protected final NAMING_SCHEME naming;
	
	/**
	 * To allow the use of this class as a static class
	 */
	static {
		_givenName = new PolyModelToISLString(NAMING_SCHEME.GIVEN_NAME);
		_dimensions = new PolyModelToISLString(NAMING_SCHEME.DIMENSIONS);
	}
	
	protected PolyModelToISLString(NAMING_SCHEME naming) {
		this.naming = naming;
	}
	
	/**
	 * Entry point for the pretty printing.
	 * 
	 * @param obj
	 * @return
	 */
//	public static String toString(EObject obj) {
//		return _givenName.doSwitch(obj);
//	}
	
	public static String toString(EObject obj, NAMING_SCHEME naming) {
		switch (naming) {
			case GIVEN_NAME:
				return _givenName.doSwitch(obj);
			default:
				return _dimensions.doSwitch(obj);
		}
	}
	
	public static String toUnionMap(List<IVariable> params, List<Relation> relations, NAMING_SCHEME naming) {
		final PolyModelToISLString instance;
		switch (naming) {
			case GIVEN_NAME:
				instance = _givenName;
				break;
			default:
				instance = _dimensions;
		}
		
		StringBuffer sb = new StringBuffer();
		
		//parameters are common
		sb.append(instance.IVariablestoString(0, params));
		sb.append( " -> {" );
		
		boolean first = true;
		for (Relation relation : relations) {
			if (!first) sb.append("; ");
			first = false;
			sb.append(relation.inputSpace + instance.IVariablestoString(params.size(), relation.domain.getIndices()));
			sb.append(" -> " + relation.outputSpace + "[");
			boolean firstExpr = true;
			for (IntLinearExpression expr : relation.mapping.getFunctions()) {
				if (!firstExpr) sb.append(",");
				firstExpr = false;
				sb.append(instance.IntLinearExpressionToString(relation.mapping.getParams(), relation.mapping.getIndices(), expr));
			}
			sb.append("] : ");
			sb.append(instance.IntLinearSystemstoString(relation.domain.getPolyhedra()));
		}
		sb.append( "}" );
	
		return sb.toString();
	}
	
	public static String toCloogInputString(List<CloogInputDomain> domainUnion, NAMING_SCHEME naming) {
		if (domainUnion.size() <= 0) {
			throw new RuntimeException("At least one domain must be given.");
		}

		final PolyModelToISLString instance;
		switch (naming) {
			case GIVEN_NAME:
				instance = _givenName;
				break;
			default:
				instance = _dimensions;
		}
		
		StringBuffer sb = new StringBuffer();
		
		int nparam = -1;
		int nindices = -1;
		//iterate over each domain
		for (CloogInputDomain cid : domainUnion) {
			//make sure number of parameters are equal
			if (nparam == -1) {
				nparam = cid.domain.getParams().size();
			} else if (nparam != cid.domain.getParams().size()) {
				throw new RuntimeException("Number of parameters must match.");
			}
			//make sure number of dimensions are equal
			if (nindices == -1) {
				nindices = cid.domain.getIndices().size();
			} else if (nindices != cid.domain.getIndices().size()) {
				throw new RuntimeException("Number of indices must match.");
			}
			//delimiter between unions
			if (sb.length() > 0) {
				sb.append(";\n");
			}
				
				
			//pretty print each domain
			sb.append("   " + cid.statementName);
			sb.append(instance.IVariablestoString(cid.domain.getNParams(), cid.domain.getIndices()));
			sb.append(" -> ");
			if (cid.domain.getIndices().size()>0) {
				String rhs = instance.IVariablestoString(cid.domain.getNParams(), cid.domain.getIndices());
				sb.append(rhs.replaceAll("]$", ","+cid.statementNumber+"]"));
			} else {
				sb.append("["+cid.statementNumber+"]");
			}
			sb.append(" : ");
			sb.append(instance.IntLinearSystemstoString(cid.domain.getPolyhedra()));
			
		}
		
		//insert beginning and end
		sb.insert(0, instance.IVariablestoString(0, domainUnion.get(0).domain.getParams()) + " -> {\n");
		sb.append("}");
		
//		"[N] -> { [ ] :  }",
//		"[N]-> {"+
//			"S1[i] -> [s0,s1,s2] : i-N< 0 and i>= 0 and s0=i and  s1=0 and  s2=0;"+
//			"S2[i,j] -> [s0,s1,s2] : i-N< 0 and i>= 0 and i-2> 0 and j-N< 0 and j-i>= 0 and s0=i and  s1=j and  s2=1;"+
//			"S3[i,j] -> [s0,s1,s2] : i-N< 0 and i>= 0 and i-2> 0 and j-N< 0 and j-i>= 0 and s0=i and  s1=i+j and  s2=1"+
//		"}",
		
		return sb.toString();
	}
	
	public static class CloogInputDomain {
		public final PolyhedralDomain domain;
		public final String statementName;
		public final int statementNumber;
		
		public CloogInputDomain(PolyhedralDomain domain, String statementName, int statementNumber) {
			this.domain = domain;
			this.statementName = statementName;
			this.statementNumber = statementNumber;
		}
	}
	
	public static class Relation {
		public final PolyhedralDomain domain;
		public final AffineMapping mapping;
		public final String inputSpace;
		public final String outputSpace;
		
		public Relation(PolyhedralDomain domain, AffineMapping mapping) {
			this.domain = domain;
			this.mapping = mapping;
			inputSpace = "";
			outputSpace = "";
		}
		
		public Relation(PolyhedralDomain domain, AffineMapping mapping, String inputSpace, String outputSpace) {
			this.domain = domain;
			this.mapping = mapping;
			this.inputSpace = inputSpace;
			this.outputSpace = outputSpace;
		}
	}
	
	@Override
	public String defaultCase(EObject object) {
		throw new RuntimeException("Unhandled object " + object + " of type " + object.eClass().getName());
	}
	
	/**
	 * List of variables in ISL form (ex. [i,j,k])
	 * 
	 * @param vars
	 * @return
	 */
	public String IVariablestoString(int numPreceedingDimensions, List<IVariable> vars) {
		StringBuffer varstr = new StringBuffer();
		for (IVariable iv : vars) {
			if (varstr.length() > 0) {
				varstr.append(",");
			}
			if (naming == NAMING_SCHEME.DIMENSIONS) {
				String dimName;
				if (iv instanceof IndexDimension) {
					dimName = ((IndexDimension)iv).getDimName();
				} else {
					dimName = "i"+(vars.indexOf(iv)+numPreceedingDimensions);
				}
				if(dimName.toUpperCase().equals("MAX") || dimName.toUpperCase().equals("MIN") )
					dimName ="_"+dimName;
				varstr.append(dimName);
			} else {
				String string = iv.toString();
				if(string.toUpperCase().equals("MAX") || string.toUpperCase().equals("MIN") )
				string ="_"+string;
				varstr.append(string);
				
			}
		}
		return "["+varstr.toString()+"]";
	}
	
	public String IntLinearSystemstoString(List<IntLinearConstraintSystem> domains) {
		StringBuffer res = new StringBuffer();
		
		boolean union = domains.size() > 1;
		for (IntLinearConstraintSystem domain : domains) {
			EList<IntLinearConstraint> equations = domain.getLinearConstraints();
			if (equations.size() > 0) {
				if (res.length() > 0) {
					res.append(" or ");
				}
				if (union) res.append("(");
				if (naming == NAMING_SCHEME.GIVEN_NAME) {
					res.append(IntegerLinearAlgebraPrinter.toString(domain, FORMAT.ISL));
				} else {
					res.append(IntegerLinearAlgebraPrinter.toString(domain, FORMAT.ISL, null));
				}
				if (union) res.append(")");
			}
				
		}
		
//		boolean union = domains.size() > 1;
//		for (IntLinearConstraintSystem domain : domains) {
//			EList<IntLinearConstraint> equations = domain.getLinearConstraints();
//			if (equations.size() > 0) {
//				if (res.length() > 0) {
//					res.append(" or ");
//				}
//				if (union) res.append("(");
//				res.append(IntLinearConstraintstoString(equations));
//				if (union) res.append(")");
//			}
//				
//		}
		
		return res.toString();
	}
		
	
	public String PolyhedralDomainWithExtraIndices(PolyhedralDomain p, List<IVariable> exIndices) {
		StringBuffer poly = new StringBuffer();
		
		//Parameter declaration
		List<IVariable> params = p.getParams();
		if (params != null && params.size() > 0) {
			poly.append(IVariablestoString(0, params)+"->");
		}

		poly.append("{ ");
		//Indices
		List<IVariable> indices = p.getIndices();
		indices.addAll(exIndices);
		poly.append(IVariablestoString(params.size(), indices));
		poly.append(" : ");
		//Constraints
		poly.append(IntLinearSystemstoString(p.getPolyhedra()));
		poly.append(" }");
		return poly.toString();
		
	}

	
	public String casePolyhedralDomain(PolyhedralDomain p) {
		StringBuffer poly = new StringBuffer();
		
		//Parameter declaration
		List<IVariable> params = p.getParams();
		if (params != null && params.size() > 0) {
			poly.append(IVariablestoString(0, params)+"->");
		}

		poly.append("{ ");
		//Indices
		poly.append(IVariablestoString(params.size(), p.getIndices()));
		poly.append(" : ");
		//Constraints
		poly.append(IntLinearSystemstoString(p.getPolyhedra()));
		poly.append(" }");
		return poly.toString();
	}
	
	public String caseAffineMapping(AffineMapping am) {
		//PolyhedralDomain dom = am.getSrcDomain();
		
		StringBuffer poly = new StringBuffer();
		
		//Parameter declaration
		List<IVariable> params = am.getParams();
		if (params != null && params.size() > 0) {
			poly.append(IVariablestoString(0, params) + " -> ");
		}
		
		poly.append("{");
		//Indices
		poly.append(IVariablestoString(params.size(), am.getIndices()));
		poly.append(" -> ");
		//Expressions
		poly.append("[");
		boolean first = true;
		for (IntLinearExpression ile : am.getFunctions()) {
			if (!first) {
				poly.append(",");
			}
			poly.append(IntLinearExpressionToString(am.getParams(), am.getIndices(), ile));
//			if (naming == NAMING_SCHEME.GIVEN_NAME) {
//				poly.append(IntegerLinearAlgebraPrinter.toString(ie, FORMAT.ISL));
//			} else {
//				int count = 0;
//				String[] names = new String[am.getParams().size() + am.getIndices().size()];
//				//force the use of dimension based names in case of symbol variables
//				for (IVariable iv : am.getParams()) {
//					if (iv instanceof SymbolVariable) {
//						names[count] = ((SymbolVariable) iv).getName();
//						((SymbolVariable) iv).setName("i"+count);
//					}
//					count++;
//				}
//				for (IVariable iv : am.getIndices()) {
//					if (iv instanceof SymbolVariable) {
//						names[count] = ((SymbolVariable) iv).getName();
//						((SymbolVariable) iv).setName("i"+count);
//					}
//					count++;
//				}
//				poly.append(IntegerLinearAlgebraPrinter.toString(ie, FORMAT.ISL, null));
//				//restore the original names
//				count = 0;
//				for (IVariable iv : am.getParams()) {
//					if (iv instanceof SymbolVariable) {
//						((SymbolVariable) iv).setName(names[count]);
//					}
//					count++;
//				}
//				for (IVariable iv : am.getIndices()) {
//					if (iv instanceof SymbolVariable) {
//						((SymbolVariable) iv).setName(names[count]);
//					}
//					count++;
//				}
//			}
			first = false;
		}
		poly.append("]");
		
		poly.append("}");
		
		return poly.toString();
	}
	
	public String IntLinearExpressionToString(List<IVariable> params, List<IVariable> indices, IntLinearExpression ile) {

		StringBuffer sb = new StringBuffer();
		
		if (naming == NAMING_SCHEME.GIVEN_NAME) {
			sb.append(IntegerLinearAlgebraPrinter.toString(ile, FORMAT.ISL));
		} else {
			int count = 0;
			String[] names = new String[params.size() + indices.size()];
			//force the use of dimension based names in case of symbol variables
			for (IVariable iv : params) {
				if (iv instanceof SymbolVariable) {
					names[count] = ((SymbolVariable) iv).getName();
					((SymbolVariable) iv).setName("i"+count);
				}
				count++;
			}
			for (IVariable iv : indices) {
				if (iv instanceof SymbolVariable) {
					names[count] = ((SymbolVariable) iv).getName();
					((SymbolVariable) iv).setName("i"+count);
				}
				count++;
			}
			sb.append(IntegerLinearAlgebraPrinter.toString(ile, FORMAT.ISL, null));
			//restore the original names
			count = 0;
			for (IVariable iv : params) {
				if (iv instanceof SymbolVariable) {
					((SymbolVariable) iv).setName(names[count]);
				}
				count++;
			}
			for (IVariable iv : params) {
				if (iv instanceof SymbolVariable) {
					((SymbolVariable) iv).setName(names[count]);
				}
				count++;
			}
		}
		
		return sb.toString();
	}
}
