package org.polymodel.verifier.IF.ada.example;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.polymodel.verifier.DIM_TYPE;
import org.polymodel.verifier.VERBOSITY;
import org.polymodel.verifier.Verifier;
import org.polymodel.verifier.VerifierOutput;
import org.polymodel.verifier.IF.ada.VerifierInput;
import org.polymodel.verifier.message.IVerifierMessage;

import fr.irisa.cairn.model.polymodel.ada.ADAInput;
import fr.irisa.cairn.model.polymodel.ada.CandidateStatement;
import fr.irisa.cairn.model.polymodel.ada.factory.ADAUserFactory;
import fr.irisa.cairn.model.polymodel.isl.factory.ISLDefaultFactory;

public class VerifierExample {


	public static void main(String[] args) {
		matrix_multiply();
		hybrid_jacobi_gauss();
		forward_substitution();
		matrix_transpose();
		BTfragment();
	}

	/* ******************************************************
	 *  Matrix Multiply
	 *  
	 *  Standard matrix multiply as a simple example to
	 *  show the input to the verifier.
	 *  The is an illegal parallelization that parallelize
	 *  the k (accumulation) dimension.
	 *  
	 *  
	 *  void matrix_multiply(int** A, int** B, int** C, int N) {
	 *  	int i,j,k;
	 *  
	 *  	#pragma omp parallel for
	 *  	for (i = 0; i < N; i++) {
	 *  		for (j = 0; j < N; j++) {
	 *  S0			C[i][j] = 0;
	 *  			for (k = 0; k < N; k++) {
	 *  S1				C[i][j] += A[i][k] * B[k][j];
	 *  			}
	 *  		}
	 *  	}
	 *  }
	 ********************************************************/
	private static void matrix_multiply() {
		
		String[][] statements = new String[][] {
				new String[] {"S0", "[N] -> { [i,j] : 0<=i<N & 0<=j<N }", 
									"[N] -> { [i,j] -> [0,i,0,j,0,0,0] }"},
				new String[] {"S1", "[N] -> { [i,j,k] : 0<=i<N & 0<=j<N & 0<=k<N }", 
									"[N] -> { [i,j,k] -> [0,i,0,j,1,k,0] }"}
			};
		String[][] reads = new String[][] {
				new String[] {"S1", "A", "[N] -> { [i,j,k] -> [i,k] }"},
				new String[] {"S1", "B", "[N] -> { [i,j,k] -> [k,j] }"},
				new String[] {"S1", "C", "[N] -> { [i,j,k] -> [i,j] }"}
			};
		String[][] writes = new String[][] {
				new String[] {"S0", "C", "[N] -> { [i,j] -> [i,j] }"},
				new String[] {"S1", "C", "[N] -> { [i,j,k] -> [i,j] }"}
			};
		String[][] dims = new String[][] {
				//Legal 1D parallelization
				new String[] {"S0", "O,P,O,S,O,S,O"},
				new String[] {"S1", "O,P,O,S,O,S,O"},
				
				//illegal parallelization of k dimension
//				new String[] {"S0", "O,S,O,S,O,P,O"},
//				new String[] {"S1", "O,S,O,S,O,P,O"}
			};
		
		verify(statements, reads, writes, dims);
	}
	

	/* ******************************************************
	 *  Figure 1 : Hybrid Jacobi-Gauss Seidel stencil
	 *  
	 *  Since pointer swaps in standard implementations of Jacobi stencils 
	 *  cannot be detected as SCoP without sophisticated pointer analysis,
	 *  the program shown duplicates the body of stencil.
	 *  
	 *  For this program, only the innermost loop can be parallelized.
	 *  
	 *  void hybrid_jacobi_gauss_seidel(int** A, int** B, int T, int N) {
	 *  	int t,i,j;
	 *  	for (t = 0; t < T/2; t++) {
	 *  		for (i = 0; i < N; i++) {
	 *  			#pragma omp parallel for
	 *  			for (j = 0; j < N; j++) {
	 *  S0				A[i][j] = (B[i][j] + A[i-1][j] + B[i+1][j] + B[i][j-1] + B[i][j+1])*0.2;
	 *  			}
	 *  		}
	 *  		//Swap removed & replaced by bound fix & copy of the loop
	 *  		for (i = 0; i < N; i++) {
	 *  			#pragma omp parallel for
	 *  			for (j = 0; j < N; j++) {
	 *  S1				B[i][j] = (A[i][j] + B[i-1][j] + A[i+1][j] + A[i][j-1] + A[i][j+1])*0.2;
	 *  			}
	 *  		}
	 *  	}
	 *  }
	 ********************************************************/
	private static void hybrid_jacobi_gauss() {
		
		String[][] statements = new String[][] {
				new String[] {"S0", "[T,N] -> { [t,i,j] : 0<=2t<T & 0<=i<N & 0<=j<N }", 
									"[T,N] -> { [t,i,j] -> [0,t,0,i,0,j,0] }"},
				new String[] {"S1", "[T,N] -> { [t,i,j] : 0<=2t<T & 0<=i<N & 0<=j<N }", 
									"[T,N] -> { [t,i,j] -> [0,t,1,i,0,j,0] }"},
			};
		String[][] reads = new String[][] {
				new String[] {"S0", "A", "[T,N] -> { [t,i,j] -> [i-1,j] }"},
				new String[] {"S0", "B", "[T,N] -> { [t,i,j] -> [i,j] }"},
				new String[] {"S0", "B", "[T,N] -> { [t,i,j] -> [i+1,j] }"},
				new String[] {"S0", "B", "[T,N] -> { [t,i,j] -> [i,j-1] }"},
				new String[] {"S0", "B", "[T,N] -> { [t,i,j] -> [i,j+1] }"},
				
				new String[] {"S1", "B", "[T,N] -> { [t,i,j] -> [i-1,j] }"},
				new String[] {"S1", "A", "[T,N] -> { [t,i,j] -> [i,j] }"},
				new String[] {"S1", "A", "[T,N] -> { [t,i,j] -> [i+1,j] }"},
				new String[] {"S1", "A", "[T,N] -> { [t,i,j] -> [i,j-1] }"},
				new String[] {"S1", "A", "[T,N] -> { [t,i,j] -> [i,j+1] }"}
			};
		String[][] writes = new String[][] {
				new String[] {"S0", "A", "[T,N] -> { [t,i,j] -> [i,j] }"},
				new String[] {"S1", "B", "[T,N] -> { [t,i,j] -> [i,j] }"},
			};
		String[][] dims = new String[][] {
				//Legal parallelization of the innermost loops
				new String[] {"S0", "O,S,O,S,O,P,O"},
				new String[] {"S1", "O,S,O,S,O,P,O"},

				//Illegal parallelization of the i loop
//				new String[] {"S0", "O,S,O,P,O,S,O"},
//				new String[] {"S1", "O,S,O,P,O,S,O"},
			};
		
		verify(statements, reads, writes, dims);
	}
	

	/* ******************************************************
	 *  Figure 2 : Forward substitution		
	 *  
	 *  This program cannot be parallelized.
	 *  
	 *  void forward_substitution(int* x, int** L, int* b, int N) {
	 *  	int i,j;
	 *  	#pragma omp parallel for private(j)
	 *  	for (i = 0; i < N; i++) {
	 *  S0		x[i] = b[i];
	 *  		for (j = 0; j < i; j++) {
	 *  S1			x[i] = x[i] - L[i][j]*x[j];
	 *  		}
	 *  S2		x[i] = x[i]/L[i][i];
	 *  	}
	 *  }
	 ********************************************************/
	private static void forward_substitution() {
		
		String[][] statements = new String[][] {
				new String[] {"S0", "[N] -> { [i] : 0<=i<N }", 
									"[N] -> { [i] -> [0,i,0,0,0] }"},
				new String[] {"S1", "[N] -> { [i,j] : 0<=i<N & 0<=j<i }", 
									"[N] -> { [i,j] -> [0,i,1,j,0] }"},
				new String[] {"S2", "[N] -> { [i] : 0<=i<N }", 
									"[N] -> { [i] -> [0,i,2,0,0] }"},
			};
		String[][] reads = new String[][] {
				new String[] {"S0", "b", "[N] -> { [i] -> [i] }"},
				new String[] {"S1", "x", "[N] -> { [i,j] -> [i] }"},
				new String[] {"S1", "L", "[N] -> { [i,j] -> [i,j] }"},
				new String[] {"S1", "x", "[N] -> { [i,j] -> [j] }"},
				new String[] {"S2", "x", "[N] -> { [i] -> [i] }"},
				new String[] {"S2", "L", "[N] -> { [i] -> [i,i] }"},
			};
		String[][] writes = new String[][] {
				new String[] {"S0", "x", "[N] -> { [i] -> [i] }"},
				new String[] {"S1", "x", "[N] -> { [i,j] -> [i] }"},
				new String[] {"S2", "x", "[N] -> { [i] -> [i] }"},
			};
		String[][] dims = new String[][] {
				//Illegal parallelization of the i loop
				new String[] {"S0", "O,P,O,S,O"},
				new String[] {"S1", "O,P,O,S,O"},
				new String[] {"S2", "O,P,O,S,O"},
			};
		
		verify(statements, reads, writes, dims);
	}

	/* ******************************************************
	 *  Figure 3 : Matrix Transpose							
	 *  parallel directive does not alter the semantic,
	 *  iff temp is declared to be private.
	 *  
	 *  For verification purposes, variables that are 
	 *  declared to be private or reduction are treated 
	 *  to have individual copy in each thread 
	 *  (following OpenMP semantics).
	 *  Therefore, accesses to temp is treated as 
	 *  temp[p1][p2] for the example below.
	 *  
	 *  
	 *  void matrix_transpose(int** A, int N) {
	 *  	int p1,p2,temp;
	 *  
	 *  	#pragma omp parallel for private(temp)
	 *  	for (p1 = 0; p1 < N; p1++) {
	 *  		#pragma omp parallel for
	 *  		for (p2 = 0; p2 < p1; p2++) {
	 *  S0			temp = A[p1][p2];
	 *  S1			A[p1][p2] = A[p2][p1];
	 *  S2			A[p2][p1] = temp;
	 *  		}
	 *  	}
	 *  }
	 ********************************************************/
	private static void matrix_transpose() {
		
		String[][] statements = new String[][] {
				new String[] {"S0", "[N] -> { [p1,p2] : 0<=p1<N & 0<=p2<p1 }", 
									"[N] -> { [p1,p2] -> [0,p1,0,p2,0] }"},
				new String[] {"S1", "[N] -> { [p1,p2] : 0<=p1<N & 0<=p2<p1 }", 
									"[N] -> { [p1,p2] -> [0,p1,0,p2,1] }"},
				new String[] {"S2", "[N] -> { [p1,p2] : 0<=p1<N & 0<=p2<p1 }", 
									"[N] -> { [p1,p2] -> [0,p1,0,p2,2] }"},
			};
		String[][] reads = new String[][] {
				new String[] {"S0", "A", "[N] -> { [p1,p2] -> [p1,p2] }"},
				new String[] {"S1", "A", "[N] -> { [p1,p2] -> [p2,p1] }"},
				// legal case  : temp is expanded for the parallel dimensions to reflect private clause of OpenMP
				new String[] {"S2", "temp", "[N] -> { [p1,p2] -> [p1,p2]}"}, 
				// illegal case: when private clause is missing, temp is treated as scalar causing data race
//				new String[] {"S2", "temp", "[N] -> { [p1,p2] -> []}"}
			};
		String[][] writes = new String[][] {
				// same cases as above for writes to temp
				new String[] {"S0", "temp", "[N] -> { [p1,p2] -> [p1,p2]}"},
//				new String[] {"S0", "temp", "[N] -> { [p1,p2] -> [] }"},
				new String[] {"S1", "A", "[N] -> { [p1,p2] -> [p1,p2] }"},
				new String[] {"S2", "A", "[N] -> { [p1,p2] -> [p2,p1] }"}
			};
		String[][] dims = new String[][] {
				new String[] {"S0", "O,P,O,P,O"},
				new String[] {"S1", "O,P,O,P,O"},
				new String[] {"S2", "O,P,O,P,O"}
			};
		
		verify(statements, reads, writes, dims);
	}
	
	/* ******************************************************
	 *  Figure 5 : 	Fragment of BT from NPB					*
	 *  
	 *  The parallelization of outermost loop is parallel
	 *  if temp and TM are private to the threads.
	 *  The privatization is reflected as additional 
	 *  dimension in the memory mapping given.
	 *  
	 *  void BTfragment (double** TM, int NZ) {
	 *  	int k,m,n;
	 *  	double temp;
	 *  	#pragma omp parallel for private(m,n,temp,TM)
	 * 		for (k = 2; k < NZ; k++) {
	 * 			for (m = 1; m <= 5; m++) {
	 *				for (n = 1; n <= 5; n++) {
	 *	S0				TM[1][m] = n+m+k;
	 *	S1				TM[2][m] = n+m+k+2;
	 *	S2				TM[3][m] = n*2+m+k;
	 *	S3				TM[4][m] = n*n+m*m+k*k;
	 *	S4				TM[5][m] = n*n+m*m+k*k / 5;
	 *				}
	 *			}
	 *			//some other computation
	 *	S5		temp = 0;
	 *			for (m = 1; m <= 5; m++) {
	 *				for (n = 1; n <= 5; n++) {
	 *	S6				temp = temp + TM[n][m];
	 *				}
	 *			}
	 *		}
	 *	}
	 ********************************************************/
	private static void BTfragment() {
		
		String[][] statements = new String[][] {
				new String[] {"S0", "[NZ] -> { [k,m,n] : 2<=k<NZ & 1<=m<=5 & n<=1<=5 }", 
									"[NZ] -> { [k,m,n] -> [0,k,0,m,0,n,0] }"},
				new String[] {"S1", "[NZ] -> { [k,m,n] : 2<=k<NZ & 1<=m<=5 & n<=1<=5 }", 
									"[NZ] -> { [k,m,n] -> [0,k,0,m,0,n,1] }"},
				new String[] {"S2", "[NZ] -> { [k,m,n] : 2<=k<NZ & 1<=m<=5 & n<=1<=5 }", 
									"[NZ] -> { [k,m,n] -> [0,k,0,m,0,n,2] }"},
				new String[] {"S3", "[NZ] -> { [k,m,n] : 2<=k<NZ & 1<=m<=5 & n<=1<=5 }", 
									"[NZ] -> { [k,m,n] -> [0,k,0,m,0,n,3] }"},
				new String[] {"S4", "[NZ] -> { [k,m,n] : 2<=k<NZ & 1<=m<=5 & n<=1<=5 }", 
									"[NZ] -> { [k,m,n] -> [0,k,0,m,0,n,4] }"},

				new String[] {"S5", "[NZ] -> { [k] : 2<=k<NZ }", 
									"[NZ] -> { [k] -> [0,k,1,0,0,0,0] }"},

				new String[] {"S6", "[NZ] -> { [k,m,n] : 2<=k<NZ & 1<=m<=5 & n<=1<=5 }", 
									"[NZ] -> { [k,m,n] -> [0,k,2,m,0,n,0] }"},
			};
		String[][] reads = new String[][] {
				new String[] {"S6", "temp", "[NZ] -> { [k,m,n] -> [k] }"},
				new String[] {"S6", "TM",   "[NZ] -> { [k,m,n] -> [k,n,m] }"},
			};
		String[][] writes = new String[][] {
				new String[] {"S0", "TM", "[NZ] -> { [k,m,n] -> [k,1,m] }"},
				new String[] {"S1", "TM", "[NZ] -> { [k,m,n] -> [k,2,m] }"},
				new String[] {"S2", "TM", "[NZ] -> { [k,m,n] -> [k,3,m] }"},
				new String[] {"S3", "TM", "[NZ] -> { [k,m,n] -> [k,4,m] }"},
				new String[] {"S4", "TM", "[NZ] -> { [k,m,n] -> [k,5,m] }"},
				new String[] {"S5", "temp", "[NZ] -> { [k] -> [k] }"},
				new String[] {"S6", "temp", "[NZ] -> { [k,m,n] -> [k] }"},
			};
		String[][] dims = new String[][] {
				new String[] {"S0", "O,P,O,S,O,S,O"},
				new String[] {"S1", "O,P,O,S,O,S,O"},
				new String[] {"S2", "O,P,O,S,O,S,O"},
				new String[] {"S3", "O,P,O,S,O,S,O"},
				new String[] {"S4", "O,P,O,S,O,S,O"},
				new String[] {"S5", "O,P,O,S,O,S,O"},
				new String[] {"S6", "O,P,O,S,O,S,O"},
			};
		
		verify(statements, reads, writes, dims);
	}
	

	/***
	 * Creates ADAInput and list of dimension types from given Strings, runs the verifier and prints out the result.
	 * 
	 * @param statements
	 * @param reads
	 * @param writes
	 * @param dims
	 */
	private static void verify(String[][] statements, String[][] reads, String[][] writes, String[][] dims) {
		ADAInput adaInput = ADAUserFactory.createADAInputFromStrings(statements, reads, writes);
		
		System.out.println(adaInput);
		
		Map<CandidateStatement, List<DIM_TYPE>> dimTypes = parseDimTypes(adaInput, dims);
		
		VerifierInput input = VerifierInput.build(adaInput, dimTypes);
		
		VerifierOutput output = Verifier.verify(ISLDefaultFactory.INSTANCE, input.prdg, input.schedules, input.memoryMaps, input.dimTypes);
		System.out.println(output);
	}
	
	/**
	 * Takes Array of String[] where the first element is the statementID and the second element is the comma delimited list of dimension types.
	 * Returns a mapping from statements to List<DIM_TYPE> to be given to the verifier.
	 * 
	 * @param adaInput
	 * @param dims
	 * @return
	 */
	private static Map<CandidateStatement, List<DIM_TYPE>> parseDimTypes(ADAInput adaInput, String[][] dims) {
		Map<CandidateStatement, List<DIM_TYPE>> dimTypes = new HashMap<CandidateStatement, List<DIM_TYPE>>();
		
		for (String[] dim : dims) {
			CandidateStatement stmt = adaInput.getStatement(dim[0]);
			dimTypes.put(stmt, parseDimTypeList(dim[1]));
		}
		
		return dimTypes;
	}
	
	/**
	 * Takes comma delimited list of dimension types (S,O,P) corresponding to Sequential, Ordering, and Parallel respectively, and returns List<DIM_TYPE> accordingly.
	 * 
	 * @param dims
	 * @return
	 */
	private static List<DIM_TYPE> parseDimTypeList(String dims) {
		List<DIM_TYPE> dimTypes = new ArrayList<DIM_TYPE>();
		
		for (String s : dims.split("\\s*,\\s*")) {
			if (s.toUpperCase().contentEquals("S")) {
				dimTypes.add(DIM_TYPE.SEQUENTIAL);
			} else if (s.toUpperCase().contentEquals("O")) {
				dimTypes.add(DIM_TYPE.ORDERING);
			} else if (s.toUpperCase().contentEquals("P")) {
				dimTypes.add(DIM_TYPE.PARALLEL);
			} else {
				throw new RuntimeException("Unexpected dimension type : " + s);
			}
		}
		
		return dimTypes;
	}
}
