#include "CEBLModel.hpp"
#include "model/FileUtils.hpp"
#include "model/ChannelsConfig.hpp"
#include "model/DeviceConfig.hpp"
#include "model/FileDataStreamConfig.hpp"
#include "model/FilterConfig.hpp"
#include "model/FeaturesConfig.hpp"
#include "model/DecisionConfig.hpp"
#include "model/ClassifiersConfig.hpp"
#include "model/Training.hpp"
#include "model/RealTimeClassification.hpp"
#include "model/DataSource.hpp"
#include "model/DataProcess.hpp"
#include "model/SessionManager.hpp"
#include "model/Preferences.hpp"
#include "model/StringTable.hpp"
#include "model/DataIO.hpp"
#include "CompiledStrings.hpp"

//----------------------------------------------------------------------
// Constructors / Destructors

CEBLModel::CEBLModel()
{
  //set this to false to begin with
  this->initialized_successfully = false;

  //initialize preferences
  preferences = new Preferences();
}

void  CEBLModel::InitModel(int ac, char ** av)
  {
  //set this to false to begin with
  this->initialized_successfully = false;

  //parse command line options
  if(!preferences->processCL(ac,av))
    {
      return;
    }
  //init cebl
  preferences->initCEBL();

  //create all internal objects
  try
    {
      //initialize string table
      string_table = new StringTable();
      string_table->loadFromString(string_table_en);
      //string_table->loadFromFile(preferences->getStringTableFilename());


      channels = new ChannelsConfig(this);
      device = new DeviceConfig(this);
      file_data_stream_config = new FileDataStreamConfig(this);
      filters = new FilterConfig(this);
      features = new FeaturesConfig(this);
      decisions = new DecisionConfig(this);
      classifiers = new ClassifiersConfig(this);
      training = new Training(this);
      realtime = new RealTimeClassification(this);
      data_source = new DataSource(this);
      data_process = new DataProcess(this);
      session_manager = new SessionManager(this);
    }
  catch(exception &e)
    {
      cerr << "Exception occured when trying to initialize the CEBL model: " << e.what() << "\n";
      return;
    }

  //now load session file, if specified
  if(preferences->getSessionFilename() != "")
    {
      try
        {
          sessionLoad(preferences->getSessionFilename());
    }
      catch(...)
        {
          cerr << "Failed to load session file " << preferences->getSessionFilename() << "\n";
        }
    }
  //if we have gotten this far, allow cebl to continue
  this->initialized_successfully = true;
}


CEBLModel::~CEBLModel()
{
  delete channels;
  delete device;
  delete file_data_stream_config;
  delete filters;
  delete features;
  delete decisions;
  delete classifiers;
  delete training;
  delete realtime;
  delete data_source;
  delete data_process;
  delete session_manager;
  delete preferences;
  delete string_table;
}



//======================================================================


void CEBLModel::preferencesProcessCL(int ac, char ** av)
{
  preferences->processCL(ac,av);
}

void CEBLModel::preferencesInitCEBL()
{
  preferences->initCEBL();
}

std::vector<std::string> CEBLModel::preferencesGetPaths()
{
  return preferences->getPaths();
}

void CEBLModel::preferencesAddOption(string name,
                                     string description,
                                     char short_name)
{
  preferences->addOption(name,description,short_name);
}

string CEBLModel::preferencesGetOption(string name)
{
  return preferences->getOption(name);
}

//======================================================================
// STRING TABLE

const char * CEBLModel::getString(string name)
{
  try
    {
      return string_table->getString(name);
    }
  catch(exception &e)
    {
      cerr << e.what();
      return "";
    }
}

//======================================================================
// DATA SOURCE




//----------------------------------------
//GETTING OPERATIONS


EEGData CEBLModel::dataReadRaw(int samples)
{
  EEGData ret = data_source->readAll();
  return ret;
}

EEGData CEBLModel::dataRead(int samples)
{
  EEGData ret = data_source->readAll();
  ret = data_process->process(ret);
  return ret;
}

EEGData CEBLModel::dataReadAllRaw()
{
  EEGData ret = data_source->readAll();
  return ret;
}

EEGData CEBLModel::dataReadAll()
{
  EEGData ret = data_source->readAll();
  ret = data_process->process(ret);
  return ret;
}

int CEBLModel::dataSamplesAvailable()
{
  return data_source->samplesAvailable();
}

std::vector<string> CEBLModel::dataGetSources()
{
  return data_source->getSources();
}

int CEBLModel::dataGetSource()
{
  return data_source->getSource();
}

bool CEBLModel::dataSourceReady()
{
  return data_source->sourceReady();
}

bool CEBLModel::dataIsStarted()
{
  return data_source->isStarted();
}

bool CEBLModel::dataGetStoreFlag()
{
  return data_source->getStoreFlag();
}

int CEBLModel::dataGetStoreNumSamples()
{
  return data_source->getStoreNumSamples();
}

EEGData CEBLModel::dataGetStoredData()
{
  return data_source->getStoredData();
}


//----------------------------------------
//SETTING OPERATIONS


void CEBLModel::dataClearStoredData()
{
  data_source->clearStoredData();
}

void CEBLModel::dataSetStoreFlag(bool flag)
{
  data_source->setStoreFlag(flag);
}

void CEBLModel::dataSetSource(int n)
{
  data_source->setSource(n);
}

void CEBLModel::dataSetSource(string source)
{
  data_source->setSource(source);
}

void CEBLModel::dataClearSamples()
{
  data_source->clearSamples();
}

void CEBLModel::dataStart()
{
  data_source->start();
}

void CEBLModel::dataStop()
{
  data_source->stop();
}




//======================================================================
// DATA PROCESS




//----------------------------------------
//GETTING OPERATIONS


bool CEBLModel::processGetReferenceEnabled()
{
  return data_process->getReferenceEnabled();
}

bool CEBLModel::processGetRemoveEnabled()
{
  return data_process->getRemoveEnabled();
}

bool CEBLModel::processGetFilterEnabled()
{
  return data_process->getFilterEnabled();
}


//----------------------------------------
//SETTING OPERATIONS


void CEBLModel::processSetReferenceEnabled(bool enabled)
{
  data_process->setReferenceEnabled(enabled);
}

void CEBLModel::processSetRemoveEnabled(bool enabled)
{
  data_process->setRemoveEnabled(enabled);
}

void CEBLModel::processSetFilterEnabled(bool enabled)
{
  data_process->setFilterEnabled(enabled);
}


//----------------------------------------
//PROCESS DATA OPERATIONS

EEGTrainingData CEBLModel::processData(EEGTrainingData &data)
{
  return data_process->process(data);
}

EEGData& CEBLModel::processData(EEGData &data)
{
  return data_process->process(data);
}

EEGData& CEBLModel::processData(EEGData &data, bool remove_disabled, bool reference, bool filter)
{
  return data_process->process(data, remove_disabled, reference, filter);
}




//======================================================================
// CHANNELS




//----------------------------------------
//GETTING OPERATIONS


string CEBLModel::channelsGetConfigFilename()
{
  return channels->getCurrentFilename();
}

bool CEBLModel::channelsConfigFileExists(string filename)
{
  return FileUtils::fileExists(filename);
}

string CEBLModel::channelsGetElectrodeName(int electrode)
{
  return channels->getElectrodeName(electrode);
}

bool CEBLModel::channelsGetElectrodeReference(int electrode)
{
  return channels->getElectrodeReference(electrode);
}

bool CEBLModel::channelsGetElectrodeEnabled(int electrode)
{
  return channels->getElectrodeEnabled(electrode);
}

int CEBLModel::channelsGetMaxNumChannels()
{
  return channels->getMaxNumChannels();
}

int CEBLModel::channelsGetNumEnabledChannels()
{
  return channels->getNumEnabled();
}

std::vector<string> CEBLModel::channelsGetEnabledNames()
{
  return channels->getEnabledNames();
}

string CEBLModel::channelsGetConfigurationString()
{
  return channels->getConfigurationString();
}


//----------------------------------------
//SETTING OPERATIONS


void CEBLModel::channelsLoadFile(string filename)
{
  channels->loadFile(filename);
}

void CEBLModel::channelsSaveFile(string filename)
{
  channels->saveFile(filename);
}

void CEBLModel::channelsSetElectrodeName(int electrode, string name)
{
  channels->setElectrodeName(electrode, name);
}

void CEBLModel::channelsSetElectrodeReference(int electrode, bool reference)
{
  channels->setElectrodeReference(electrode, reference);
}

void CEBLModel::channelsSetElectrodeEnabled(int electrode, bool enabled)
{
  channels->setElectrodeEnabled(electrode, enabled);
}

void CEBLModel::channelsSetConfigurationFromString(string config)
{
  channels->setConfigurationFromString(config);
}




//======================================================================
// DEVICE




//----------------------------------------
//GETTING OPERATIONS


string CEBLModel::deviceGetLocation()
{
  return device->getLocation();
}

bool CEBLModel::deviceIsReady()
{
  return device->isReady();
}

string CEBLModel::deviceGetError()
{
  return device->getError();
}

string CEBLModel::deviceGetInquiry()
{
  return device->getInquiry();
}

int CEBLModel::deviceGetSampleRate()
{
  return device->getSampleRate();
}

int CEBLModel::deviceGetBlockSize()
{
  return device->getBlockSize();
}

bool CEBLModel::deviceExists()
{
  return device->exists();
}


//----------------------------------------
//SETTING OPERATIONS


void CEBLModel::deviceSetDeviceLocation(string filename)
{
  device->setDeviceLocation(filename);
}

void CEBLModel::deviceSetSampleRate(int sample_rate)
{
  device->setSampleRate(sample_rate);
}

void CEBLModel::deviceSetBlockSize(int block_size)
{
  device->setBlockSize(block_size);
}

void CEBLModel::deviceSearch()
{
  device->scanForDevices();
}





//======================================================================
//File Data Stream




//----------------------------------------
//GETTING OPERATIONS


string CEBLModel::fileStreamGetFilename()
{
  return this->file_data_stream_config->getFilename();
}

bool CEBLModel::fileStreamIsReady()
{
  return this->file_data_stream_config->isReady();
}

int CEBLModel::fileStreamGetSampleRate()
{
  return this->file_data_stream_config->getSampleRate();
}

int CEBLModel::fileStreamGetNumSamples()
{
  return this->file_data_stream_config->getNumSamples();
}

int CEBLModel::fileStreamGetNumChannels()
{
  return this->file_data_stream_config->getNumChannels();
}

int CEBLModel::fileStreamGetNumClasses()
{
  return this->file_data_stream_config->getNumClasses();
}

int CEBLModel::fileStreamGetNumSequences()
{
  return this->file_data_stream_config->getNumSequences();
}

EEGTrainingData & CEBLModel::fileStreamGetTrainingData()
{
  return this->file_data_stream_config->getTrainingData();
}

EEGData & CEBLModel::fileStreamGetData()
{
  return this->file_data_stream_config->getData();
}


//----------------------------------------
//SETTING OPERATIONS


void CEBLModel::fileStreamOpenFile(string filename)
{
  this->file_data_stream_config->openFile(filename);
}

void CEBLModel::fileStreamSetSampleRate(int sample_rate)
{
  this->file_data_stream_config->setSampleRate(sample_rate);
}




//======================================================================
// SESSION SAVING and LOADING




//----------------------------------------
//GETTING OPERATIONS


bool CEBLModel::sessionShouldSaveAs()
{
  return session_manager->shouldSaveAs();
}

//----------------------------------------
//SETTING OPERATIONS


void CEBLModel::sessionSave()
{
  session_manager->save();
}

void CEBLModel::sessionSaveAs(string filename)
{
  session_manager->saveAs(filename);
}

void CEBLModel::sessionLoad(string filename)
{
  session_manager->load(filename);
}



//======================================================================
//FILTER


//----------------------------------------
//GETTING OPERATIONS


std::vector<string> CEBLModel::filterGetNameList()
{
  return filters->getNameList();
}

std::vector<string> CEBLModel::filterGetPathList()
{
  return filters->getPathList();
}

bool CEBLModel::filterIsTrained(string filter)
{
  return filters->isTrained(filter);
}

int CEBLModel::filterGetNumLags()
{
  return filters->getNumLags();
}

string CEBLModel::filterGetSelected()
{
  return filters->getSelected();
}

std::vector<int> CEBLModel::filterGetSelectedComponents()
{
  return filters->getSelectedComponents();
}

EEGData CEBLModel::filterGetComponents(EEGData training_data)
{
  return filters->getComponents(training_data);
}

string CEBLModel::filterGetSelectedComponentsString()
{
  return filters->getSelectedComponentsString();
}

bool CEBLModel::filterGetSelectedComponentsValid()
{
  return filters->getSelectedComponentsValid();
}

EEGData CEBLModel::filterApply(EEGData data)
{
  return filters->apply(data);
}

int CEBLModel::filterGetNumExpectedChannels()
{
  return filters->getNumExpectedChannels();
}

ublas::matrix<double> CEBLModel::filterGetFilterMatrix()
{
  return filters->getFilterMatrix();
}


//----------------------------------------
//SETTING OPERATIONS


void CEBLModel::filterSetSelectedComponentsString(string components)
{
  filters->setSelectedComponentsString(components);
}

void CEBLModel::filterSetSelected(string filter)
{
  filters->setSelected(filter);
}

void CEBLModel::filterTrain(EEGData training_data, string filter)
{ 
  filters->train(training_data, filter);
}

void CEBLModel::filterSetNumLags(int n)
{
  filters->setNumLags(n);
}


//======================================================================
//FEATURES


//----------------------------------------
//GETTING OPERATIONS


std::vector<string> CEBLModel::featuresGetNameList()
{
  return features->getNameList();
}

std::vector<string> CEBLModel::featuresGetPathList()
{
  return features->getPathList();
}

bool CEBLModel::featureIsTrained(string feature)
{
  return features->isTrained(feature);
}

string CEBLModel::featuresGetSelected()
{
  return features->getSelected();
}

std::map<std::string, CEBL::Param> CEBLModel::featureGetParams(string feature)
{
  return features->getParams(feature);
}


//----------------------------------------
//SETTING OPERATIONS


void CEBLModel::featuresSetSelected(string feature)
{
  features->setSelected(feature);
}

void CEBLModel::featureTrain(string feature)
{
  features->train(feature);
}

void CEBLModel::featureReset(string feature)
{
  features->reset(feature);
}

void CEBLModel::featureSetParams(std::map<std::string, CEBL::Param> params, string feature)
{
  features->setParams(params,feature);
}

//----------------------------------------
//USE FEATURE

EEGData CEBLModel::featuresExtract(EEGData &data)
{
  return features->extract(data);
}
EEGTrainingData CEBLModel::featuresExtract(EEGTrainingData &data)
{
  return features->extract(data);
}

void CEBLModel::featuresHalt()
{
  features->halt();
}



//======================================================================
//FEATURES


//----------------------------------------
//GETTING OPERATIONS
std::vector<string> CEBLModel::decisionGetNameList()
{
  return decisions->getNameList();
}

std::vector<string> CEBLModel::decisionGetPathList()
{
  return decisions->getPathList();
}

string CEBLModel::decisionGetSelected()
{
  return decisions->getSelected();
}

std::map<std::string, CEBL::Param> CEBLModel::decisionGetParams(string decision)
{
  return decisions->getParams(decision);
}


//----------------------------------------
//SETTING OPERATIONS


void CEBLModel::decisionSetSelected(string feature)
{
  decisions->setSelected(feature);
}

void CEBLModel::decisionSetParams(std::map<std::string, CEBL::Param> params, string decision)
{
  decisions->setParams(params,decision);
}
//DECISION OPERATIONS

void CEBLModel::decisionUpdateWithProbabilities
(std::vector<std::vector<double> >probs)
{
  decisions->updateWithProbabilities(probs);
}

void CEBLModel::decisionUpdateWithProbabilities(std::vector<double> probs)
{
  decisions->updateWithProbabilities(probs);
}

void CEBLModel::decisionUpdateWithClassification
(ublas::vector<int> classes)
{
  decisions->updateWithClassification(classes);
}

void CEBLModel::decisionUpdateWithClassification(int cls)
{
  decisions->updateWithClassification(cls);
}

void CEBLModel::decisionInit(int num_classes)
{
  decisions->init(num_classes);
}

std::vector<double> CEBLModel::decisionDecideClasses()
{
  return decisions->decideClasses();
}

//======================================================================
//CLASSIFIERS



//----------------------------------------
//GETTING OPERATIONS

std::vector<string> CEBLModel::classifiersGetNameList()
{
  return classifiers->getNameList();
}

std::vector<string> CEBLModel::classifiersGetPathList()
{
  return classifiers->getPathList();
}

bool CEBLModel::classifierIsTrained(string classifier)
{
  return classifiers->isTrained(classifier);
}

std::map<std::string, CEBL::Param> CEBLModel::classifierGetParams(string classifier)
{
  return classifiers->getParams(classifier);
}

string CEBLModel::classifiersGetSelected()
{
  return classifiers->getSelected();
}

bool CEBLModel::classifierGetUseProbs()
{
  return classifiers->getUseProbs();
}

std::vector<std::vector<double> >  CEBLModel::classifierGetLastProbs()
{
  return classifiers->getLastProbs();
}

int CEBLModel::classifierGetTrainedClasses()
{
  return classifiers->getTrainedClasses();
}

int CEBLModel::classifierGetTrainedLags()
{
  return classifiers->getTrainedLags();
}

//----------------------------------------
//SETTING OPERATIONS


void CEBLModel::classifierReset(CEBL::Param params, string classifier)
{
  classifiers->reset(params);
}

void CEBLModel::classifiersSetSelected(string classifier)
{
  classifiers->setSelected(classifier);
}


void CEBLModel::classifierTrain(EEGTrainingData &training_data, string classifier)
{
  classifiers->train(training_data,classifier);
}

void CEBLModel::classifierHaltTrain()
{
  classifiers->haltTrain();
}

void CEBLModel::classifierSetParams(std::map<std::string, CEBL::Param> params, string classifier)
{
  classifiers->setParams(params,classifier);
}

void CEBLModel::classifierSetUseProbs(bool flag)
{
  classifiers->setUseProbs(flag);
}

//----------------------------------------
//USE CLASSIFIER
ublas::vector<int> CEBLModel::classifierUse(EEGData &data)
{
  return classifiers->use(data);
}

//======================================================================
// Training

//----------------------------------------
//GETTING OPERATIONS

std::vector<string> CEBLModel::trainingGetClassLabels()
{
  return training->getClassLabels();
}

string CEBLModel::trainingGetClassLabel(int class_num)
{
  return training->getClassLabel(class_num);
}

int CEBLModel::trainingGetNumClasses()
{
  return training->getNumClasses();
}

int CEBLModel::trainingGetNumSequences()
{
  return training->getNumSequences();
}

int CEBLModel::trainingGetSequenceLength()
{
  return training->getSequenceLength();
}

int CEBLModel::trainingGetPauseLength()
{
  return training->getPauseLength();
}

EEGTrainingData CEBLModel::trainingGetData()
{
  return training->getData();
}

bool CEBLModel::trainingDataIsLoaded()
{
  return training->dataIsLoaded();
}

bool CEBLModel::trainingIsDataFileLoaded()
{
  return training->isDataFileLoaded();
}

string CEBLModel::trainingGetDataFilename()
{
  return training->getDataFilename();
}

bool CEBLModel::trainingIsActive()
{
  return training->isActive();
}

bool CEBLModel::trainingFailed()
{
  return training->failed();
}

string CEBLModel::trainingGetFailureMessage()
{
  return training->getFailureMessage();
}

bool CEBLModel::trainingIsPaused()
{
  return training->isPaused();
}

int CEBLModel::trainingGetTrainingClass()
{
  return training->getTrainingClass();
}

int CEBLModel::trainingGetTrainingSequence()
{
  return training->getTrainingSequence();
}

bool CEBLModel::trainingFeedbackEnabled()
{
  return training->feedbackEnabled();
}
bool CEBLModel::trainingIsTrainingClassifier()
{
  return training->isTrainingClassifier();
}
std::vector<double> CEBLModel::trainingGetClassProportions()
{
  return training->getClassProportions();
}


//----------------------------------------
//SETTING OPERATIONS

void CEBLModel::trainingStart()
{
  training->start();
}

void CEBLModel::trainingStop()
{
  training->stop();
}

void CEBLModel::trainingSetNumClasses(int n)
{
  training->setNumClasses(n);
}

void CEBLModel::trainingSetNumSequences(int n)
{
  training->setNumSequences(n);
}

void CEBLModel::trainingSetSequenceLength(int n)
{
  training->setSequenceLength(n);
}

void CEBLModel::trainingSetPauseLength(int n)
{
  training->setPauseLength(n);
}

void CEBLModel::trainingSetClassLabels(std::vector<string> labels)
{
  training->setClassLabels(labels);
}

void CEBLModel::trainingSetClassLabel(int class_number, string label)
{
  training->setClassLabel(class_number,label);
}

void CEBLModel::trainingLoadData(string filename)
{
  training->loadData(filename);
}

void CEBLModel::trainingClearData()
{
  training->clearData();
}

void CEBLModel::trainingSaveData(string filename)
{
  training->saveData(filename);
}

void CEBLModel::trainingSetFeedbackEnabled(bool flag)
{
  training->setFeedbackEnabled(flag);
}

//======================================================================
// RealTime Classification


//----------------------------------------
//GETTING OPERATIONS

bool CEBLModel::realtimeIsReady()
{
  return realtime->isReady();
}

bool CEBLModel::realtimeLastTrainFailed()
{
  return realtime->lastTrainFailed();
}

bool CEBLModel::realtimeIsClassifying()
{
  return realtime->isClassifying();
}

std::vector<int> CEBLModel::realtimeReadClassificationQueue()
{
  return realtime->readClassificationQueue();
}

std::vector<double> CEBLModel::realtimeGetClassProportions()
{
  return realtime->getClassProportions();
}

std::vector<int> CEBLModel::realtimePeekClassificationQueue() const
{
  return realtime->peekClassificationQueue();
}

int CEBLModel::realtimeGetSelectedClass() const
{
  return realtime->getSelectedClass();
}



//----------------------------------------
//SETTING OPERATIONS


void CEBLModel::realtimeClearClassificationQueue()
{
  realtime->clearClassificationQueue();
}


//CONTROL CLASSIFICATION
void CEBLModel::realtimeTrainClassifier()
{
  realtime->trainClassifier();
}

void CEBLModel::realtimeStartClassifying()
{
  realtime->startClassifying();
}

void CEBLModel::realtimeStopClassifying()
{
  realtime->stopClassifying();
}

void CEBLModel::realtimeTrainClassifierThreaded()
{
  realtime->trainClassifierThreaded();
}

void CEBLModel::realtimeTrainClassifierHalt()
{
  realtime->trainClassifierHalt();
}

bool CEBLModel::realtimeIsTrainingClassifier()
{
  return realtime->isTrainingClassifier();
}
void CEBLModel::realtimeClearSelectedClass()
{
  realtime->clearSelectedClass();
}

//======================================================================
// DATA FUNCTIONS
  
EEGTrainingData CEBLModel::dataLoadTrainingDataFile(string filename)
{
  return DataIO::loadTrainingDataFromFile(filename);
}
