/*
* CEBL : CSU EEG Brain-Computer Interface Lab
*
* Author: Jeshua Bratman - jeshuabratman@gmail.com
*
* This file is part of CEBL.
*
* CEBL is free software; you can redistribute it and/or modify it.
* We only ask that if you use our code that you cite the source in
* your project or publication.
*
* EEG Group (www.cs.colostate.edu/eeg)
* Department of Computer Science
* Colorado State University
* 
*/

/*! CEBLModel.hpp
 * \class CEBLModel
 * \author Jeshua Bratman
 *
 * Main internal CEBL model.
 */

#ifndef CEBLMODEL_H
#define CEBLMODEL_H

#include "CEBLIncludes.hpp"
#include "model/EEGData.hpp"
#include "model/EEGTrainingData.hpp"
#include "Param.hpp"


//forward declarations of model internals
class ChannelsConfig;
class FilterConfig;
class FeaturesConfig;
class DecisionConfig;
class ClassifiersConfig;
class Training;
class RealTimeClassification;
class DeviceConfig;
class FileDataStreamConfig;
class DataSource;
class DataProcess;
class SessionManager;
class StringTable;
class Preferences;


class CEBLModel
{
private:
  ChannelsConfig * channels;
  DeviceConfig * device;
  FileDataStreamConfig * file_data_stream_config;
  FilterConfig * filters;
  FeaturesConfig * features;
  DecisionConfig * decisions;
  ClassifiersConfig * classifiers;
  Training * training;
  RealTimeClassification * realtime;
  DataSource *data_source;
  DataProcess *data_process;
  SessionManager * session_manager;
  Preferences * preferences;
  StringTable * string_table;

  //flags
  bool initialized_successfully;

public:
  CEBLModel();
  void InitModel(int ac, char **av);
  ~CEBLModel();
  bool initializedSuccessfully()  {return initialized_successfully;}

  ChannelsConfig *getChannelsConfig() { return channels; }
  DeviceConfig *getDeviceConfig() { return device; }
  FileDataStreamConfig *getFileDataStreamConfig() { return file_data_stream_config; }
  DataSource *getDataSource() { return data_source; }
  DataProcess *getDataProcess() { return data_process; }
  FilterConfig *getFilterConfig() { return filters; }
  FeaturesConfig *getFeaturesConfig() { return features; }
  DecisionConfig *getDecisionConfig() { return decisions; }
  ClassifiersConfig *getClassifiersConfig() { return classifiers; }
  Preferences *getPreferences() { return preferences; }
  Training *getTraining() { return training; }

  //preferences
  void preferencesProcessCL(int ac, char ** av);
  void preferencesInitCEBL();
  std::vector<std::string> preferencesGetPaths();

  /*! Add an option to preferences which can be set via the command lien
    \param name name of option
    \param description 
    \param short_name one-letter name for option
  */
  void preferencesAddOption(string name,
                            string description,
                            char short_name='\0');


  /*! Get the value of a set option.
    \param name 
    
    \return value of option or the string "None" if it was not set
  */
  string preferencesGetOption(string name);
  
  //string table
  const char *getString(string name);

  //======================================================================
  // DATA SOURCE

  //----------------------------------------
  //GETTING  OPERATIONS

  /*! Reads a specified number of samples from data source.
    \param samples number of samples

    \return eeg data
  */
  EEGData dataReadRaw(int samples);

  /*! Reads and processes a specified number of samples from data source.
    \param samples number of samples

    \return eeg data
  */
  EEGData dataRead(int samples);

  /*! Reads all data from data source.

    \return eeg data from source
  */
  EEGData dataReadAllRaw();

  /*! Reads and processes all data from data source.

    \return eeg data from source
  */
  EEGData dataReadAll();

  /*! Check how many samples are available.

    \return number of available samples
  */
  int dataSamplesAvailable();

  /*! Get the list of data sources.

    \return vector of strings representing all available data sources
  */
  std::vector<string> dataGetSources();


  /*! Get selected source index.

    \return index of source
  */
  int dataGetSource();

  /*! Check if data source is ready.

    \return flag
  */
  bool dataSourceReady();

  /*! Check if data source is currently recording.

    \return flag
  */
  bool dataIsStarted();

  /*! Check if data is being buffered as it is read from data source.

    \return flag
  */
  bool dataGetStoreFlag();

  /*! Get number of stored samples in stored data buffer.

    \return samples
  */
  int dataGetStoreNumSamples();

  /*! Get stored data buffer.

    \return eeg data buffer
  */
  EEGData dataGetStoredData();


  //----------------------------------------
  //SETTING OPERATIONS

  /*! Remove all stored data.
   */
  void dataClearStoredData();

  /*! Set flag indicating whether or not to store data as it is read from data source.
    \param flag
  */
  void dataSetStoreFlag(bool flag);

  /*! Set data source by index.
    \param n index of data source
  */
  void dataSetSource(int n);

  /*! Set data source by name.
    \param source name of data source
  */
  void dataSetSource(string source);

  /*! Clear buffered samples.
   */
  void dataClearSamples();

  /*! Start the data source recording.
   */
  void dataStart();

  /*! Stop the data source recording.
   */
  void dataStop();

  //======================================================================
  //CHANNELS

  //----------------------------------------
  //GETTING OPERATIONS
  /*! Gets currently loaded config filename.

    \return filename
  */
  string channelsGetConfigFilename();

  /*! Checks if specified filename exists.
    \param filename filename to check

    \return file exists or does not
  */
  bool channelsConfigFileExists(string filename);

  /*! Returns name of specified electrode.
    \param electrode electrode index

    \return name
  */
  string channelsGetElectrodeName(int electrode);

  /*! Returns reference status of specified electrode.
    \param electrode electrode index

    \return reference
  */
  bool channelsGetElectrodeReference(int electrode);

  /*! Returns enabled status of specified electrode.
    \param electrode electrode index

    \return enabled
  */
  bool channelsGetElectrodeEnabled(int electrode);

  /*! Maximum number of channels that CEBL can handle.

    \return number of channels
  */
  int channelsGetMaxNumChannels();

  /*! Get number of channels currently enabled.

    \return number of channels
  */
  int channelsGetNumEnabledChannels();

  /*! Gets the list of channel names.

    \return names
  */
  std::vector<string> channelsGetEnabledNames();

  /*! Returns string of complete channel configuration file.

    \return config
  */
  string channelsGetConfigurationString();

  /*! Get reference status.

    \return reference
  */
  bool processGetReferenceEnabled();

  /*! Get enabled status.

    \return enabled
  */
  bool processGetRemoveEnabled();

  /*! Get filter flag.

    \return filter
  */
  bool processGetFilterEnabled();



  //----------------------------------------
  //SETTING OPERATIONS

  /*! Loads channels configuration file.
    \param filename file to load
  */
  void channelsLoadFile(string filename);

  /*! Save channels cofnigurationt o file.
    \param filename
  */
  void channelsSaveFile(string filename);

  /*! Set the name of specified electrode.
    \param electrode electrode index
    \param name
  */
  void channelsSetElectrodeName(int electrode, string name);

  /*! Set reference flag of specified electrode
    \param electrode electrode index
    \param reference
  */
  void channelsSetElectrodeReference(int electrode, bool reference);

  /*! Set enabled flag of specified electrode.
    \param electrode electrode index
    \param enabled
  */
  void channelsSetElectrodeEnabled(int electrode, bool enabled);

  /*! Sets channels configuration based on full string of configuration file.
    \param config full configuration string
  */
  void channelsSetConfigurationFromString(string config);

  /*! Set reference flag in process.
    \param enabled
  */
  void processSetReferenceEnabled(bool enabled);

  /*! Set remove disabled flag in process.
    \param enabled
  */
  void processSetRemoveEnabled(bool enabled);

  /*! Set filter flag in process.
    \param enabled
  */
  void processSetFilterEnabled(bool enabled);

  //----------------------------------------
  //PROCESS DATA

  /*! Process all the data in a training data structure.
    \param data 
    
    \return processed training data
  */
  EEGTrainingData processData(EEGTrainingData &data);

  /*! Processes data based on enabled flags.
    \param data reference to eeg data

    \return reference to processed data
  */
  EEGData& processData(EEGData &data);

  /*! Processes data based on parameters.
    \param data reference to eeg data to process
    \param remove_disabled remove disabled channels from data
    \param reference reference channels against ones with reference flag
    \param filter filter the data

    \return reference to processed data
  */
  EEGData& processData(EEGData &data, bool remove_disabled, bool reference, bool filter);

  //======================================================================
  //DEVICE

  //----------------------------------------
  //GETTING OPERATIONS

  /*! Get the location of the mindset device file.

    \return filename
  */
  string deviceGetLocation();

  /*! Check if mindset is ready to record.

    \return
  */
  bool deviceIsReady();

  /*! Check if selected device file exists.

    \return
  */
  bool deviceExists();

  /*! Get string of errors from device configuration.

    \return errors
  */
  string deviceGetError();

  /*! Get inquiry string from mindset device.

    \return inquiry
  */
  string deviceGetInquiry();

  /*! Get sample rate currently selected by the device.

    \return sample rate (Hz)
  */
  int deviceGetSampleRate();

  /*! Get block size currently selected by the device.

    \return block size
  */
  int deviceGetBlockSize();

  //----------------------------------------
  //SETTING OPERATIONS

  /*! Set the device file location
    \param filename
  */
  void deviceSetDeviceLocation(string filename);

  /*! Set the sample rate the device is recording at.
    \param sample_rate
  */
  void deviceSetSampleRate(int sample_rate);

  /*! Set the block size used to collect samples in the device.
    \param block_size
  */
  void deviceSetBlockSize(int block_size);

  /*! Search for a valid mindset device file and try to connect to it.
   */
  void deviceSearch();

  //======================================================================
  //File Data Stream

  //----------------------------------------
  //GETTING OPERATIONS

  /*! Get file currently loaded by file stream.
    This is where data will be streamed from.

    \return filename
  */
  string fileStreamGetFilename();

  /*! Check if the file stream is ready to return data.

    \return
  */
  bool fileStreamIsReady();

  /*! Get sample rate that samples will be read from file data.

    \return sample rate (Hz)
  */
  int fileStreamGetSampleRate();

  /*! Get number of samples available from the file.

    \return num samples
  */
  int fileStreamGetNumSamples();

  /*! Get number of channels in the data from selected file.

    \return  channels
  */
  int fileStreamGetNumChannels();

  /*! Get number of classes the file data is organized into.

    \return classes
  */
  int fileStreamGetNumClasses();

  /*! Get number of sequences the file data is organized into.

     \return sequences
   */
  int fileStreamGetNumSequences();

  /*! Get training data object being used by file stream.

    \return
  */
  EEGTrainingData & fileStreamGetTrainingData();

  /*! Get flattened eeg data object of all the samples in the selected file

    \return
  */
  EEGData & fileStreamGetData();


  //----------------------------------------
  //SETTING OPERATIONS

  /*! Open a file for streaming.
    \param filename
  */
  void fileStreamOpenFile(string filename);

  /*! Set sample rate to read data at.
    \param sample_rate
  */
  void fileStreamSetSampleRate(int sample_rate);


  //======================================================================
  //FILTER

  //----------------------------------------
  //GETTING OPERATIONS

  /*! Get names of filter libraries available.

    \return vector of names
  */
  std::vector<string> filterGetNameList();

  /*! Get paths of filter libraries available.

    \return vector of paths
  */
  std::vector<string> filterGetPathList();

  /*! Check if specified filter is trained.
    If the filter does not need to be trained, this will return true as well.

    \param filter name of filter or "" for selected filter

    \return
  */
  bool filterIsTrained(string filter = "");

  /*! Check how many lags are being used to filter data.

    \return num lags
  */
  int filterGetNumLags();

  /*! Get name of selected filter.

    \return name
  */
  string filterGetSelected();

  /*! Get list of components selected to be filtered out.

    \return vector of components
  */
  std::vector<int> filterGetSelectedComponents();

  /*! Get a string of all components being removed.

    \return string of components
  */
  string filterGetSelectedComponentsString();

  /*! Check if selected set of components are within range.

    \return
  */
  bool filterGetSelectedComponentsValid();


  /*! Extracts components from collected data set
    \param training_data data to extract components from

    \return components
  */
  EEGData filterGetComponents(EEGData training_data);

  /*! Apply filter to data and return filtered data.
    \param data

    \return
  */
  EEGData filterApply(EEGData data);

  /*! Return the number of channels trained filter expects to recieve.

    \return num channels
  */
  int filterGetNumExpectedChannels();

  /*! Get the internal filter matrix.

    \return filter matrix
  */
  ublas::matrix<double> filterGetFilterMatrix();

  //----------------------------------------
  //SETTING OPERATIONS

  /*! Set selected components to filter out by string.
    String can contain numbers separated by commas, or ranges
    indicated with colons. E.g. "3,5,7:13,18".
    \param components
  */
  void filterSetSelectedComponentsString(string components);

  /*! Set selected filter by name.
    \param filter name of filter or "" for selected filter
  */
  void filterSetSelected(string filter = "");

  /*! Train specified filter removing selected components.
    \param filter name of filter or "" for selected filter
  */
  void filterTrain(EEGData training_data, string filter = "");

  /*! Set number of lags filter uses.
    \param n
  */
  void filterSetNumLags(int n);


  //======================================================================
  //FEATURES

  //----------------------------------------
  //GETTING OPERATIONS

  /*! Gets the list of loaded feature libraries.

    \return vector of feature names
  */
  std::vector<string> featuresGetNameList();

  /*! Gets the full path of all loaded feature libraries.

    \return vector of feature paths
  */
  std::vector<string> featuresGetPathList();

  /*! Check if specified feature has been trained.
    If feature does not require training, this will return true.
    \param feature feature name or "" to use selected feature

    \return
  */
  bool featureIsTrained(string feature = "");

  /*! Get the name of the selected feature.

    \return name of selected feature
  */
  string featuresGetSelected();

  /*! Get list of parameters for specified feature.
    \param feature feature name or "" to use selected feature

    \return list of parameters with values
  */
  std::map<std::string, CEBL::Param> featureGetParams(string feature = "");

  //----------------------------------------
  //SETTING OPERATIONS

  /*! Set selected feature by name.
    \param feature
  */
  void featuresSetSelected(string feature = "");

  /*! Train specified, or selected, feature.
    \param feature feature name or "" to use selected feature
  */
  void featureTrain(string feature = "");

  /*! Reset specified, or selected, feature.
    \param feature feature name or "" to use selected feature
  */
  void featureReset(string feature = "");

  /*! Set params for selected or specified feature.
    \param params list of set params
    \param feature feature name or "" to use selected feature
  */
  void featureSetParams(std::map<std::string, CEBL::Param> params, string feature = "");

  //USE FEATURE



  /*! Extract features from data and return the result.
    You must have already trained selected feature if necessary.
    Note that the returned data will most likely be of a
    different dimensionality from the input data.
    \param data

    \return features
  */
  EEGData featuresExtract(EEGData &data);

  /*! Extract features from training data structure.
    \param data 
    
    \return 
  */
  EEGTrainingData featuresExtract(EEGTrainingData &data);

  /*! Halt feature extraction. This is expected to be called from
    a separate thread than the one extracting the features. Note
    that this may not halt the process immediatly. All features have
    a flag to halt and specified inturruption points. Calling this
    function will cause selected feature to halt when it reaches the
    next inturruption point.
   */
  void featuresHalt();

  //======================================================================
  //DECISION

  //----------------------------------------
  //GETTING OPERATIONS

  /*! Get list of names for loaded decision plugins.

    \return vector of names
  */
  std::vector<string> decisionGetNameList();

  /*! Get list of paths for loaded decision plugins.

    \return vector of paths
  */
  std::vector<string> decisionGetPathList();


  /*! Get name of selected decision plugin.

    \return name
  */
  string decisionGetSelected();

  /*! Get parameters for specified or selected decision.
    \param decision name of decision or "" for selected decison.

    \return
  */
  std::map<std::string, CEBL::Param> decisionGetParams(string decision = "");

  //----------------------------------------
  //SETTING OPERATIONS

  /*! Set decision by name which will be used.
    \param decision name of decision
  */
  void decisionSetSelected(string decision = "");

  /*! Set parameters of specified or selected decision plugin.
    \param params parameters for specified decision
    \param decision name of decision or "" for selected decison.
  */
  void decisionSetParams(std::map<std::string, CEBL::Param> params, string decision="");

  //----------------------------------------
  //DECISION OPERATIONS

  /*! Update decision with probabilities for several samples.
    This function is identical to calling decisionUpdateWithProbabilities
    for each sample's probabilities.
    \param probs should be NSAMPLES x NCLASSES
  */
  void decisionUpdateWithProbabilities(std::vector<std::vector<double> >probs);

  /*! Update decision with probabilities for a single sample.
    \param probs probabilities for each class should be of size NCLASSES
  */
  void decisionUpdateWithProbabilities(std::vector<double> probs);


  /*! Update decision from classification for multiple samples.
    Same as calling decisionUpdateWithClassification(int) for each sample.
    \param classes should be of length nSamples
  */
  void decisionUpdateWithClassification(ublas::vector<int> classes);


  /*! Update decision with classification for a single sample.
    \param cls class the sampled has been classified as
  */
  void decisionUpdateWithClassification(int cls);


  /*! Initialize selected decision library.
    \param num_classes
  */
  void decisionInit(int num_classes);

  /*! Get proportions for each class. The result of this calls
    should be used to set the proportions for each class in
    the visual interface.

    \return
  */
  std::vector<double> decisionDecideClasses();

  //======================================================================
  //CLASSIFIER

  //----------------------------------------
  //GETTING OPERATIONS
  std::vector<string> classifiersGetNameList();
  std::vector<string> classifiersGetPathList();
  bool classifierIsTrained(string classifier = "");
  string classifiersGetSelected();
  std::map<std::string, CEBL::Param> classifierGetParams(string feature = "");
  bool classifierGetUseProbs();
  int classifierGetTrainedClasses();
  int classifierGetTrainedLags();

  //----------------------------------------
  //SETTING OPERATIONS
  void classifierReset(CEBL::Param params, string classifier = "");
  void classifiersSetSelected(string classifier = "");
  void classifierTrain(EEGTrainingData &training_data, string classifier = "");
  void classifierHaltTrain();
  void classifierSetParams(std::map<std::string, CEBL::Param> params, string classifier = "");
  void classifierSetUseProbs(bool flag);

  //USE CLASSIFIER
  //! apply classifier to data and get a vector of classes
  ublas::vector<int> classifierUse(EEGData &data);
  std::vector<std::vector<double> > classifierGetLastProbs();


  //======================================================================
  // Training

  //----------------------------------------
  //GETTING OPERATIONS

  /*! Get the labels of the classes.

    \return
  */
  std::vector<string> trainingGetClassLabels();

  /*! Get the label of a specific class .
    \param class_num

    \return
  */
  string trainingGetClassLabel(int class_num);

  /*! Get number of classes to train on.

    \return
  */
  int trainingGetNumClasses();

  /*! Get the number of sequences to train on.

    \return
  */
  int trainingGetNumSequences();

  /*! Get length of each training sequence in seconds.

    \return
  */
  int trainingGetSequenceLength();

  /*! Get length of pause between sequences.

    \return
  */
  int trainingGetPauseLength();

  /*! Get the currently loaded data.

    \return
  */
  EEGTrainingData trainingGetData();

  /*! Is data ready, either from a file or from training session.

    \return
  */
  bool trainingDataIsLoaded();

  /*! Is a trainig data file loaded.

    \return
  */
  bool trainingIsDataFileLoaded();

  /*! Get filename of loaded training datafile.

    \return
  */
  string trainingGetDataFilename();

  /*! Is training currently occuring.

    \return
  */
  bool trainingIsActive();

  /*! Did training fail.

    \return
  */
  bool trainingFailed();

  /*! Get failure message.

    \return
  */
  string trainingGetFailureMessage();

  /*! Is training paused between sequences.

    \return
  */
  bool trainingIsPaused();

  /*! Get the current class being trained.

    \return
  */
  int trainingGetTrainingClass();

  /*! Get the current sequence being trained.

    \return
  */
  int trainingGetTrainingSequence();


  /*! Check if classification feedback is enabled.
    
    \return 
  */
  bool trainingFeedbackEnabled();



  /*! Check if training process is currently training a classifier.
    
    \return 
  */
  bool trainingIsTrainingClassifier();


  /*! Get proportions for each class.

    \return
  */
  std::vector<double> trainingGetClassProportions();

  //----------------------------------------
  //SETTING OPERATIONS
  /*! Start the training process.
   */
  void trainingStart();

  /*! Stop the training process.
  */
  void trainingStop();

  /*! Set number of classes to train on.
    \param n
  */
  void trainingSetNumClasses(int n);

  /*! Set the number of sequences to train on.
    \param n
  */
  void trainingSetNumSequences(int n);

  /*! Set the length of each training sequence in seconds.
    \param n
  */
  void trainingSetSequenceLength(int n);

  /*! Set the length of pause between each sequence.
    \param n
  */
  void trainingSetPauseLength(int n);

  /*! Set class labels.
    \param labels
  */
  void trainingSetClassLabels(std::vector<string> labels);

  /*! Set label for a specific class.
    \param class_number
    \param label
  */
  void trainingSetClassLabel(int class_number, string label);

  /*! Load a data file.
    \param filename
  */
  void trainingLoadData(string filename);

  /*! Clear loaded data.
   */
  void trainingClearData();

  /*! Save the data to a file.
    \param filename
  */
  void trainingSaveData(string filename);


  /*! Set whether or not to provide classification feedback.
    \param flag 
  */
  void trainingSetFeedbackEnabled(bool flag);

  //======================================================================
  // Real-Time Classification

  //----------------------------------------
  //GETTING OPERATIONS
  /*! Check if realtime classification system is ready to start classifying.

    \return
  */
  bool realtimeIsReady();

  /*! Check if last training processed failed.
    
    \return 
  */
  bool realtimeLastTrainFailed();

  /*! Check if realtime classification system is currently classifying.

    \return
  */
  bool realtimeIsClassifying();

  /*! Reads the queue of classified samples and clears the queue.

    \return
  */
  std::vector<int> realtimeReadClassificationQueue();

  /*! Get proportions for each class.

    \return
  */
  std::vector<double> realtimeGetClassProportions();

  /*! Reads the queue of classified samples without clearing the queue.

    \return
  */
  std::vector<int> realtimePeekClassificationQueue() const;

  /*! Get last selected class or -1 if there is none.
    
    \return 
  */
  int realtimeGetSelectedClass() const;


  //----------------------------------------
  //SETTING OPERATIONS

  /*! Clear the queue of classified samples.
   */
  void realtimeClearClassificationQueue();

  //CONTROL CLASSIFICATION
  /*! Train the selected classifier on training data.
   */
  void realtimeTrainClassifier();

  /*! Start the classification and adding to queue.
   */
  void realtimeStartClassifying();

  /*! Stop the classification.
   */
  void realtimeStopClassifying();

  /*! Train the classifier in separate thread.
   */
  void realtimeTrainClassifierThreaded();

  /*! Stop the threaded classifier training.
   */
  void realtimeTrainClassifierHalt();

  /*! Check if classifier is being trained .

    \return
  */
  bool realtimeIsTrainingClassifier();

  /*! Indicate that the selected class has been read. 
   */
  void realtimeClearSelectedClass();

  //======================================================================
  // SESSION SAVING and LOADING

  //----------------------------------------
  //GETTING OPERATIONS

  /*! Check if session needs to have a filename specified. Otherwise we
    can just save the session to the previously loaded/saved file.

    \return
  */
  bool sessionShouldSaveAs();


  //----------------------------------------
  //SETTING OPERATIONS

  /*! Update the session and save it to the selected file.
   */
  void sessionSave();

  /*! Update the session and ave it to specified file.
    \param filename
  */
  void sessionSaveAs(string filename);

  /*! Load session from specifiedfile.
    \param filename
  */
  void sessionLoad(string filename);



  //======================================================================
  // DATA FUNCTIONS
  /*! Loads training data from a file and returns it.

    \param filename 
    
    \return 
  */
  EEGTrainingData dataLoadTrainingDataFile(string filename);


};
#endif

