/*
* CEBL : CSU EEG Brain-Computer Interface Lab
*
* Author: Jeshua Bratman - jeshuabratman@gmail.com
*
* This file is part of CEBL.
*
* CEBL is free software; you can redistribute it and/or modify it.
* We only ask that if you use our code that you cite the source in
* your project or publication.
*
* EEG Group (www.cs.colostate.edu/eeg)
* Department of Computer Science
* Colorado State University
* 
*/


//--------------------------------------------------
//   cppR_construction.hpp
// Matrix and Vector construction and manipulation functions

#ifndef CPPR_CONSTRUCTION_H //make sure this only gets included once
#define CPPR_CONSTRUCTION_H


#include <cppR/cppR_includes.hpp>


namespace cppR
{
  //==============================

  /*! Creates a vector by flattening a specified matrix.
    \param m NxM matrix to convert to vector
    
    \return N * M length vector
  */
  template < typename T >
  ublas::vector<T> createVector(const ublas::matrix<T> &m)
  {
    ublas::vector<T> ret(m.size1()* m.size2());
    int count = 0;
    for(unsigned int i=0; i<m.size1(); i++)
      {
        for(unsigned int j=0; j<m.size2(); j++)
          ret(count++) = m(i,j);
      }
    return ret;
  }


  /*! Create a vector of specified size by repeating value.
    Same as #rep.
    \param value value to repeat
    \param size size of resulting vecto
    
    \return 
  */
  template < typename T >
  ublas::vector<T> createVector(const T& value, int size)
  {
    ublas::vector<T> ret(size);
    for(int i=0; i<size; i++)
      {
        ret(i) = value;
      }
    return ret;
  }

  //==============================
  //createMatrix

  //! \anchor createMatrix1
  /*! Create a matrix from a vector. Reshapes vector based on specified
    number of rows and columns. Creates matrix by column from vector unless
    byrow is set. If rows * cols is greater than the size of vec, the entries
    in vec will be reused.

    \param vec vector to read from.
    \param rows number of rows
    \param cols number of columns
    \param byrow flag to create matrix by row instead of column
    
    \return 


  */
  template < typename T >
  ublas::matrix<T>
  createMatrix(const ublas::vector<T> &vec,
               int rows, int cols, bool byrow=false)
  {
    int size = vec.size();
    if(rows <= 0 && cols > 0)
      rows = size/cols;
    else if(cols <= 0 && rows > 0)
      cols = size / rows;
    else if(rows <= 0 && cols <=0)
      {
        rows = 1;
        cols = size;
      }

    ublas::matrix<T> new_matrix(rows,cols);
    for(int i=0; i<rows; i++)
      {
        for(int j=0; j<cols; j++)
          {
            if(byrow)
              {
                new_matrix(i,j) = vec[(i*rows + j) % size];
              }
            else
              {
                new_matrix(i,j) = vec[(j*rows + i) % size];
              }
          }
      }
    return new_matrix;
  }

  /*! Std version of \ref createMatrix1 
    \param vec
    \param rows 
    \param cols 
    \param byrow 
    
    \return 
  */
  template < typename T >
  ublas::matrix<T>
  createMatrix(const std::vector<T> &vec, int rows, int cols, bool byrow=false)
  {
    ublas::vector<T> v;
    v.resize(vec.size());
    std::copy(vec.begin(), vec.end(), v.begin());
    return createMatrix(v,rows,cols,byrow);
  }


  /*! Create a matrix using a single value for every entry.
    \param value value to place in each entry
    \param rows number of rows to make
    \param cols number of cols to make
    
    \return 
  */
  template < typename T >
  ublas::matrix<T>
  createMatrix(T value, int rows, int cols)
  {
    int size = 1;
    if(rows <= 0 && cols > 0)
      rows = size/cols;
    else if(cols <= 0 && rows > 0)
      cols = size / rows;
    else if(rows <= 0 && cols <=0)
      {
        rows = 1;
        cols = size;
      }
    ublas::matrix<T> new_matrix(rows,cols);
    for(int i=0; i<rows; i++)
      {
        for(int j=0; j<cols; j++)
          {
            new_matrix(i,j) = value;
          }
      }
    return new_matrix;
  }

  //==============================
  /*! Binds two vectors together.
    \param v1 size N vector
    \param v2 size M vector
    
    \return vector of size N+M
  */
  template < typename T >
  ublas::vector<T> bind(ublas::vector<T> v1, ublas::vector<T> v2)
  {
    ublas::vector<T> ret = v1;
    ret.resize(v1.size()+v2.size());
    for(unsigned int i = v1.size(); i < v2.size(); i++)
      {
        ret[i] = v2[i-v1.size()];
      }

    return ret;
  }

  //==============================
  /*! Repeat specified vector n times into a new vector.
    \param vec vector to repeat
    \param n number of times to repeatn
    
    \return vector of size vec.size()*n
  */
  template < typename T >
  ublas::vector<T>
  rep(const ublas::vector<T> &vec, int n)
  {
    unsigned int size = vec.size();
    ublas::vector<T> new_vector(size * n);
    for( int i=0; i<n; i++)
      {
        for(unsigned int j=0; j < size; j++)
          {
            new_vector[i*size+j] = vec[j];
          }
      }
    return new_vector;
  }


  /*! Repeat a single value n times into a new vector.
    \param value 
    \param n 
    
    \return vector of repeated value
  */
  template < typename T >
  ublas::vector<T>
  rep(const T &value, int n)
  {
    ublas::vector<T> new_vector(n);
    for( int i=0; i<n; i++)
      {
        new_vector[i] = value;
      }
    return new_vector;
  }

  //==============================
  //diag

  /*! Get the diagonal of a matrix and return it as a vector.
    \param m matrix to get diagonal from
    
    \return vector of diagonal entries
  */
  template < typename T >
  ublas::vector<T> diag(const ublas::matrix<T> &m)
  {
    ublas::vector<T> ret(m.size1());
    for(unsigned int i=0; i<m.size1(); i++)
      {
        if(i<m.size2())
          ret[i] = m(i,i);
        else
          break;
      }
    return ret;
  }

  /*! Create a matrix with diagonal entries taken from specified vector.
    \param v vector to use as diagonal
    
    \return diagonal matrix
  */
  template < typename T >
  ublas::matrix<T> diag(const ublas::vector<T> &v)
  {
    ublas::matrix<T> ret = createMatrix(
                                        rep(static_cast<T>(0),
                                            v.size() * v.size()),
                                        v.size(),
                                        v.size()
                                        );
    for(unsigned int i=0; i<v.size(); i++)
      {
        ret(i, i) = v[i];
      }
    return ret;
  }

  /*! Creates a square matrix of size 'size' with diagonal entries
    equal to 'value'.
    \param value value to use as diagonal entry
    \param size size of matrix
    
    \return diagonal matrix with value on the diagonal
  */
  template < typename T >
  ublas::matrix<T> diag(const T &value, int size)
  {
    ublas::matrix<T> ret = createMatrix(
                                        rep(static_cast<T>(0), size * size),
                                        size,
                                        size
                                        );
    for(int i=0; i<size; i++)
      {
        ret(i, i) = value;
      }
    return ret;
  }



  //==============================
  /*! Create a vector with values min, min+1, ..., max.
    \param min starting value inclusive
    \param max ending value inclusive
    
    \return 
  */
  template < typename T >
  ublas::vector<T>
  vectorRange(T min, T max)
  {
    cppR_assert(max > min, "Max must be greater than min in vecRange.");

    ublas::vector<int> ret;
    int size = max - min + 1;
    ret.resize(size);
    int val = min;
    for(int i=0;i<size;i++)
      {
        ret[i] = val;
        val++;
      }
    return ret;
  }


}//end of namepsace


//--------------------------------------------------



#endif
