/*
* CEBL : CSU EEG Brain-Computer Interface Lab
*
* Author: Jeshua Bratman - jeshuabratman@gmail.com
*
* This file is part of CEBL.
*
* CEBL is free software; you can redistribute it and/or modify it.
* We only ask that if you use our code that you cite the source in
* your project or publication.
*
* EEG Group (www.cs.colostate.edu/eeg)
* Department of Computer Science
* Colorado State University
* 
*/

//--------------------------------------------------
//   cppR_manipulation.hpp
// Matrix and Vector manipulation and slicing

#ifndef CPPR_MANIPULATION_H //make sure this only gets included once
#define CPPR_MANIPULATION_H


#include <cppR/cppR_includes.hpp>


//--------------------------------------------------


namespace cppR
{
  //==============================
  /*! Select a submatrix from a matrix. The matrix slice returned by this
    function can be assigned which will modify the selected submatrix of 
    the original matrix. Note: use the ublas functions row(M,n) and 
    col(M,n) if you only want single columns or rows. Submatrix will not
    work for a single row or column.

    You can think of this function as the R brackets: 
    m[r1:r2,c1:c2] would be submatrix(m, r1, r2, c1, c2)

    \param m matrix to select from
    \param r1 starting row inclusive
    \param r2 ending row inclusive, 0 means all rows
    \param c1 starting column inclusive
    \param c2 ending column inclusive, 0 means all columns
    
    \return matrix slice representing selected submatrix
  */
  template < typename T >
  ublas::matrix_slice<ublas::matrix<T> >
  submatrix(ublas::matrix<T> &m, int r1,int r2,int c1,int c2)
  {
    if(r2 == 0)
      r2 = m.size1()-1;
    if(c2 == 0)
      c2 = m.size2()-1;

    //make upper bound inclusive
    r2++;
    c2++;
    cppR_assert(r1 <= r2, 
                "in submatrix: row start must be less than or equal to row end"
                );
    cppR_assert(c1 <= c2, 
                "in submatrix: col start must be less than or equal to col end"
                );
    cppR_assert(r1 >= 0, 
                "in submatrix: row start must be greater than or equal to 0");
    cppR_assert(c1 >= 0, 
                "in submatrix: col start must be greater than or equal to 0"
                );
    cppR_assert(unsigned(r2-r1) <= (m.size1()),
                "in submatrix: rows range larger than rows of matrix"
                );
    cppR_assert(unsigned(c2-c1) <= (m.size2()),
                "in submatrix: col range larger than columns of matrix"
                );
    return subslice(m,r1,1,r2-r1,c1,1,c2-c1);
  }

  //==============================
  /*! Identical to #submatrix except the return value cannot be assigned.
    Use this to make a new matrix from a submatrix if the original is
    const.

    \param m 
    \param r1 
    \param r2 
    \param c1 
    \param c2 
    
    \return 
  */
  template < typename T >
  const ublas::matrix_slice<ublas::matrix<T> >
  submatrix(const ublas::matrix<T> &m, int r1,int r2,int c1,int c2)
  {
    if(r2 == 0)
      r2 = m.size1()-1;
    if(c2 == 0)
      c2 = m.size2()-1;

    //make upper bound inclusive
    r2++;
    c2++;

    cppR_assert(r1 <= r2,
                "in submatrix: row start must be less than or equal to row end"
                );
    cppR_assert(c1 <= c2,
                "in submatrix: col start must be less than or equal to col end"
                );
    cppR_assert(r1 >= 0,
                "in submatrix: row start must be greater than or equal to 0"
                );
    cppR_assert(c1 >= 0,
                "in submatrix: col start must be greater than or equal to 0"
                );
    cppR_assert(r2-r1 <= m.size1(),
                "in submatrix: rows range larger than rows of matrix"
                );
    cppR_assert(c2-c1 <= m.size2(),
                "in submatrix: col range larger than columns of matrix"
                );
    ublas::matrix<T> m1 = m;

    return subslice(m1,r1,1,r2-r1,c1,1,c2-c1);
  }



  //==============================
  //submatrixAssign
  /*! Assign submatrix of one matrix to submatrix of another.

    Deprecated: Use #submatrix instead.

    \param m1 matrix to assign to
    \param m1start1 m1 start row
    \param m1end1 m1 end row
    \param m1start2 m1 start column
    \param m1end2 m1 end column
    \param m2 matrix to assign from
    \param m2start1 m2 start row
    \param m2end1 m2 end row
    \param m2start2 m2 start column
    \param m2end2 m2 end column
  */
  template < typename T >
  void submatrixAssign(ublas::matrix<T> &m1, 
                       int m1start1, int m1end1, int m1start2, int m1end2,
                       const ublas::matrix<T> &m2,
                       int m2start1, int m2end1, int m2start2, int m2end2)
  {
    //make sure dimensions are right
    if(m1end1-m1start1!=m2end1-m2start1)
      {
        std::cerr << "ERROR: number of rows to copy differs\n";
        return;
      }
    if(m1end2-m1start2!=m2end2-m2start2)
      {
        std::cerr << "ERROR: number of cols to copy differs\n";
        return;
      }
    ublas::matrix_slice<ublas::matrix<T> >
      slice1(m1,
             ublas::slice(m1start1,1,m1end1-m1start1),
             ublas::slice(m1start2,1,m1end2-m1start2)
             );

    //TODO: remove this pointless copy...
    ublas::matrix<T> m3 = m2;

    ublas::matrix_slice<ublas::matrix<T> >
      slice2(m3,
             ublas::slice(m2start1,1,m2end1-m2start1),
             ublas::slice(m2start2,1,m2end2-m2start2)
             );
    slice1 = slice2;
  }


  //==============================
  /*! Creates a vector of booleans which are true when value
    equals that index in the matrix.
    \param value value to check
    \param v vector
    
    \return boolean mask of length equal to that of v
  */
  template < typename T >
  std::vector<bool> createMask(T value, ublas::vector<T> v)
  {
    std::vector<bool> ret(v.size());
    for(unsigned int i=0; i<v.size(); i++)
      ret[i] = v[i]==value;
    return ret;
  }


  //==============================
  /*! Select rows of matrix based on a vector mask.
    This will return a new matrix which is constructed from the
    original matrix using only the rows specified in the mask.
    \param m matrix
    \param mask should have length equal to number of rows in m
    
    \return new matrix
  */
  template < typename T >
  ublas::matrix<T> rowMask(const ublas::matrix<T> &m,
                           std::vector<bool> mask)
  {
    ublas::matrix<T> ret(count(mask.begin(), mask.end(), true),m.size2());
    int count = 0;
    for(unsigned int i=0; i<mask.size(); i++)
      {
        if(mask[i])
          row(ret,count++) = row(m, i);
      }
    return ret;
  }

  //==============================
  /*! Attach two matrices together horizontally and return result.
    
    \param m1 size NxM matrix 
    \param m2 size NxQ matrix
    
    \return size Nx(M+Q) matrix
  */
  template < typename T >
  ublas::matrix<T> cbind(const ublas::matrix<T> &m1,
                         const ublas::matrix<T> &m2)
  {

    ublas::matrix<T> ret = m1;

    //assert that the matrices have the same number of rows
    cppR_assert(m1.size1() == m2.size1(),
                "in cbind: rows of matrices must match");

    int original_size2 = m1.size2();
    ret.resize(m1.size1(), m1.size2() + m2.size2());

    submatrix(ret,
              0, m1.size1()-1,
              original_size2, original_size2 + m2.size2()-1) = m2;

    return ret;
  }


  //==============================
  /*! Attach two matrices together vertically and return result.
    
    \param m1 size MxN matrix 
    \param m2 size QxN matrix
    
    \return size (M+Q)xN matrix
  */
  template < typename T >
  ublas::matrix<T> rbind(const ublas::matrix<T> &m1,
                         const ublas::matrix<T> &m2)
  {

    ublas::matrix<T> ret = m1;

    //assert that the matrices have the same number of columns
    cppR_assert(m1.size2() == m2.size2(),
                "in rbind: columns of matrices must match");


    int original_size1 = m1.size1();
    ret.resize(m1.size1()+m2.size1(), m1.size2());

    submatrix(ret,
              original_size1, original_size1 + m2.size1()-1,
              0, m1.size2()-1) = m2;

    return ret;
  }

  //==============================
  /*! Reverse elements in matrix. 
    If matrix has more than 1 column, all columns will be put into a single
    column matrix
    \param m1 NxM matrix to reverse
    
    \return (N*m)x1 matrix
  */
  template < typename T >
  ublas::matrix<T> rev(const ublas::matrix<T> & m1)
  {
    ublas::matrix<T> ret;
    int size = m1.size1() * m1.size2();
    ret.resize(size,1);

    int count = 0;
    for(unsigned int i=0;i<m1.size1();i++)
    {
      for(unsigned int j=0;j<m1.size2();j++)
        {
          ret(count++,0) = m1(i,j);
        }
    }
    return ret;
  }

  //==============================
  /*! Reverse a vector
    \param v 
    
    \return vector reversed
  */
  template < typename T >
  ublas::vector<T> rev(const ublas::vector<T> & v)
  {
    ublas::vector<T> ret;
    int count = 0;
    for(unsigned int i=0;i<v.size();i++)
    {
      ret[count++] = v[i];
    
    }
    return ret;
  }


}//end of namespace


#endif
