/*
* CEBL : CSU EEG Brain-Computer Interface Lab
*
* Author: Jeshua Bratman - jeshuabratman@gmail.com
*
* This file is part of CEBL.
*
* CEBL is free software; you can redistribute it and/or modify it.
* We only ask that if you use our code that you cite the source in
* your project or publication.
*
* EEG Group (www.cs.colostate.edu/eeg)
* Department of Computer Science
* Colorado State University
* 
*/


//--------------------------------------------------
//   cppR_math.hpp
// Matrix and Vector utilities

#ifndef CPPR_MATH_H //make sure this only gets included once
#define CPPR_MATH_H

#include <cppR/cppR_includes.hpp>
#include <cppR/cppR_construction.hpp>

using std::complex;

//--------------------------------------------------
namespace cppR
{
  //--------------------------------------------------
  //cppR math DATA STRUCTURES

  /*! Struct returned by eigen function. 

   */
  template < typename T >
  struct EigStruct
  {
    ublas::matrix<T> vectors; //! eigenvectors
    ublas::vector<T> values;  //! eigenvalues
  };

  /*! Struct returned by svd function.
   */
  template < typename T >
  struct SvdStruct
  {
    ublas::vector<T> d; //! singular values of matrix
    ublas::matrix<T> u; //! left singular vectors
    ublas::matrix<T> v; //! right singular vectors
  };

  //==============================
  /*! Compute component-wise product of two matrices.
    Matrices must be of the same size.
    \param m NxM matrix
    \param n NxM matrix
    
    \return 
  */
  template < typename T >
  ublas::matrix<T> compProd(const ublas::matrix<T> & m,
                            const ublas::matrix<T> & n)
  {
    cppR_assert(m.size1() == n.size1() && m.size2() == n.size2(),
                "in compProd: matrices must be of the same size.");

    ublas::matrix<T> ret(m.size1(),m.size2());

    for(unsigned int row=0; row<m.size1(); row++)
      for(unsigned int col=0; col<m.size2(); col++)
        ret(row,col) = m(row,col) * n(row,col);

    return ret;
  }

  //==============================
  //compDiv

  //==============================
  /*! Compute component-wise quotient of two matrices.
    Matrices must be of the same size.
    \param m NxM matrix
    \param n NxM matrix
    
    \return 
  */
  template < typename T >
  ublas::matrix<T>
  compDiv(const ublas::matrix<T> & m, const ublas::matrix<T> & n)
  {
    assert(m.size1() == n.size1() && m.size2() == n.size2());

    ublas::matrix<T> ret(m.size1(),m.size2());

    for(unsigned row=0; row<m.size1(); row++)
      for(unsigned col=0; col<m.size2(); col++)
        ret(row,col) = m(row,col) / n(row,col);

    return ret;
  }


  //==============================
  /*! Computues solution to set of real linear equations in matrix form.
    Uses a diagonal matrix of 1's as right-hand side so the solution will
    will be the inverse of the specified matrix.
    cppR::solve uses lapack routine gesv to perform the inverse.
    \param m1 matrix to solve

    
    \return solved matrix
  */
  template < typename T >
  ublas::matrix<T> solve(const ublas::matrix<T> & m1)
  {
    int size = m1.size1();
    ublas::matrix<T, ublas::column_major> M(size,size);
    ublas::matrix<T, ublas::column_major> temp(size,size);
    M = m1;
    temp = diag(1,size);
    lapack::gesv(M,temp);

    //return a row_major matrix
    ublas::matrix<T> ret = temp;
    return ret;

  };

  /* Computes solution to set of real linear equations in matrix form.
     Uses m2 as right-hand side of equations.

    cppR::solve uses lapack routine gesv to perform the inverse.
    \param m1 
    \param m2 
    
    \return 
  */
  template < typename T >
  ublas::matrix<T> solve(const ublas::matrix<T> & m1,
                         const ublas::matrix<T> & m2)
  {
    int size = m1.size1();
    ublas::matrix<T, ublas::column_major> M(size,size);
    ublas::matrix<T, ublas::column_major> temp(size,size);
    M = m1;
    temp = m2;
    lapack::gesv(M,temp);

    //return a row_major matrix
    ublas::matrix<T> ret = temp;
    return ret;

  };

  //==============================
  /*! Computes eigen values and eigen vectors from specified matrix.
    cppR::eigen uses the lapack syev routine.
    \param m1 real symmetric matrix

    \return eigen struct containing vectors and values
  */
  template < typename T >
  EigStruct<T> eigen(const ublas::matrix<T> & m1)
  {
    int size = m1.size1();
    ublas::matrix<T, ublas::column_major> M(size,size);
    ublas::matrix<double, ublas::column_major> B(size,size);
    ublas::vector<double> D(size);
    M = m1;
    lapack::syev('V','U',M,D,lapack::optimal_workspace());

    EigStruct<T> ret;
    ret.values = D;
    ret.vectors = M;
    return ret;

  };

  //==============================
  /*! Performs the singular value decomposition on specified matrix
    Returns a structure containing the singular values, and matrices
    of both left and right singular vectors.
    cppR::svd uses the lapack gesdd routine.
    \param m1 general rectangular matrix
    
    \return 
  */
  template < typename T >
  SvdStruct<T> svd(const ublas::matrix<T> & m1)
  {
    SvdStruct<T> ret;

    ublas::matrix<double, ublas::column_major> A(m1.size1(),m1.size2());
    ublas::matrix<double, ublas::column_major> U(m1.size1(),m1.size2());
    ublas::matrix<double, ublas::column_major> V(m1.size1(),m1.size2());
    ublas::vector<double> D(m1.size1());

    A = m1;

    lapack::gesdd(A, D, U, V);

    ret.u = U;
    ret.v = V;
    ret.d = D;
    return ret;
  };

  //==============================
  /*! Returns rank of matrix by counting the number of singular values
    greater than 1e-9 (due to machine precision).
    \param m 
    
    \return rank of matrix
  */
  template < typename T >
  int rank(const ublas::matrix<T> & m)
  {
    SvdStruct<double> svd_value = svd(m);
    int rank = 0;
    for(int i=0; i<svd_value.d.size(); i++)
      {
        //if singular value is not 'small' (1e-9 is small enough)
        if(svd_value.d[i] > 1e-9)
          rank++;
      }
    return rank;
  }

  //==============================
  /*! Compute determinant of matrix. Uses lu factorization method.
    \param m 
    
    \return determinant of matrix
  */
  template < typename T >
  double det(const boost::numeric::ublas::matrix_expression<T>& m) {
    assert(m().size1() == m().size2());

    boost::numeric::ublas::permutation_matrix<std::size_t> pivots(m().size1());

    // create copy
    boost::numeric::ublas::matrix<typename T::value_type> mLu(m);

    lu_factorize(mLu, pivots);

    double detval = 1.0;

    for (std::size_t i=0; i < pivots.size(); ++i) {
      if (pivots(i) != i)
        detval *= -1.0;
      detval *= mLu(i,i);
    }
    return detval;
  }

  //==============================
  //Lag
  /*! Perform time embedded lagging on matrix.
    Note: matrix should be of the form: nFeatures X nSamples
    \param data_orig 
    \param n_lags 
    
    \return lagged matrix
  */
  template < typename T >
  ublas::matrix<T> Lag(const ublas::matrix<T> &data_orig, int n_lags)
  {
    ublas::matrix<T> data = data_orig;
    //check the number of lags
    if(n_lags < 1)
      return data;

    if(n_lags >= ncol(data))
      {
        std::cerr << "ERROR: Cannot apply this many lags to data.\n";
        return data;
      }

    //lag the data
    int rows = nrow(data);
    int cols = ncol(data);

    ublas::matrix<T> lagged_data = cppR::createMatrix(0, 
                                                      (n_lags+1) * rows,
                                                      cols-n_lags);
    //ublas::matrix<double> d2 = data;
    for(int i=0; i<= n_lags; i++)
      {
        //R Code: laggedData[ ((lag * rows + 1) : ((lag+1) * rows)), ]
        //                   <- data[, ((lag+1) : (cols-nLags+lag))];
        ublas::matrix_slice<ublas::matrix<double> > m1 = 
          submatrix(lagged_data, i * rows,((i+1) * rows) - 1,0,0);
        ublas::matrix_slice<ublas::matrix<double> > m2 =
          submatrix(data, 0, 0, i, (cols-n_lags+i) -1);
        m1 = m2;
      }

    return lagged_data;
  }

  //==============================
  //complex matrices

  /*! Returns real portion of a complex matrix.
    \param m 
    
    \return 
  */
  template <typename T>
  ublas::matrix<T>
  Re(ublas::matrix<std::complex<T> > m)
  {
    ublas::matrix<T> ret(m.size1(),m.size2());

    for(int i=0;i<ret.size1();i++)
      for(int j=0;j<ret.size2();j++)
        ret(i,j) = real(m(i,j));
    return ret;
  }

  /*! Returns complex conjugate of complex matrix.
    \param m 
    
    \return 
  */
  template <typename T>
  ublas::matrix<std::complex<T> >
  Conj(ublas::matrix<std::complex<T> > m)
  {
    ublas::matrix<complex<double> > ret(m.size1(),m.size2());
    for(int i=0;i<ret.size1();i++)
      for(int j=0;j<ret.size2();j++)
        ret(i,j) = conj(m(i,j));
    return ret;
  }


  //==============================
  //FAST FOURIER TRANSFORM

#ifdef FFTW3_H

  //! \anchor fft1
  /*! Computes the fourier transform of given matrix.
    Result is a complex matrix.

    Note: this function is not included unless the FFTW3
    header file has already been included. This is to 
    allow compilation of cppR without necessarily linking
    to the FFTW3 library.

    \param m 
    \param inverse do inverse fft instead
    
    \return complex matrix
  */
  ublas::matrix<std::complex<double> >
  fft(const ublas::matrix<complex<double> > &m, bool inverse=false)
  {
    ublas::matrix<complex<double> > ret(m.size1(), m.size2());

    int rows,cols;
    rows = m.size1();
    cols = m.size2();

    //create in and out matrices
    fftw_complex *in,*out;
    in = (fftw_complex*) fftw_malloc(rows * cols * sizeof(fftw_complex));
    out = (fftw_complex*) fftw_malloc(rows * cols * sizeof(fftw_complex));

    //create plan
    fftw_plan p;
    if(!inverse)
      {
        p = fftw_plan_dft_2d(rows,cols,
                             in, out,
                             FFTW_FORWARD, FFTW_ESTIMATE | FFTW_DESTROY_INPUT);
      }
    else
      {
        p = fftw_plan_dft_2d(rows,cols,
                             in, out,
                             FFTW_BACKWARD, FFTW_ESTIMATE | FFTW_DESTROY_INPUT);
      }
    //copy input data into in
    for(int i=0; i<rows; i++)
      for(int j=0; j<cols; j++)
        {
          in[j+cols*i][0] = real(m(i,j));
          in[j+cols*i][1] = imag(m(i,j));
        }

    //run the fft
    fftw_execute(p);

    //copy output data into return matrix
    for(int i=0; i<rows; i++)
      for(int j=0; j<cols; j++)
        {
          ret(i,j) = complex<double>(out[j+cols*i][0], out[j+cols*i][1]);
        }

    //destroy plan
    fftw_destroy_plan(p);

    //free arrays
    fftw_free(in);
    fftw_free(out);

    //return matrix
    return ret;
  }


  /*! Performs fourier transform of a real matrix. 
    This function simply turns the real matrix into a complex
    one and then uses \ref fft1.

    \param m see \ref fft1
    \param inverse see \ref fft1
    
    \return 
  */
  ublas::matrix<std::complex<double> >
  fft(const ublas::matrix<double> &m, bool inverse=false)
  {
    int rows,cols;
    rows = m.size1();
    cols = m.size2();

    ublas::matrix<complex<double> > ret(rows,cols);

    for(int i=0; i<rows; i++)
      for(int j=0; j<cols; j++)
        {
          ret(i,j) = complex<double>(m(i,j),0.0);
        }

    return fft(ret);
  }
#endif //endif for FFTW
  //END OF FFT

  //==============================
  //OPERATORS

  //! Scalar addition for a ublas matrix.
  template < typename T >
  ublas::matrix<T>
  operator+=(const ublas::matrix<T> & m, const T v)
  {
    ublas::matrix<T> ret = m;
    for(int i=0;i<m.size1();i++)
      for(int j=0;j<m.size2();j++)
        ret(i,j) += v;
    return ret;
  }

  //! Scalar addition for a ublas matrix.
  template < typename T >
  ublas::matrix<T>
  operator+(const ublas::matrix<T> & m, const T v)
  {
    ublas::matrix<T> ret = m;
    for(int i=0;i<m.size1();i++)
      for(int j=0;j<m.size2();j++)
        ret(i,j) += v;
    return ret;
  }

  //! Scalar subtraction for a ublas matrix.
  template < typename T >
  ublas::matrix<T>
  operator-=(const ublas::matrix<T> & m, const T v)
  {
    ublas::matrix<T> ret = m;
    for(int i=0;i<m.size1();i++)
      for(int j=0;j<m.size2();j++)
        ret(i,j) -= v;
    return ret;
  }

  //! Scalar subtraction for ublas matrix.
  template < typename T >
  ublas::matrix<T>
  operator-(const ublas::matrix<T> & m, const T v)
  {
    ublas::matrix<T> ret = m;
    for(int i=0;i<m.size1();i++)
      for(int j=0;j<m.size2();j++)
        ret(i,j) -= v;
    return ret;
  }

}//end of namespace


//--------------------------------------------------



#endif
