/*
* CEBL : CSU EEG Brain-Computer Interface Lab
*
* Author: Jeshua Bratman - jeshuabratman@gmail.com
*
* This file is part of CEBL.
*
* CEBL is free software; you can redistribute it and/or modify it.
* We only ask that if you use our code that you cite the source in
* your project or publication.
*
* EEG Group (www.cs.colostate.edu/eeg)
* Department of Computer Science
* Colorado State University
* 
*/


//--------------------------------------------------
//   cppR_stats.hpp
// Matrix and Vector utilities

#ifndef CPPR_STATS_H //make sure this only gets included once
#define CPPR_STATS_H

#include <cppR/cppR_includes.hpp>

//--------------------------------------------------
namespace cppR
{
  //==============================

  /*! Creates a vector of means for each row in a matrix.
    \param data 
    
    \return means
  */
  template < typename T >
  ublas::vector<T>
  rowMeans(ublas::matrix<T> &data)
  {
    unsigned int rows = data.size1();
    unsigned int cols = data.size2();
    ublas::vector<T> means(rows);

    for(unsigned int i=0; i<rows; i++)
      {
        ublas::matrix_row<ublas::matrix<T> > mr (data, i);
        T mean = T(ublas::sum(mr))/T(cols);
        means(i) = mean;
      }
    return means;
  }

  //==============================
  /*! Creates a vector of means for each column in a matrix.
    \param data 
    
    \return means
  */
  template < typename T >
  ublas::vector<T>
  colMeans(ublas::matrix<T> &data)
  {
    int rows = data.size1();
    int cols = data.size2();
    ublas::vector<T> means(cols);

    for(int i=0; i<cols; i++)
      {
        ublas::matrix_column<ublas::matrix<T> > mc (data, i);
        T mean = T(ublas::sum(mc))/T(rows);
        means(i) = mean;
      }
    return means;
  }

  //==============================

  /*! Creates a vector of sums for each row in a matrix.
    \param data 
    
    \return sums
  */
  template < typename T >
  ublas::vector<T>
  rowSums(ublas::matrix<T> &data)
  {
    int rows = data.size1();
    ublas::vector<T> sums(rows);

    for(int i=0; i<rows; i++)
      {
         ublas::matrix_row<ublas::matrix<T> > mc (data, i);
         T s = T(ublas::sum(mc));
         sums(i) = s;
      }
    return sums;
  }
  //==============================
  /*! Creates of vector of sums for each column in a matrix.
    \param data 
    
    \return sums
  */
  template < typename T >
  ublas::vector<T>
  colSums(ublas::matrix<T> &data)
  {
    int cols = data.size2();
    ublas::vector<T> sums(cols);

    for(int i=0; i<cols; i++)
      {
         ublas::matrix_column<ublas::matrix<T> > mc (data, i);
         T s = T(ublas::sum(mc));
         sums(i) = s;
      }
    return sums;
  }


  //==============================
  /*! Transpose matrix.
    \param m 
    
    \return 
  */
  template < typename T >
  ublas::matrix<T> t(const ublas::matrix<T> &m) { return trans(m); };

  //==============================
  /*! Number of columns in a matrix.
    \param m 
    
    \return 
  */
  template < typename T >
  int ncol(const ublas::matrix<T> &m) { return m.size2(); };

  //==============================
  /*! Number of rows in a matrix
    \param m 
    
    \return 
  */
  template < typename T >
  int nrow(const ublas::matrix<T> &m) { return m.size1(); };



  //==============================
  /*! frobeniusNorm of a matrix.

    Computed as: \f$\displaystyle\sqrt{\sum_{i=0}^n \sum_{j=0}^m M_{i,j}^2} \f$
    
    \param m matrix M of size nXm
    
    \return 
  */
  template < typename T >
  double frobeniusNorm(ublas::matrix<T> m)
  {
    return sqrt(sum(square(m)));
  }



  //==============================
  /*! Returns uniques vector of elements.
    \param v 
    
    \return unique elements
  */
  template < typename T >
  ublas::vector<T> unique(const ublas::vector<T> &v)
  {
    ublas::vector<T> ret = v;
    typename ublas::vector<T>::iterator it;

    std::sort(ret.begin(), ret.end());
    it = std::unique(ret.begin(), ret.end());
    ret.resize(it - ret.begin());

    return ret;
  }


  //==============================
  /*! Count occurences of specific value in a vector
    \param value 
    \param v 
    
    \return number of occurences
  */
  template < typename T >
  int count(const T &value, const ublas::vector<T> &v)
  {
    int c = 0;
    for(unsigned int i=0; i<v.size(); i++)
      if(v[i]==value)
        c++;
    return c;
  }


  //==============================
  /*! Square each element in matrix.
    \param m 
    
    \return 
  */
  template < typename T >
  ublas::matrix<T> square(const ublas::matrix<T> &m)
  {
    ublas::matrix<T> s = m;
    for(unsigned int row=0; row<m.size1(); row++)
      for(unsigned int col=0; col<m.size2(); col++)
        s(row,col)*=m(row,col);
    return s;
  }


  //==============================
  /*! Sum of a vector. 
    \param v 
    
    \return 
  */
  template < typename T >
  T sum(const ublas::vector<T> & v)
  {
    T sum = 0;
    for(unsigned int i=0;i<v.size();i++)
      sum+=v[i];
    return sum;
  }

  //==============================
  /*! Sum of a matrix. 
    \param m 
    
    \return 
  */
  template < typename T >
  T sum(const ublas::matrix<T> & m)
  {
    T s = T(0);
    for(unsigned int row=0; row<m.size1(); row++)
      for(unsigned int col=0; col<m.size2(); col++)
        s+=m(row,col);

    return s;
  }

  //==============================
  /*! Square root of vector. 
    \param v1 
    
    \return 
  */
  template < typename T >
  ublas::vector<T> vsqrt(const ublas::vector<T> & v1)
  {
    ublas::vector<T> roots(v1.size());
    for(unsigned int i=0;i<v1.size();i++)
      {
        double m = static_cast<double>(v1[i]);
        roots[i] = static_cast<T>(sqrt(m));
      }
    return roots;
  }

  //==============================
  /*! Min value in a matrix. 
    \param m 
    
    \return 
  */
  template < typename T >
  T min(const ublas::matrix<T> & m)
  {
    T s = m(0,0);
    for(unsigned int row=0; row<m.size1(); row++)
      for(unsigned int col=0; col<m.size2(); col++)
        if(m(row,col) < s)
          s = m(row,col);
    return s;
  }

  //==============================
  /*! Min value in a vector. 
    \param v 
    
    \return 
  */
  template < typename T >
  T min(const ublas::vector<T> & v)
  {
    T s = v[0];
    for(unsigned int row=1; row<v.size(); row++)
      if(v[row] < s)
        s = v[row];
    return s;
  }

  //==============================
  /*! Max value in a matrix. 
    \param m 
    
    \return 
  */
  template < typename T >
  T max(const ublas::matrix<T> & m)
  {
    T s = m(0,0);
    for(unsigned int row=0; row<m.size1(); row++)
      for(unsigned int col=0; col<m.size2(); col++)
        if(m(row,col) > s)
          s = m(row,col);
    return s;
  }

  //==============================
  /*! Max value in a vector. 
    \param v 
    
    \return 
  */
  template < typename T >
  T max(const ublas::vector<T> & v)
  {
    T s = v[0];
    for(unsigned int row=1; row<v.size(); row++)
      if(v[row] > s)
        s = v[row];
    return s;
  }

  //==============================
  /*! Tells which index is the max in a vector. 
    \param v 
    
    \return 
  */
  template < typename T >
  int whichMax(const ublas::vector<T> &v)
  {
    T mx = max(v);
    for(unsigned int i=0; i<v.size(); i++)
      if(v[i]==mx)
        return i;
    return 0;
  }

  //==============================
  /*! Tells which index is the min in a vector. 
    \param v 
    
    \return 
  */
  template < typename T >
  int whichMin(const ublas::vector<T> &v)
  {
    T mn = min(v);
    for(unsigned int i=0; i<v.size(); i++)
      if(v[i]==mn)
        return i;
    return 0;
  }
  //==============================
  /*! Mean of a vector. 
    \param v 
    
    \return 
  */
  template < typename T >
  T mean(const ublas::vector<T> & v)
  {
    T mean = sum(v);
    mean/=v.size();
    return mean;
  }

  //==============================
  /*! Mean of a matrix. 
    \param m 
    
    \return 
  */
  template < typename T >
  T mean(const ublas::matrix<T> & m)
  {
    T mean = sum(m);
    mean/=m.size1() * m.size2();
    return mean;
  }


  //==============================
  /*! Variance of a vector. 
    \param v 
    \param m 
    
    \return 
  */
  template < typename T >
  double
  var(const ublas::vector<T> & v, double m)
  {
    double d;
    double sum = 0;
    for(unsigned int i=0; i<v.size(); i++)
      {
        sum+=(v[i]-m) * (v[i]-m);
      }
    sum/=v.size();
    return sum;
  }

  /*! Variance of a vector. 
    \param v 
    
    \return 
  */
  template < typename T >
  double
  var(const ublas::vector<T> & v)
  {
    return(var(v,mean(v)));
  }


  /*! Standard deviation of matrix. 
    \param v 
    
    \return 
  */
  template < typename T >
  double sd(const ublas::vector<T> & v)
  {
    return sqrt(var(v));
  }

  /*! Standard deviation of a matrix. 
    \param m 
    
    \return 
  */
  template < typename T >
  ublas::vector<double> sd(const ublas::matrix<T> & m)
  {
    ublas::vector<double> ret(ncol(m));
    for(unsigned int i=0;i<ncol(m);i++)
      {
        ublas::vector<double> temp = column(m,i);
        ret[i] = sd(temp);
      }
    return ret;
  }

}//end of namespace


#endif
