/*
* CEBL : CSU EEG Brain-Computer Interface Lab
*
* Author: Jeshua Bratman - jeshuabratman@gmail.com
*
* This file is part of CEBL.
*
* CEBL is free software; you can redistribute it and/or modify it.
* We only ask that if you use our code that you cite the source in
* your project or publication.
*
* EEG Group (www.cs.colostate.edu/eeg)
* Department of Computer Science
* Colorado State University
* 
*/

//--------------------------------------------------
//   cppR_utils.hpp
// Matrix and Vector utilities


#ifndef CPPR_UTILS_H //make sure this only gets included once
#define CPPR_UTILS_H

#include <fstream>
#include <algorithm>
#include <numeric>
#include <vector>
#include <iterator>
#include <boost/lexical_cast.hpp>
using namespace std;

#include <cppR/cppR_includes.hpp>

//--------------------------------------------------


//create a global random number generator
typedef boost::mt19937 base_generator_type;
static base_generator_type generator(time(NULL));

//--------------------------------------------------


namespace cppR
{
  //==============================
  /*! Creates a vector of n random numbers from a uniform distribution.
    \param n 
    
    \return vector of random numbers
  */
  template < typename T >
  ublas::vector<T> runif(int n)
  {
    ublas::vector<T> v(n);
    boost::uniform_real<T> uni_dist(0,1);
    boost::variate_generator<base_generator_type&, boost::uniform_real<T> > uni(generator, uni_dist);
    std::generate(v.data().begin(),v.data().end(),uni);
    return v;
  };

  //==============================
  /*! Creates a vector of n random numbers from a normal distribution.
    \param n number to generate
    \param mean mean of distribution
    \param sd standard deviation of distribution
    
    \return vector of random numbers
  */
  template < typename T >
  ublas::vector<T>
  rnorm(int n, double mean, double sd)
  {
    boost::normal_distribution<double> norm_dist(mean,sd);
    boost::variate_generator<base_generator_type&, boost::normal_distribution<double> > norm(generator, norm_dist);
    ublas::vector<double> v(n);
    std::generate(v.data().begin(),v.data().end(),norm);
    return v;
  }




  //==============================
  /*! Converts a ublas vector to a std vector.
    \param vec 
    
    \return 
  */
  template < typename T >
  std::vector<T>
  asStdVector(ublas::vector<T> vec)
  {
    std::vector<T> ret;
    for(unsigned int i=0;i<vec.size();i++)
      ret.push_back(vec[i]);
    return ret;
  }

  //==============================
  /*! Converts a std vector into a ublas vector. 
    \param vec std vector
    
    \return ublas vector
  */
  template < typename T >
  ublas::vector<T>
  asUblasVector(std::vector<T> vec)
  {
    ublas::vector<T> ret;
    ret.resize(vec.size());
    for(unsigned int i=0;i<vec.size();i++)
      ret(i) = vec[i];
    return ret;
  }

  //==============================
  /*! Randomly samples values in vector.
    \param vec 
    
    \return randomized vecto
  */
  template < typename T >
  ublas::vector<T>
  sample(ublas::vector<T> vec)
  {
    ublas::vector<T> ret = vec;
    std::random_shuffle(ret.begin(), ret.end());
    return ret;
  }

  //==============================
  //apply


  /*! Applies a 1 argument function to each value in a matrix.
    \param m 
    \param func function to apply
    
    \return 
  */
  template < typename T >
  ublas::matrix<T>
  apply(const ublas::matrix<T> &m, T (*func)(T))
  {
    ublas::matrix<T> ret(m.size1(),m.size2());
    for(unsigned i=0;i<m.size1();i++)
      for(unsigned j=0;j<m.size2();j++)
        ret(i,j) = func(m(i,j));
    return ret;
  }

  /*! Applies a 2-argument function to each value in a matrix.
    \param m 
    \param func function
    \param arg2 second argument to function
    
    \return 
  */
  template < typename T, typename U >
  ublas::matrix<T>
  apply(const ublas::matrix<T> &m,  T (*func)(T, U), U arg2)
  {
    ublas::matrix<T> ret(m.size1(),m.size2());
    for(unsigned int i=0;i<m.size1();i++)
      for(unsigned int j=0;j<m.size2();j++)
        ret(i,j) = func(m(i,j), arg2);
    return ret;
  }

  /*! Applies a 1 argument function to each value in a vector.
    \param m 
    \param func 
    
    \return 
  */
  template < typename T >
  ublas::vector<T>
  apply(const ublas::vector<T> &m, T (*func)(T))
  {
    ublas::vector<T> ret(m.size());
    for(unsigned i=0;i<m.size();i++)
        ret(i) = func(m(i));
    return ret;
  }

  //==============================

  /*! Apply function to each row of a matrix and return a vector of
    the result for each row. 

    Similar to the R function call: apply(matrix, 1, func).

    \param m matrix
    \param func function to apply to each row
    
    \return vector of function applied to each row of m
  */
  template < typename T >
  ublas::vector<T>
  rowApply(const ublas::matrix<T> &m, T (*func)(const ublas::vector<T>&))
  {
    ublas::vector<T> ret;
    ret.resize(m.size1());
    //loop through rows of m
    for(unsigned i=0;i<m.size1();i++)
      {
        //get the result of the function on this row
        T res = func(ublas::vector<T>(row(m,i)));
        ret[i] = res; 
      }
    return ret;
  }

  /*! Apply function to each column of a matrix and return a vector of
    the result for each row. 

    Similar to the R function call: apply(matrix, 2, func).

    \param m matrix
    \param func function to apply to each column
    
    \return vector of function applied to each column of m
  */
  template < typename T >
  ublas::vector<T>
  columnApply(const ublas::matrix<T> &m, T (*func)(const ublas::vector<T>&))
  {
    ublas::vector<T> ret;
    ret.resize(m.size1());
    //loop through cols of m
    for(unsigned i=0;i<m.size2();i++)
      {
        //get the result of the function on this row
        T res = func(ublas::vector<T>(column(m,i)));
        ret[i] = res; 
      }
    return ret;
  }



  //=======================================
  /*! Prints a matrix to standard output.
    \param m 
  */
  template < typename T >
  void
  printMatrix(const ublas::matrix<T> &m)
  {
    cout.setf(ios::fixed,ios::floatfield);
    cout.precision(5);
    for(unsigned int i=0;i<m.size1();i++)
      {
        for(unsigned int j=0;j<m.size2(); j++)
          {
            if(m(i,j) >= 0)
              cout << " ";
            cout << m(i,j) << "\t";
          }
        cout << "\n";
      }
  }
  //=======================================
  /*! Prints a vector to standard output.
    \param v 
  */
  template < typename T >
  void
  printVector(const ublas::vector<T> &v)
  {
    cout.setf(ios::fixed,ios::floatfield);
    cout.precision(5);
    for(unsigned int i=0;i<v.size();i++)
      {
        cout << v[i] << "\t";
      }
    cout << "\n";

  }
  //=======================================
  /*! Prints a std vector to standard output.
    \param v 
  */
  template < typename T >
  void
  printVector(const std::vector<T> &v)
  {
    cout.setf(ios::fixed,ios::floatfield);
    cout.precision(5);
    for(unsigned int i=0;i<v.size();i++)
      {
        cout << v[i] << "\t";
      }
    cout << "\n";
  }


 //=======================================

  /*! Prints the dimensions of a matrix to standard output.
    \param m 
  */
  template < typename T >
  void
  printMatrixDim(const ublas::matrix<T> &m)
  {
    cout << "[" << m.size1() << "  " << m.size2() << "]\n";
  }

  //=======================================

  /*! Writes a matrix as an R-readable table to a file.
    Load this in R with the read.table function.
    \param m 
    \param file 
  */
  template < typename T >
  void
  writeTable(const ublas::matrix<T> &m, string filename)
  {
    ofstream ofile;
    ofile.open(filename.c_str());

    if(ofile.fail())
      return;

    for(unsigned int i=0;i<m.size1();i++)
      {
        for(unsigned int j=0;j<m.size2(); j++)
          {
            ofile << m(i,j) << " ";
          }
        ofile << "\n";
      }

    ofile.close();
  }

  /*! Reads a matrix table from a file written by R's write.table

    \param file 
  */
  template < typename T >
  ublas::matrix<T>
  readTable(string filename)
  {
    ublas::matrix<T> ret;
    ifstream ifile;
    ifile.open(filename.c_str());

    if(ifile.fail())

      return ret;

    //loop through lines of file
    char buffer[1024];
    string str_temp;
    T val_temp;
    int i = 0;
    while(!ifile.eof())
      {
        ifile.getline(buffer,sizeof(buffer),'\n');
        //skip first line
        if(i <= 0)
          {
            i++;
            continue;
          }
        
        std::vector<T> row_vector;
        stringstream line_ss;
        line_ss << buffer ;
        line_ss >> str_temp;             
        //loop through whitespace delimited strings
        while(!line_ss.eof())
          {
            line_ss >> str_temp;
            //attempt to parse string as double
            try
              {
                val_temp = boost::lexical_cast<T>(str_temp);
              } 
            catch(...){ continue; }
            row_vector.push_back(val_temp);
          }
        //set row of matrix
        if(row_vector.size() > 0)
          {
            ret.resize(ret.size1()+1, row_vector.size());
            row(ret,ret.size1()-1) = cppR::asUblasVector(row_vector);
          }
        i++;
      }

    ifile.close();
    return ret;
  }



}//end of namespace

#endif
