/*!
 * CEBLViewGTK.cpp
 * \author Jeshua Bratman
 *
 * GTK View for CEBL
 */


#include "CEBLViewGTK.hpp"
#include "ToolbarMenu.hpp"
#include "WidgetUtils.hpp"

//combo factories
#include "DataSourceCombo.hpp"
#include "InterfaceCombo.hpp"

//tabs
#include "TabEEGRecording.hpp"
#include "TabDevice.hpp"
#include "TabMonitor.hpp"
#include "TabFilter.hpp"
#include "TabFeatures.hpp"
#include "TabDecision.hpp"
#include "TabClassifiers.hpp"
#include "TabTraining.hpp"
#include "TabRealTimeClassification.hpp"

//model classes
#include "../model/StringTable.hpp"
#include "../CompiledStrings.hpp"

//other widgets
#include "StatusBar.hpp"
#include "InfoBar.hpp"

//----------------------------------------------------------------------
// Constructors / Destructors

CEBLViewGTK::CEBLViewGTK(CEBLModel *model, int ac, char ** av) : CEBLView(model)
{
  this->ac = ac;
  this->av = av;

  //set up some defaults
  WINDOW_WIDTH = 800;
  WINDOW_HEIGHT = 700;

  //initialize tabs
  tabs.resize(0);

  //create string table
  //TODO: load path from prefs
  string string_table_filename = string(HOMEDIR) 
    + "/.cebl/conf/gui-string-table-en.txt";
  string_table = new StringTable();
  string_table->loadFromString(gui_string_table_en);
  //string_table->loadFromFile(string_table_filename);
}


CEBLViewGTK::~CEBLViewGTK()
{
  delete main_menu;
  for(unsigned int i=0;i<tabs.size();i++)
    delete tabs[i];
  delete data_source;
  delete interface_combo;
  delete status_bar;
  delete string_table;
  delete info_bar;
}

//----------------------------------------------------------------------
// GTK Event Callbacks

gboolean CEBLViewGTK::CB_WindowDelete(GtkWidget *widget,
                                      GdkEvent  *event,gpointer data)
{
  CEBLViewGTK * view  = (CEBLViewGTK*) data;
  for(unsigned int i=0; i<view->tabs.size();i++)
    {
      view->tabs[i]->onHide();
    }
  return false; //destroy event will occur
}
void CEBLViewGTK::CB_WindowDestroy(GtkWidget *widget, gpointer data)
{
  gtk_main_quit();
}


//----------------------------------------------------------------------
// Start the view

void CEBLViewGTK::Start()
{
  // GTK Initialization
  gtk_init(&ac, &av);

  // create the gui
  CreateMainWindow();
  CreateLayout();

  //show the main container
  gtk_widget_show(main_container);
  //show the window
  gtk_widget_show(main_window);
  //start up gtk
  gtk_main();
}



//----------------------------------------------------------------------
//Create the GUI


void CEBLViewGTK::CreateMainWindow()
{
  main_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);

  gtk_window_set_title(GTK_WINDOW(main_window),"CSU Electroencepholagram Brain-Computer Interface Lab (CEBL)");

  // extract the window's colors for later use
  GtkStyle *style = gtk_rc_get_style(main_window);
  BG_RED = style->bg[0].red;
  BG_GREEN = style->bg[0].green;
  BG_BLUE = style->bg[0].blue;

  // set window properties
  gtk_window_set_default_size(GTK_WINDOW(main_window), WINDOW_WIDTH, WINDOW_HEIGHT);
  gtk_container_set_border_width(GTK_CONTAINER(main_window),2);

  //connect destroy and delete signals
  g_signal_connect (G_OBJECT (main_window), "delete_event",
        	    G_CALLBACK (CB_WindowDelete), this);
  g_signal_connect (G_OBJECT (main_window), "destroy",
        	    G_CALLBACK (CB_WindowDestroy), this);

  //set main window in widget utils
  WidgetUtils::setMainWindow(main_window);
}


//! Create the layout of the window
void CEBLViewGTK::CreateLayout()
{

  //create main container and add it to the window
  main_container = gtk_vbox_new(false,0);
  gtk_container_add(GTK_CONTAINER(main_window), main_container);

  //----------------------------------------
  // TOP

  // menu bar
  main_menu = new ToolbarMenu(this);
  main_menu->attachMenu(main_window,main_container);


  //----------------------------------------
  // SEPARATE LEFT AND RIGHT

  // horizontal pane
  GtkWidget * main_pane = gtk_hpaned_new();
  gtk_box_pack_start(GTK_BOX(main_container),main_pane, true, true, 1);
  gtk_widget_show(main_pane);

  GtkWidget *right_container = gtk_vbox_new(false,0);
  gtk_widget_show(right_container);
  gtk_paned_pack2(GTK_PANED(main_pane), right_container, true, false);


  //----------------------------------------
  // RIGHT

  //title bar
  GtkWidget *align = gtk_alignment_new(0,0,0,1);
  //GtkWidget *title_label = gtk_label_new("");
  //gtk_container_add(GTK_CONTAINER(align),title_label);
  gtk_widget_show_all(align);
  gtk_box_pack_start(GTK_BOX(right_container), align, false, true, 5);

  //create notebook
  main_notebook = gtk_notebook_new();
  gtk_box_pack_start(GTK_BOX(right_container), main_notebook, true, true, 0);

  //gtk_notebook_popup_enable(GTK_NOTEBOOK(main_notebook));
  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(main_notebook), GTK_POS_LEFT);

  //create combo factories
  data_source = new DataSourceCombo(this);
  interface_combo = new InterfaceCombo(this);

  //create tabs
  tabs.push_back(new TabEEGRecording("EEG Recording", main_notebook, this));
  tabs.push_back(new TabDevice("EEG Data Source", main_notebook, this));
  tabs.push_back(new TabMonitor("Monitor", main_notebook, this));
  tabs.push_back(new TabFilter("Filter", main_notebook, this));
  tabs.push_back(new TabFeatures("Features", main_notebook, this));
  tabs.push_back(new TabClassifiers("Classifiers", main_notebook, this));
  tabs.push_back(new TabDecision("Decision Process", main_notebook, this));
  tabs.push_back(new TabTraining("Training", main_notebook, this));
  tabs.push_back(new TabRealTimeClassification("Real-Time Classification", main_notebook, this));
  old_page = 0;


  //show the notebook
  gtk_widget_show(main_notebook);

  //add change tab signal to notebook
  g_signal_connect(G_OBJECT(main_notebook),
        	   "switch-page",
        	   G_CALLBACK(CB_ChangeTab),
        	   (gpointer) this);

  //Status and Info Bars
  info_bar = new InfoBar(this);
  gtk_box_pack_end(GTK_BOX(main_container), info_bar->getContainer(), false, false, 1);
  status_bar = new StatusBar();
  gtk_box_pack_end(GTK_BOX(main_container), status_bar->getContainer(), false, false, 1);
}


//----------------------------------------------------------------------
// UPDATING TAB VIEWS

/*
 * Change the visible tab from old_page to page_num
 */
void CEBLViewGTK::CB_ChangeTab(GtkNotebook *notebook,
        		       GtkNotebookPage *page,
        		       guint page_num,
        		       gpointer data)
{
  CEBLViewGTK *view = (CEBLViewGTK*)data;

  //hide last page
  if(view->old_page >= 0
     && view->old_page < int(view->tabs.size()))
    {
      view->tabs[view->old_page]->onHide();
      view->tabs[view->old_page]->updateModel();
    }
  //show selected page
  view->tabs[page_num]->onShow();
  view->tabs[page_num]->updateView();
  view->old_page = page_num;
}

/*
 * Update display for main window (and current tab)
 */
void CEBLViewGTK::updateView()
{
   if(this->old_page >= 0
     && this->old_page < int(this->tabs.size()))
    {
      tabs[this->old_page]->updateView();
    }
   this->getDataSource()->updateView();
}

/*
 * Updates settings in corresponding model for current tab
 */
void CEBLViewGTK::updateModel()
{
   if(this->old_page >= 0
     && this->old_page < int(this->tabs.size()))
    {
      tabs[this->old_page]->updateModel();
    }
}

/*
 * Update the InfoBar information
 */
void CEBLViewGTK::updateInfoBar(){
  if(info_bar!=NULL)
    info_bar->update();
}

/*
 * Gets a string from the string table for display
 * See conf/gui-string-table-en.txt for strings
 */
const char * CEBLViewGTK::getString(string name)
{
  try
    {
      return string_table->getString(name);
    }
  catch(exception &e)
    {
      cerr << e.what();
      return "";
    }
}

