#include "EEGMonitor.hpp"
#include "TextUtils.hpp"
#include "cppR/cppR.hpp"
#include "Tab.hpp"
#include <values.h>
#include <iostream>

//constructor create widgets and initializes member variables
EEGMonitor::EEGMonitor(Tab *parent, int nchannels)
{
  //save a pointer to the parent tab
  this->parent_tab = parent;
  //initialize plot
  plot = NULL;
  if(nchannels < 0)
    this->num_channels = parent_tab->getView()->getModel()->channelsGetNumEnabledChannels();
  else
    this->num_channels = nchannels;
  //DEFAULTS
  num_display_samples = 0;
  zoom = 1.0;
  initialized = false;
  plot_wait = false;

  //create widgets
  GtkWidget *container = gtk_vbox_new(false, 0);
  plot_box = gtk_vbox_new(false, 0);
  scroll_window = gtk_scrolled_window_new(NULL,NULL);
  gtk_container_border_width(GTK_CONTAINER(scroll_window),0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll_window),
        			 GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(container),scroll_window, true, true, 0);


  //create controls
  box_controls = gtk_hbox_new(false, 0);

  //num samples spin button
  spin_samples = gtk_spin_button_new_with_range(1, MAXINT, 1);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_samples),1);
  gtk_widget_set_size_request(spin_samples, 80, 25);
  g_signal_connect(G_OBJECT(spin_samples),
        	   "value-changed",
        	   G_CALLBACK(CB_ChangeNumSamples),
        	   (gpointer) this);
  //zoom spin button
  spin_zoom = gtk_spin_button_new_with_range(.01, 500, .01);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_zoom),zoom);
  gtk_widget_set_size_request(spin_zoom, 80, 25);
  g_signal_connect(G_OBJECT(spin_zoom),
        	   "value-changed",
        	   G_CALLBACK(CB_ChangeZoom),
        	   (gpointer) this);

  //Place widgets
  gtk_box_pack_start(GTK_BOX(box_controls),gtk_label_new("Currently displaying "),false,false,2);
  gtk_box_pack_start(GTK_BOX(box_controls),spin_samples,false,false,2);
  gtk_box_pack_start(GTK_BOX(box_controls),gtk_label_new(" samples at a time."),false,false,2);

  gtk_box_pack_end(GTK_BOX(box_controls),spin_zoom,false,false,2);
  gtk_box_pack_end(GTK_BOX(box_controls),gtk_label_new("Zoom: "),false,false,2);

  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll_window),plot_box);
  gtk_box_pack_start(GTK_BOX(container),box_controls,false,false,5);

  //set outer container as parent class' container
  this->setWidgetContainer(container);

  //create plot
  this->initPlot();
}



//create and initialize the plot
void EEGMonitor::initPlot()
{
  this->num_channels = min(this->num_channels,72);
  plot = new EEGPlot(this->num_channels);

  //create default labels if labels are empty, or labels defaults
  if(labels.size() == 0 || labels[0] == "1")
    {
      labels.resize(0);
      for(int i=0; i<num_channels; i++)
        {
          labels.push_back(TextUtils::IntToString(i+1));
        }
    }


  //first decide on plot window sizes
  int window_width = 0;
  int window_height = 0;
  if(!this->is_detached)
    {
      gtk_window_get_size(GTK_WINDOW(parent_tab->getView()->getMainWindow()),&window_width,&window_height);
      window_width -= 100;
      window_height -= 200;
    }
  else
    {
      gtk_window_get_size(GTK_WINDOW(this->detached_window),&window_width,&window_height);
    }
  if(num_display_samples==0)
    num_display_samples = window_width;

  //create Plot
  plot->setLabels(this->labels);
  plot->setWindowWidth(window_width);
  plot->setWindowHeight(window_height);
  plot->setNumDisplaySamples(num_display_samples);
  plot->setBGColor(parent_tab->getView()->getBGRED(),
        	   parent_tab->getView()->getBGGREEN(),
        	   parent_tab->getView()->getBGBLUE());

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_samples),num_display_samples);

  //init plot and pack it into the window
  plot->init();
  gtk_box_pack_start(GTK_BOX(plot_box), (*plot), true, true, 0);

  //notify class that all plot initialization is complete at least once
  initialized = true;
}

//remove and recreate plot
void EEGMonitor::reInitPlot(int nchannels)
{
  if(nchannels >= 0)
    {
      this->num_channels = nchannels;
      this->num_channels = min(nchannels,72);
    }
  //set a timeout so as not to try plotting while plot doesn't exist
  plot_wait = true;

  // if we haven't even initialized once yet, skip this
  // otherwise remove and delete plot to prepare for a new one
  if(initialized)
    {
      if(plot!=NULL)
        {
          gtk_container_remove(GTK_CONTAINER(plot_box),(*plot));
          delete plot;
          plot = NULL;
        }
    }

  // create the plot
  initPlot();

  // turn off plot wait so plotting can continue
  plot_wait = false;
}

//delete plot and other allocated variables
EEGMonitor::~EEGMonitor()
{
  if(plot!=NULL)
    delete plot;
}

/**************************************************
 SETTERS
*/

/*
 * Change visibility of controls
 * Hides if visible = false
 * Shows if visible = true
 */
void EEGMonitor::setControlsVisible(bool visible)
{
  if(!visible)
    {
      gtk_widget_hide(box_controls);
      gtk_widget_hide(box_controls2);
    }
  else
    {
      gtk_widget_show(box_controls);
      gtk_widget_show(box_controls2);
    }
}

/* Set labels for the plot
 */
void EEGMonitor::setLabels(std::vector<std::string> labels)
{
  this->labels = labels;
}

/* Set zoom for the plot
 */
void EEGMonitor::setZoom(double zoom)
{
  this->zoom = zoom;
  if(plot!=NULL)
    {
      plot->setZoom(zoom);
    }
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(this->spin_zoom),zoom);
}

/* Set number of samples to display on the plot
 */
void EEGMonitor::setNumDisplaySamples(int n)
{
  this->num_display_samples = n;
  if(plot!=NULL)
    {
      plot->setNumDisplaySamples(n);
    }
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_samples),n);
}

/**************************************************
 PLOTTING
*/
void EEGMonitor::Plot(ublas::matrix<double> data)
{
  if(data.size2() < 1 || plot == NULL)
    {
      return;
    }

  //plot wait means we are waiting for plot to be re-created
  if(plot_wait)
    return;

  //plot the data
  plot->plot(data);
}


/**************************************************
 CALLBACKS
*/

void EEGMonitor::CB_ChangeNumSamples(GtkWidget *spin, gpointer data)
{
  EEGMonitor * monitor = ((EEGMonitor*)data);
  //int old_samples = monitor->num_display_samples;
  monitor->num_display_samples = int(gtk_spin_button_get_value(GTK_SPIN_BUTTON(monitor->spin_samples)));
  if(monitor->plot != NULL)
    monitor->plot->setNumDisplaySamples(monitor->num_display_samples);
}


void EEGMonitor::CB_ChangeZoom(GtkWidget *spin, gpointer data)
{
  EEGMonitor * monitor = ((EEGMonitor*)data);
  monitor->zoom = double(gtk_spin_button_get_value(GTK_SPIN_BUTTON(monitor->spin_zoom)));
  if(monitor->plot != NULL)
    monitor->plot->setZoom(monitor->zoom);
}

