/*! EEGPlot.hpp
 * \author Jeshua Bratman
 *
 * A gtk plot for eeg data.
 */


#include "EEGPlot.hpp"
#include <iostream>
using namespace std;

const char *PLOT_COLORS[] = {"black",
        		     "red",
        		     "darkorange",
        		     "purple",
        		     "darkgreen",
        		     "brown",
        		     "darkblue",
        		     "violet",
        		     "darkred",
        		     "orange",
        		     "darkbrown",
        		     "blue",
        		     "darkviolet"};
const int NUM_PLOT_COLORS = 13;




//Constructor
EEGPlot::EEGPlot(int num_channels)
{
  this->num_channels = num_channels;
  this->num_plotted_channels = 0;

  //create the buffers for data points
  px = new double*[num_channels];
  py = new double*[num_channels];
  datasets = new GtkPlotData*[num_channels];
  for(int channel=0;channel<num_channels;channel++)
    {
      datasets[channel] = GTK_PLOT_DATA(gtk_plot_data_new());
      px[channel] = NULL;
      py[channel] = NULL;
    }

  current_plot = NULL;

  //default configuration
  this->num_samples = 0;
  this->window_width = 500;
  this->window_height = 500;
  this->num_display_samples = 400;
  this->zoom = 1.0;
  this->num_layers = 0;
  this->ymax = 1000;
  this->initialized = false;
  this->redraw_timeout_id = -1;
}
//Destructor
EEGPlot::~EEGPlot()
{
  if(this->redraw_timeout_id >= 0)
    gtk_timeout_remove(this->redraw_timeout_id);

  for(int channel=0;channel<num_channels;channel++)
    {
      if(px != NULL && px[channel] != NULL)
        delete[] px[channel];
      if(py != NULL && py[channel] != NULL)
        delete[] py[channel];
    }
  if(px!=NULL)
    delete[] px;
  if(py!=NULL)
    delete[] py;
  if(datasets!=NULL)
    delete[] datasets;
}

//creates a new plot layer
GtkWidget *EEGPlot::newLayer(GtkWidget *canvas)
{
  num_layers++;
  return gtk_plot_new_with_size(NULL, .65, .45);
}

//get the vbox widget
EEGPlot::operator GtkWidget*()
{
  return vbox_container;
}


//sets channel labels
void EEGPlot::setLabels(std::vector<std::string> labels)
{
  this->labels = labels;
}

//-----------------------------------------------
//initialize the plot
void EEGPlot::init()
{
  GdkColor color;
  //gfloat scale = 1.0;
  gint page_width = int(this->window_width);
  gint page_height = int(this->window_height * 0.9);

  //create container
  this->vbox_container = gtk_vbox_new(FALSE,0);
  gtk_widget_show(this->vbox_container);

  //create canvas
  this->canvas = gtk_plot_canvas_new(page_width,page_height, 1.);
  GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(canvas), GTK_PLOT_CANVAS_DND_FLAGS);
  gtk_box_pack_start(GTK_BOX(vbox_container),canvas, TRUE, TRUE, 0);
  gtk_widget_show(canvas);

  //background color
  gdk_color_parse("white", &color);
  color.red = BG_RED;
  color.green = BG_GREEN;
  color.blue = BG_BLUE;

  gdk_color_alloc(gtk_widget_get_colormap(canvas), &color);
  gtk_plot_canvas_set_background(GTK_PLOT_CANVAS(canvas), &color);

  //create plot
  this->current_plot = newLayer(canvas);
  gdk_color_alloc(gtk_widget_get_colormap(current_plot), &color);
  gtk_plot_set_background(GTK_PLOT(current_plot), &color);

  //range and ticks
  gtk_plot_set_ticks(GTK_PLOT(current_plot), GTK_PLOT_AXIS_X, 20, 1);
  gtk_plot_set_ticks(GTK_PLOT(current_plot), GTK_PLOT_AXIS_Y, 40, 1);
  gtk_plot_set_xrange(GTK_PLOT(current_plot), 0, getDisplaySamples());
  gtk_plot_set_yrange(GTK_PLOT(current_plot), 0, this->ymax);

  gtk_plot_axis_show_ticks(gtk_plot_get_axis(GTK_PLOT(current_plot),GTK_PLOT_AXIS_RIGHT), false, false);
  gtk_plot_axis_show_ticks(gtk_plot_get_axis(GTK_PLOT(current_plot),GTK_PLOT_AXIS_LEFT), false, false);
  gtk_plot_axis_set_tick_labels(gtk_plot_get_axis(GTK_PLOT(current_plot),GTK_PLOT_AXIS_LEFT), NULL);

  //hide tick labels
  gtk_plot_axis_show_labels(gtk_plot_get_axis(GTK_PLOT(current_plot),GTK_PLOT_AXIS_LEFT), 0);
  gtk_plot_axis_show_labels(gtk_plot_get_axis(GTK_PLOT(current_plot),GTK_PLOT_AXIS_RIGHT), 0);

  //axis
  gtk_plot_axis_set_visible(gtk_plot_get_axis(GTK_PLOT(current_plot), GTK_PLOT_AXIS_TOP), FALSE);
  gtk_plot_axis_set_visible(gtk_plot_get_axis(GTK_PLOT(current_plot), GTK_PLOT_AXIS_RIGHT), FALSE);
  gtk_plot_axis_set_visible(gtk_plot_get_axis(GTK_PLOT(current_plot), GTK_PLOT_AXIS_BOTTOM), FALSE);
  gtk_plot_axis_set_visible(gtk_plot_get_axis(GTK_PLOT(current_plot), GTK_PLOT_AXIS_LEFT), FALSE);

  //axis name
  gtk_plot_axis_set_title(gtk_plot_get_axis(GTK_PLOT(current_plot), GTK_PLOT_AXIS_LEFT), "");
  gtk_plot_axis_move_title(gtk_plot_get_axis(GTK_PLOT(current_plot),GTK_PLOT_AXIS_LEFT),0,.145,.4);

  //hide grids
  //vertical,vertical dotted,horizontal,horizontal dotted
  gtk_plot_grids_set_visible(GTK_PLOT(current_plot), FALSE, TRUE, FALSE, FALSE);

  //show the plot
  gtk_widget_show(current_plot);

  //legends
  gtk_plot_hide_legends(GTK_PLOT(current_plot));

  //create lines
  this->lines = gtk_plot_canvas_plot_new(GTK_PLOT(current_plot));

  //placement of lines
  x1 = .05;
  x2 = .95;
  y1 = .05;
  y2 = .99;
  gtk_plot_canvas_put_child(GTK_PLOT_CANVAS(canvas), lines, x1, y1, x2, y2);
  GtkPlotCanvasChild *child;

  for(int channel=0;channel<this->num_channels;channel++)
    {
      //labels
      gtk_plot_data_show_labels(datasets[channel], FALSE);


      gdk_color_parse(PLOT_COLORS[channel%NUM_PLOT_COLORS], &color);
      gdk_color_alloc(gdk_colormap_get_system(), &color);
      gtk_plot_data_set_line_attributes(datasets[channel],
        				GTK_PLOT_LINE_SOLID,
        				GDK_CAP_NOT_LAST, GDK_JOIN_MITER, 1, &color);

      //add label to plot
      double text_size = window_width/90.0;
      double x_pos = 0.02;
      double y_pos = (double(channel)/num_channels) * .953  + 0.05;
      string label;
      if(unsigned(channel) < labels.size())
        label = labels[channel];
      else
        label = "unnamed";
      //don't allow empty labels
      if(label == "")
        label = "unnamed";

      child = gtk_plot_canvas_text_new("Times-Roman", (int)text_size, 0, NULL, NULL, TRUE,
        			       GTK_JUSTIFY_CENTER,
        			       label.c_str());

      gtk_plot_canvas_put_child(GTK_PLOT_CANVAS(canvas), child, x_pos, y_pos, 0, 0);

      //add data to plot and show
      gtk_plot_add_data(GTK_PLOT(current_plot), datasets[channel]);
      gtk_widget_show(GTK_WIDGET(datasets[channel]));
    }
   //paint canvas
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(canvas));
  this->initialized = true;
}




//-----------------------------------------------------------

void EEGPlot::setNumDisplaySamples(int n)
{
  this->num_display_samples = n;
  if(this->initialized)
    {
      this->queueRedraw();
    }
}

//-----------------------------------------------------------

void EEGPlot::setZoom(double zoom)
{
  if(zoom == this->zoom)
    {
      return;
    }

  //set the zoom
  this->zoom = zoom;
  if(zoom <= 0.01)
    zoom = 0.01;

  //TODO set the data points and replot



  //queue a redraw
  if(this->initialized)
    {
      this->queueRedraw();
    }
}

//-----------------------------------------------------------

void EEGPlot::plot(ublas::matrix<double> data)
{

  if(data.size1() <= 0 || data.size2() <= 0)
    return;


  if(data.size1() > 72){
    data.resize(72,data.size2());
    //    cerr << "ERROR trying to plot too many channels. Reducing to 72.\n";
  }
  //----------------------------------------
  //select how many samples are going to be displayed

  int size;
  int max_samples = getDisplaySamples();
  int orig_size = num_samples;
  int num_new_samples = data.size2();
  int num_old_samples = num_samples;

  num_samples += num_new_samples;
  if(num_samples > max_samples)
    {
      num_old_samples = max_samples-num_new_samples;
      if(num_new_samples > max_samples)
        {
          num_new_samples = max_samples;
          num_old_samples = 0;
        }
      size = max_samples;
    }
  else
    {
      size = num_samples;
    }
  num_samples = size;

  //----------------------------------------
  //plot the data
  //std::cout << "Recieved " << data.size2() << " samples. \n";

  //std::cout << "Using " << num_old_samples << " old samples. \n";
  //std::cout << "Using " << num_new_samples << " new samples. \n";
  //std::cout << "Size =  " << size << ". \n";

  int nchannels = num_channels;
  if(unsigned(num_channels) > data.size1())
    nchannels = data.size1();
  this->num_plotted_channels = nchannels;

  double ymax = this->ymax;
  double yscale = zoom * 5.0 / (num_channels * .2);
  double yoffset = ymax/nchannels;
  double *px_new,*py_new;
  int new_array_counter;
  for(int channel=0;channel<nchannels;channel++)
    {
      px_new = new double[size];
      py_new = new double[size];

      //copy over old data
      new_array_counter = 0;
      for(int i = orig_size-num_old_samples;
          i < orig_size;
          i++)
        {
          py_new[new_array_counter] = py[channel][i];
          px_new[new_array_counter] = (new_array_counter+1);
              new_array_counter++;
        }

      //copy over new data
      for(unsigned int i = data.size2() - num_new_samples;
          i < data.size2();
          i++)
        {
          py_new[new_array_counter] = data(channel,i)*yscale + ((nchannels-channel))*yoffset;
          px_new[new_array_counter] = new_array_counter;
          new_array_counter++;
        }

      //delete old arrays over arrays
      if(px != NULL && px[channel] != NULL)
        delete[] px[channel];
      if(py != NULL && py[channel] != NULL)
        delete[] py[channel];

      //copy over new data
      px[channel] = px_new;
      py[channel] = py_new;
    }
  this->redraw();
}

//--------------------------------------------------

void EEGPlot::redraw()
{
  if(this == NULL || current_plot == NULL)
    return;

  if(this->redraw_timeout_id >= 0)
    gtk_timeout_remove(this->redraw_timeout_id);

  gtk_plot_set_xrange(GTK_PLOT(current_plot), 0, this->num_display_samples);
  int dataset_index;
  for(int channel=0;channel<this->num_plotted_channels;channel++)
    {

      dataset_index = channel;//this->num_plotted_channels - 1 - channel;

      gtk_plot_data_set_numpoints(datasets[dataset_index], num_samples);
      gtk_plot_data_set_x(datasets[dataset_index], px[channel]);
      gtk_plot_data_set_y(datasets[dataset_index], py[channel]);
    }
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(canvas));
}


//--------------------------------------------------

void EEGPlot::queueRedraw()
{
  if(this->redraw_timeout_id < 0)
    {
      this->redraw_timeout_id = gtk_timeout_add(200, timedRedraw, this);
    }
  else
    {
      gtk_timeout_remove(this->redraw_timeout_id);
      this->redraw_timeout_id = gtk_timeout_add(200, timedRedraw, this);
    }
}
//--------------------------------------------------

//! Inform of the width of container plot is in.
void EEGPlot::setWindowWidth(double width)
{
  if(width <= 1224)
    this->window_width = width;
  else
    this->window_width = 1224;
}

//! Inform of the height of container plot is in.
void EEGPlot::setWindowHeight(double height)
{
 if(height <= 1024)
    this->window_height = height;
  else
    this->window_height = 1024;
}

//--------------------------------------------------

gint EEGPlot::timedRedraw(gpointer data)
{
  EEGPlot *plot = (EEGPlot*)data;
  plot->redraw();
  return 0;
}

