/*!
 * InterfaceCombo.cpp
 * \author Jeshua Bratman
 *
 * Combo box factory for uniform selection of training interfaces.
 */


#include "CEBLViewGTK.hpp"

#include "InterfaceCombo.hpp"
#include "Exceptions.hpp"
#include "interfaces/EEGInterface.hpp"
#include "interfaces/KeyboardPie.hpp"
#include "interfaces/RobotPie.hpp"
#include "interfaces/CursorPie.hpp"

using namespace std;


//constructor
InterfaceCombo::InterfaceCombo(CEBLViewGTK *view)
{
  this->view = view;
  this->updating_view = false;
  this->selected_interface = 0;

  //currently list of interfaces is hardcoded in here
  //eventually this should change if we make a plugin
  //based system for the interfaces
  this->interfaces.push_back("Simple Pie");
  this->interface_pointers.push_back(NULL);
  this->interfaces.push_back("Keyboard Pie");
  this->interface_pointers.push_back(NULL);
  this->interfaces.push_back("Robot Pie");
  this->interface_pointers.push_back(NULL);
  this->interfaces.push_back("Cursor Pie");
  this->interface_pointers.push_back(NULL);
}

InterfaceCombo::~InterfaceCombo()
{
  //delete all created interfaces
  for(unsigned int i=0;i < interface_pointers.size();i++)
    {
      if(this->interface_pointers[this->selected_interface] != NULL)
        delete this->interface_pointers[i];
    }
}

//get pointer to selected interface
//and creates the interface if it doesn't already exist

EEGInterface * InterfaceCombo::getInterface()
{
  EEGInterface * ret = this->interface_pointers[this->selected_interface];
  //create a pointer to the interface if it doesn't yet exist
  if(ret == NULL)
    {
      if(this->interfaces[this->selected_interface] == "Simple Pie")
        {
          this->interface_pointers[this->selected_interface] = new PieInterface();
        }
      else if(this->interfaces[this->selected_interface] == "Keyboard Pie")
        {
          this->interface_pointers[this->selected_interface] = new KeyboardPie();
        }
      else if(this->interfaces[this->selected_interface] == "Robot Pie")
        {
          this->interface_pointers[this->selected_interface] = new RobotPie();
        }
      else if(this->interfaces[this->selected_interface] == "Cursor Pie")
        {
          this->interface_pointers[this->selected_interface] = new CursorPie();
        }
      ret = this->interface_pointers[this->selected_interface];
    }
  CEBLModel * model = view->getModel();
  ret->setNumClasses(model->trainingGetNumClasses());
  ret->setClassLabels(model->trainingGetClassLabels());
  return ret;
}

//create a new combo widget
GtkWidget * InterfaceCombo::getCombo()
{
  GtkWidget* combo = gtk_combo_box_new_text();

  this->combos.push_back(combo);
  this->num_interfaces.push_back(0);
  this->updateComboBox(combos.size()-1);

  g_signal_connect(G_OBJECT(combo),
        	   "changed",
        	   G_CALLBACK(CB_changeInterface),
        	   (gpointer) this);
  return combo;
}

//update the view from the model
void InterfaceCombo::updateView()
{
  this->updating_view = true;
  for (unsigned int i=0;i<combos.size();i++)
    {
      this->updateComboBox(i);
    }
  this->updating_view = false;
}


//update a combo box from the model
void InterfaceCombo::updateComboBox(int index)
{
  GtkWidget * combo = combos[index];
  int num_interfaces = this->num_interfaces[index];

  //remove all the previous interfaces
  for(int i=num_interfaces-1;i>=0;i--)
    {
      gtk_combo_box_remove_text(GTK_COMBO_BOX(combo),i);
    }

  //eventually we might want to update interfaces
  //vector here if its going to change.
  //currently this whole function is a little unnecessary
  //because interfaces are not changing dynamically
  num_interfaces = interfaces.size();

  int selected = this->selected_interface;
  for(int i=0;i<num_interfaces;i++)
    {
      gtk_combo_box_append_text(GTK_COMBO_BOX(combo),interfaces[i].c_str());
    }
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo),selected);
  this->num_interfaces[index] = num_interfaces;
}

//returns the selected interface ID
int InterfaceCombo::getSelectedInterfaceID()
{
  return this->selected_interface;
}

//CALLBACKS
void InterfaceCombo::CB_changeInterface(GtkWidget * w, gpointer data)
{
  InterfaceCombo * d = (InterfaceCombo*)data;
  if(!d->updating_view)
    {
      string selected_text = gtk_combo_box_get_active_text(GTK_COMBO_BOX(w));
      for(unsigned int i=0; i < d->interfaces.size();i++)
        if(selected_text==d->interfaces[i])
          d->selected_interface = i;
      d->view->updateView();
      d->updateView();
    }
}


