/*!
 * InterfaceConfigurationWindow.cpp
 * \author Jeshua Bratman
 *
 * Options window for configuring EEG training and classifying.
 */

#include "InterfaceConfigurationWindow.hpp"
#include "CEBLViewGTK.hpp"
#include "WidgetPanel.hpp"
#include "InterfaceCombo.hpp"
#include "interfaces/EEGInterface.hpp"
#include "../Exceptions.hpp"

//----------------------------------------------------------------------
// Constructors / Destructors

InterfaceConfigurationWindow::InterfaceConfigurationWindow(CEBLViewGTK * view)
{
  this->interface_parameters_panel = NULL;
  this->view = view;
  this->window_created = false;
  this->window_open = false;
  this->updating_view = false;
}

InterfaceConfigurationWindow::~InterfaceConfigurationWindow()
{

}

//----------------------------------------------------------------------


GtkWidget * InterfaceConfigurationWindow::getButton()
{
  GtkWidget* btn = gtk_button_new_with_label("Open Interface Configuration");
  this->open_close_buttons.push_back(btn);
  g_signal_connect(G_OBJECT(btn),
        	   "clicked",
        	   G_CALLBACK(CB_openCloseWindow),
        	   (gpointer) this);
  return btn;
}



//----------------------------------------------------------------------

void InterfaceConfigurationWindow::createWindow()
{
  if(!this->window_created)
    {
      int num_classes = this->view->getModel()->trainingGetNumClasses();
      EEGInterface * interface = this->view->getInterfaceCombo()->getInterface();
      GtkWidget * main_window = this->view->getMainWindow();
      std::vector<string> class_labels = this->view->getModel()->trainingGetClassLabels();

      // create window
      this->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);

      // set window options
      gtk_window_set_title(GTK_WINDOW(window), "Configure Interface");
      gtk_window_set_position(GTK_WINDOW(window),GTK_WIN_POS_CENTER_ON_PARENT);
      gtk_window_set_transient_for(GTK_WINDOW(window),GTK_WINDOW(main_window));
      gtk_window_set_destroy_with_parent(GTK_WINDOW(window), true);
      gtk_window_set_default_size(GTK_WINDOW(window),200,0);

      //main container
      GtkWidget *container = gtk_vbox_new(false, 0);
      gtk_container_add(GTK_CONTAINER(window), container);

      g_signal_connect(G_OBJECT(window), "delete_event",
        	       G_CALLBACK(CB_windowDelete), this);
      //Top Decoration
      GtkWidget *params_label_box = gtk_hbox_new(false, 0);
      GtkWidget *classes_label_box = gtk_hbox_new(false, 0);

      //preferences icon
      GtkWidget *pref_image;
      pref_image = gtk_image_new_from_stock(GTK_STOCK_PREFERENCES,GTK_ICON_SIZE_LARGE_TOOLBAR);
      gtk_box_pack_start(GTK_BOX(params_label_box), pref_image, false, true, 0);

      //--------------------------------------------------
      //PARAMETERS
      GtkWidget *param_text;
      param_text = gtk_label_new("");
      gtk_label_set_markup(GTK_LABEL(param_text),"<b><big>Interface Parameters</big></b>");
      gtk_box_pack_start(GTK_BOX(params_label_box), param_text, false, true, 0);

      //add boxes
      gtk_box_pack_start(GTK_BOX(container), params_label_box, false, false, 0);
      gtk_box_pack_start(GTK_BOX(container), gtk_hseparator_new(), false, false, 0);

      //-------------------------------------------------
      //config
      gtk_box_pack_start(GTK_BOX(container),gtk_hseparator_new(), false, false, 0);
      gtk_box_pack_start(GTK_BOX(container),gtk_label_new("Interface Params"), false, false, 0);

      //Add parameters if there is a feature extractor
      GtkWidget *panel_container = gtk_hbox_new(false,0);
      if(interface != NULL)
        {
          if(interface_parameters_panel!=NULL)
            delete interface_parameters_panel;
          std::map<std::string, CEBL::Param> params = interface->getParamsList();
          interface_parameters_panel = new WidgetPanel(params);

          if(params.size() > 0)
            panel_container = interface_parameters_panel->getContainer();
          else
            panel_container = gtk_label_new("No options for selected interface.");
        }
      gtk_box_pack_start(GTK_BOX(container),panel_container, false, false, 0);

      //--------------------------------------------------
      //CLASSES
      GtkWidget *pref_text;
      pref_text = gtk_label_new("");
      gtk_label_set_markup(GTK_LABEL(pref_text),"<b><big>Classes</big></b>");
      gtk_box_pack_start(GTK_BOX(classes_label_box), pref_text, false, true, 0);

      //add boxes
      gtk_box_pack_start(GTK_BOX(container), classes_label_box, false, false, 0);
      gtk_box_pack_start(GTK_BOX(container), gtk_hseparator_new(), false, false, 0);

      //ENTRY BOXES

      //vector of gtk widgets
      class_labels_entries.resize(num_classes);

      //loop through each class
      for(int i=0; i<num_classes;i++)
        {
          GtkWidget *box = gtk_hbutton_box_new();
          gtk_button_box_set_layout(GTK_BUTTON_BOX(box),GTK_BUTTONBOX_START);

          std::stringstream l;
          l << "Class " << (i);
          std::string label = l.str();
          gtk_box_pack_start(GTK_BOX(box),gtk_label_new(label.c_str()), false, false, 0);
          //create entry
          class_labels_entries[i] = gtk_entry_new();
          gtk_entry_set_text(GTK_ENTRY(class_labels_entries[i]), class_labels[i].c_str());
          gtk_box_pack_start(GTK_BOX(box),class_labels_entries[i], true, true, 0);

          gtk_box_pack_start(GTK_BOX(container),box, true, true, 0);
        }


      //OK BUTTON
      GtkWidget *btn_ok = gtk_button_new_from_stock(GTK_STOCK_OK);
      g_signal_connect(G_OBJECT(btn_ok), "clicked",
        	       G_CALLBACK(CB_windowClose), this);
      gtk_box_pack_start(GTK_BOX(container), btn_ok, false, false, 0);

      //SHOW WINDOW
      this->window_created = true;
    }
}


void InterfaceConfigurationWindow::closeWindow()
{
 if(this->window_open && this->window_created)
    {
      gtk_widget_hide(this->window);
      this->window_open = false;
      this->applyConfig();
    }
}

void InterfaceConfigurationWindow::openWindow()
{
  this->updateView();
  if(!this->window_open)
    {
      if(!this->window_created)
        {
          this->createWindow();
        }
      gtk_widget_show_all(this->window);
      this->window_open = true;
    }
}


//----------------------------------------------------------------------

void InterfaceConfigurationWindow::applyConfig()
{
  int num_classes = this->view->getModel()->trainingGetNumClasses();
  EEGInterface * interface = this->view->getInterfaceCombo()->getInterface();

  this->class_labels.resize(num_classes);
  for(unsigned i=0; i<class_labels_entries.size();i++)
    {
      if(int(i) < num_classes)
        class_labels[i] = gtk_entry_get_text(GTK_ENTRY(class_labels_entries[i]));
    }

  //update model
  this->view->getModel()->trainingSetClassLabels(class_labels);

  //update interface
  if(interface!=NULL)
    {
      if(interface_parameters_panel!=NULL)
        interface->setParamsList(interface_parameters_panel->getParams());
      interface->setClassLabels(class_labels);
    }


}

//----------------------------------------------------------------------

void InterfaceConfigurationWindow::updateView()
{
  this->updating_view = true;
  if(this->window_created)
    {
      gtk_widget_destroy(this->window);
      this->window = NULL;
      this->window_created = false;
      this->createWindow();
      if(this->window_open)
        {
          gtk_widget_show_all(this->window);
        }
    }
  this->updating_view = false;
}

//----------------------------------------------------------------------
// CALLBACKS

void InterfaceConfigurationWindow::CB_openCloseWindow(GtkWidget *widget, gpointer data)
{
  InterfaceConfigurationWindow * iw = (InterfaceConfigurationWindow *)data;

 if(iw->window_open)
    {
      iw->closeWindow();
    }
  else
    {
      iw->openWindow();
    }
}

gboolean InterfaceConfigurationWindow::CB_windowDelete(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  InterfaceConfigurationWindow * iw = (InterfaceConfigurationWindow *)data;
  iw->closeWindow();
  return true;
}

void InterfaceConfigurationWindow::CB_windowClose(GtkWidget *widget, gpointer data)
{
  InterfaceConfigurationWindow * iw = (InterfaceConfigurationWindow *)data;
  iw->closeWindow();
}

