/*!
 * TabFeatures.cpp
 * \author Jeshua Bratman
 *
 */

#include "TabFeatures.hpp"
#include "EEGMonitor.hpp"
#include "WidgetUtils.hpp"
#include "DataSourceCombo.hpp"

//----------------------------------------------------------------------
// Constructors / Destructor

TabFeatures::~TabFeatures()
{
  if(plot!=NULL)
    delete plot;
}


//----------------------------------------------------------------------
// Create the GUI

void TabFeatures::CreateGUI()
{
  //add title
  GtkWidget *title = gtk_label_new("");
  gtk_label_set_markup(GTK_LABEL(title),"<big>Features</big>");
  TabAdd(title);
  TabAdd(gtk_hseparator_new());



  this->continue_plotting = false;

  //-------------------------------------------------
  //Feature Selection

  GtkWidget *hbox_features = gtk_hbox_new(false, 0);

  this->updating_view = false;
  this->num_features = 0;
  this->selected_feature = "";
  combo_features = gtk_combo_box_new_text();


  //configure the combo box and add callback
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_features),0);
  g_signal_connect(G_OBJECT(combo_features),
        	   "changed",
        	   G_CALLBACK(CB_changeFeature),
        	   (gpointer) this);

  gtk_box_pack_start(GTK_BOX(hbox_features), combo_features, false, false, 0);

  //is_made check box
  check_feature_trained = gtk_check_button_new_with_label("Feature Trained");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_feature_trained),false);
  gtk_widget_set_sensitive(check_feature_trained,false);

  gtk_box_pack_end(GTK_BOX(hbox_features),check_feature_trained,false, false, 0);

  gtk_widget_show_all(hbox_features);
  TabAdd(hbox_features);

  //----------------------------------------
  //create buttons

  GtkWidget * btn_box1 = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(btn_box1),GTK_BUTTONBOX_START);

  btn_start = gtk_button_new_with_label("Start Monitoring");
  //  gtk_widget_set_size_request(btn_start,30,20);
  g_signal_connect(G_OBJECT(btn_start),
        	   "clicked",
        	   G_CALLBACK(CB_startPlot),
        	   (gpointer) this);

  btn_stop = gtk_button_new_with_label("Stop");
  gtk_widget_set_sensitive(btn_stop,false);
  //  gtk_widget_set_size_request(btn_stop,30,2);
  g_signal_connect(G_OBJECT(btn_stop),
        	   "clicked",
        	   G_CALLBACK(CB_stopPlot),
        	   (gpointer) this);

  gtk_box_pack_start(GTK_BOX(btn_box1),btn_start,false,false,2);
  gtk_box_pack_start(GTK_BOX(btn_box1),btn_stop,false,false,2);
  gtk_box_pack_start(GTK_BOX(btn_box1),gtk_label_new("Data Source: "), false, false, 2);
  gtk_box_pack_start(GTK_BOX(btn_box1),getView()->getDataSource()->getCombo(),false,false,2);


  //create space for widget panel
  panel_box = gtk_vbox_new(false, 0);
  panel = NULL;
  //pack everything into frame
  GtkWidget *features_box = gtk_vbox_new(false,0);
  gtk_box_pack_start(GTK_BOX(features_box),panel_box,false,false,2);
  gtk_box_pack_start(GTK_BOX(features_box),btn_box1,false,false,2);
  TabFrameAdd(features_box,"Feature Configuration");

  //plot
  plot = NULL;



  updateView();

  //by default set first feature in combo box active if it has not been set
  if(gtk_combo_box_get_active(GTK_COMBO_BOX(combo_features))==-1)
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo_features),0);


}


//update view from model

void TabFeatures::updateView()
{
  //set flag to indicate process of updating view
  //in order to stop callbacks from triggering
  this->updating_view = true;

  //get information from model about features
  CEBLModel * model = getView()->getModel();
  std::vector<string> names = model->featuresGetNameList();
  std::vector<string> paths = model->featuresGetPathList();
  string model_selected_feature = model->featuresGetSelected();

  {
    this->selected_feature = model_selected_feature;
    this->feature_names = names;
    for(int i = this->num_features-1; i >= 0; i--)
      {
        gtk_combo_box_remove_text(GTK_COMBO_BOX(combo_features),i);
      }
    this->num_features = names.size();

    // add in all features
    for(unsigned int i=0;i<names.size();i++)
      {
        gtk_combo_box_append_text(GTK_COMBO_BOX(combo_features),names[i].c_str());
        if(names[i]==model_selected_feature)
          {
            gtk_combo_box_set_active(GTK_COMBO_BOX(combo_features),i);
          }
      }
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_feature_trained),
        			 model->featureIsTrained());
    // create panel
    if (this->panel != NULL)
      {
        gtk_container_remove(GTK_CONTAINER(this->panel_box),this->panel->getContainer());
        delete this->panel;
        this->panel = NULL;
      }

    //update panel
    if(model_selected_feature != "")
      {
        try
          {
            std::map<std::string, CEBL::Param> params = model->featureGetParams(model_selected_feature);
            this->panel = new WidgetPanel(params);
          }
        catch(...)
          {
            this->panel = NULL;
            cerr << "ERROR: exception occured when trying to get parameters for feature.\n";
          }
      }
    if (this->panel != NULL)
      {
        gtk_container_add(GTK_CONTAINER(this->panel_box),this->panel->getContainer());
        gtk_widget_show_all(this->panel_box);

      }
  }//end of conditional concerning whether update should occur

  //update button sensitivity
  // change button sensitivity based on stat of collection
  if(getView()->getModel()->dataIsStarted())
    {
      gtk_widget_set_sensitive(combo_features,false);
      gtk_widget_set_sensitive(btn_start,false);
      gtk_widget_set_sensitive(panel_box,false);
      gtk_widget_set_sensitive(btn_stop,true);
    }
  else
    {
      gtk_widget_set_sensitive(combo_features,true);
      gtk_widget_set_sensitive(panel_box,true);
      gtk_widget_set_sensitive(btn_start,true);
      gtk_widget_set_sensitive(btn_stop,false);
    }
  this->updating_view = false;
}

//update the model from the widgets
void TabFeatures::updateModel()
{
  // set the selected feature's params
  if(this->panel != NULL){
    this->getView()->getModel()->featureSetParams(this->panel->getParams());
  }
}

//! this function is called whenever a different tab is selected
void TabFeatures::onHide()
{
  updateModel();
  getView()->updateInfoBar();
  this->stopPlotting();

  /* lags is currently unused by the classifier
   * uncomment this section when lags is once again used
  CEBL::Params params = getView()->getModel()->featureGetParams();
  int lags = (params["lags"]).getInt();

  if(lags>=0){
    getView()->getModel()->classifierReset(params);
  }
  */

}

//----------------------------------------------------------------------
// PLOTTING


//! Create the plot canvas
void TabFeatures::createPlot()
{
  plot_size = this->getView()->getModel()->channelsGetNumEnabledChannels();
  if(plot==NULL)
    {
      plot = new EEGMonitor(this,plot_size);
      plot->reInitPlot(plot_size);
      TabAdd(*plot, true, true, 0);
    }
  else
    {
      plot->reInitPlot(plot_size);
    }
}


//! start model reading data and plotting data here
void TabFeatures::startPlotting()
{
  if(this->panel != NULL){
    this->getView()->getModel()->featureSetParams(this->panel->getParams());
  }

  try
    {
      getView()->getModel()->dataStart();
    }
  catch(exception &e)
    {
      WidgetUtils::AlertError("Error Starting Data Source", "There was an error starting the data source. \n("+string(e.what())+")");
      return;
    }
  if(!getView()->getModel()->dataIsStarted())
    {
      WidgetUtils::AlertError("Error Starting Data Source", "There was an error starting the data source. Check device configuration.");
    }
  else
    {
      this->continue_plotting = true;
      this->createPlot();
      g_timeout_add(100, timedPlot, this);
    }
  this->updateView();
}


//! plot on timeout call
gint TabFeatures::timedPlot(gpointer parent)
{
  TabFeatures * tab = (TabFeatures*)parent;
  if(!tab->getView()->getModel()->dataIsStarted())
    {
      tab->stopPlotting();
    }
  if(!tab->getView()->getModel()->featureIsTrained())
    {
      tab->stopPlotting();
    }
  if(!tab->continuePlotting())
    {
      return false;
    }
  try
    {
      EEGData data = tab->getView()->getModel()->dataReadAll();
      data = tab->getView()->getModel()->featuresExtract(data);
      //upate plot if number of channels is different

      if(min(data.size1(),72) != tab->plot_size)
        {
          tab->plot_size = min(data.size1(),72);
          tab->plot->reInitPlot(tab->plot_size);
        }

      ublas::matrix<double> d = (data.getMatrix());

      if(tab->plot!=NULL)
        {
          tab->plot->Plot(data);
        }
      else
        {
          tab->stopPlotting();
        }
    }
  catch(DataExceptionUnderflow& e)
    {
      cout << e.what() << "\n";
    }
  catch(exception& e)
    {
      string msg = "Failed to process data. \n(" + string(e.what()) + ")";
      WidgetUtils::AlertError("Error Processing Data",msg);
      tab->stopPlotting();
    }
  return true;
}

//! stop plotting from data source
void TabFeatures::stopPlotting()
{
  this->continue_plotting = false;
  getView()->getModel()->dataStop();
  updateView();
}


//----------------------------------------------------------------------
// CALLBACKS


void TabFeatures::CB_startPlot(GtkWidget *w, gpointer data)
{
  TabFeatures * tab = (TabFeatures*)data;
  tab->startPlotting();
}
void TabFeatures::CB_stopPlot(GtkWidget *w, gpointer data)
{
  TabFeatures * tab = (TabFeatures*)data;
  tab->stopPlotting();
}


void TabFeatures::CB_changeFeature(GtkWidget *w, gpointer data)
{
  TabFeatures* tab = (TabFeatures*)data;

  //as long as the update view is not currently running,
  //set the model's selected feature to the selected feature
  //in combo box
  if(tab->num_features > 0 && !tab->updating_view)
    {
      string feature = gtk_combo_box_get_active_text(GTK_COMBO_BOX(tab->combo_features));
      try
        {
          if(feature != tab->selected_feature)
            {
              // set the previously selected feature's params
              if(tab->panel != NULL){
        	tab->getView()->getModel()->featureSetParams(tab->panel->getParams());
              }
              // now tell model what feature is selected
              tab->getView()->getModel()->featuresSetSelected(feature);

            }
        }
      catch(...)
        {
          cerr << "Error selecting feature.\n";
        }
      tab->updateView();
    }

}



