/*!
 * TabFilter.cpp
 * \author Jeshua Bratman
 *
 */

#include "TabFilter.hpp"
#include "WidgetUtils.hpp"
#include "EEGMonitor.hpp"

//----------------------------------------------------------------------
// Constructors / Destructor


TabFilter::~TabFilter()
{

}


//----------------------------------------------------------------------
// Create the GUI


void TabFilter::CreateGUI()
{
  //add title
  GtkWidget *title = gtk_label_new("");
  gtk_label_set_markup(GTK_LABEL(title),"<big>Filter</big>");
  TabAdd(title);
  TabAdd(gtk_hseparator_new());



  this->updating_view = true;
  this->num_filters = 0;
  this->selected_filter = "";
  this->is_collecting = false;
  this->training_collected = false;
  this->num_seconds = 7;
  //-------------------------------------------------
  //Filter Selection
  GtkWidget *hbox_filters = gtk_hbox_new(false, 0);

  combo_filters = gtk_combo_box_new_text();

  //configure the combo box and add callback
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_filters),0);
  g_signal_connect(G_OBJECT(combo_filters),
                   "changed",
                   G_CALLBACK(CB_changeFilter),
                   (gpointer) this);

  gtk_box_pack_start(GTK_BOX(hbox_filters), combo_filters, false, false, 0);

  //is_made check box
  check_filter_made = gtk_check_button_new_with_label("Filter Made");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_filter_made),false);
  gtk_widget_set_sensitive(check_filter_made,false);

  gtk_box_pack_end(GTK_BOX(hbox_filters),check_filter_made,false, false, 0);


  TabAdd(hbox_filters);

  //boxes
  btn_box1 = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(btn_box1),GTK_BUTTONBOX_START);
  GtkWidget *hbox1 = gtk_hbox_new(false,0);
  GtkWidget *hbox2 = gtk_hbox_new(false,0);

  btn_box2 = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(btn_box2),GTK_BUTTONBOX_START);

  btn_box3 = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(btn_box3),GTK_BUTTONBOX_START);




  //buttons and other widgets

  btn_collect = gtk_button_new_with_label("Collect Training Data");
  g_signal_connect(G_OBJECT(btn_collect),
                   "clicked",
                   G_CALLBACK(CB_collectData),
                   (gpointer) this);
  spin_seconds = gtk_spin_button_new_with_range(1,30,1);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_seconds),num_seconds);
  g_signal_connect(G_OBJECT(spin_seconds),
                   "value-changed",
                   G_CALLBACK(CB_changeNumSeconds),
                   (gpointer) this);
  gtk_widget_set_size_request(spin_seconds, 40,25);
  

  btn_show = gtk_button_new_with_label("Show Components");
  spin_lags = gtk_spin_button_new_with_range(0,50,1);
  gtk_widget_set_size_request(spin_lags,40,25);

  g_signal_connect(G_OBJECT(spin_lags),
                   "value-changed",
                   G_CALLBACK(CB_changeNumLags),
                   (gpointer) this);

  g_signal_connect(G_OBJECT(btn_show),
                   "clicked",
                   G_CALLBACK(CB_showComponents),
                   (gpointer) this);


  entry_components = gtk_entry_new();
  g_signal_connect(G_OBJECT(entry_components),
                   "changed",
                   G_CALLBACK(CB_selectComponents),
                   (gpointer) this);


  label_selected = gtk_label_new("(not set)");
  gtk_label_set_max_width_chars(GTK_LABEL(label_selected),35);


  //create filter
  btn_create_filter = gtk_button_new_with_label("Create Filter");
  g_signal_connect(G_OBJECT(btn_create_filter),
                   "clicked",
                   G_CALLBACK(CB_createFilter),
                   (gpointer) this);

  //pack widgets
  gtk_box_pack_start(GTK_BOX(btn_box3),gtk_label_new("Seconds to collect: "), false, false, 0);
  gtk_box_pack_start(GTK_BOX(btn_box3),spin_seconds, false, false, 0);
  gtk_box_pack_start(GTK_BOX(btn_box3),btn_collect, false, false, 0);
  gtk_box_pack_start(GTK_BOX(hbox1),gtk_label_new("Lags: "), false, false, 0);
  gtk_box_pack_start(GTK_BOX(hbox1),spin_lags, false, false, 0);
  gtk_box_pack_start(GTK_BOX(hbox1),gtk_label_new("  "), false, false, 0);
  gtk_box_pack_start(GTK_BOX(hbox1),btn_show, false, false, 0);
  gtk_box_pack_start(GTK_BOX(hbox2),gtk_label_new("Select Components (e.g. 1,3,6:9): "), false, false, 0);
  gtk_box_pack_start(GTK_BOX(hbox2),entry_components, false, false, 0);
  gtk_box_pack_start(GTK_BOX(hbox2),gtk_label_new(" = "), false, false, 0);
  gtk_box_pack_start(GTK_BOX(hbox2),label_selected, false, false, 0);
  gtk_box_pack_start(GTK_BOX(btn_box2),btn_create_filter, false, false, 0);

  //put all boxes into one and add it to the tab
  GtkWidget * btn_boxes = gtk_vbox_new(false,0);
  gtk_box_pack_start(GTK_BOX(btn_boxes), btn_box3, false, false, 0);
  gtk_box_pack_start(GTK_BOX(btn_boxes), hbox1, false, false, 1);
  gtk_box_pack_start(GTK_BOX(btn_boxes), btn_box1, false, false, 0);
  gtk_box_pack_start(GTK_BOX(btn_boxes), hbox2, false, false, 0);
  gtk_box_pack_start(GTK_BOX(btn_boxes), btn_box2, false, false, 0);
  TabFrameAdd(btn_boxes,"Filter Configuration");

  //plot
  plot = NULL;

  gtk_widget_set_sensitive(btn_create_filter,false);

  this->updating_view = false;

  updateView();

  if(gtk_combo_box_get_active(GTK_COMBO_BOX(combo_filters))==-1)
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo_filters),0);
}


//update view from model

void TabFilter::updateView()
{

  //set flag to indicate process of updating view
  //in order to stop callbacks from triggering
  this->updating_view = true;


  //get information from model about filters
  CEBLModel * model = getView()->getModel();
  std::vector<string> names = model->filterGetNameList();
  std::vector<string> paths = model->filterGetPathList();
  string model_selected_filter = model->filterGetSelected();

  //if the model's filters have changed or the selected filter has changed
  //update the filter list
  if(names != this->filter_names 
     || this->selected_filter != model_selected_filter)
    {
      this->selected_filter = model_selected_filter;
      this->filter_names = names;
      for(int i = this->num_filters-1; i >= 0; i--)
        {
          gtk_combo_box_remove_text(GTK_COMBO_BOX(combo_filters),i);
        }
      this->num_filters = names.size();

      // add in all filters
      for(unsigned int i=0;i<names.size();i++)
        {
          gtk_combo_box_append_text(GTK_COMBO_BOX(combo_filters),
                                    names[i].c_str());
          if(names[i]==model_selected_filter)
            {
              gtk_combo_box_set_active(GTK_COMBO_BOX(combo_filters),i);
            }
        }
    }//end of conditional concerning whether update should occur

  //get number of lags
  int n = model->filterGetNumLags();
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_lags),n);

  //get components string
  string comps = model->filterGetSelectedComponentsString();
  gtk_label_set_text(GTK_LABEL(label_selected),comps.c_str());

  //set entry if it is different from the currently selected components
  string input = string(gtk_entry_get_text(GTK_ENTRY(entry_components)));
  model->filterSetSelectedComponentsString(input);
  string comps2 = model->filterGetSelectedComponentsString();
  if(comps != comps2)
    {
      comps = model->filterGetSelectedComponentsString();
      if(model->filterGetSelectedComponentsValid())
        {
          gtk_entry_set_text(GTK_ENTRY(entry_components),comps.c_str());
        }
    }

  //update sensitivity on create filter button, and filter trained box
  bool components_valid = model->filterGetSelectedComponentsValid();
  gtk_widget_set_sensitive(btn_create_filter,components_valid);

  bool filter_made = model->filterIsTrained();
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_filter_made),
                               filter_made);


  this->updating_view = false;

}

//update the model from the widgets
void TabFilter::updateModel()
{
  
}

//----------------------------------------------------------------------
// PLOTTING

//! Create the plot canvas
void TabFilter::createPlot(int num_channels)
{
  if(plot==NULL)
    {
      plot = new EEGMonitor(this, num_channels);
      plot->setZoom(10);
      TabAdd(*plot, true, true, 0);
    }
  else
    {
      plot->reInitPlot(num_channels);
      plot->setZoom(10);
    }
}

//----------------------------------------------------------------------
// COLLECTING

void TabFilter::startCollect()
{
  if(!this->is_collecting)
    {
      this->training_data.clear();
      this->training_collected = false;
      CEBLModel * model = this->getView()->getModel();
      try {
        model->dataStart();
      } catch(exception & e) {
        string msg = "There was an error starting the data source. \n("+string(e.what())+")";
        WidgetUtils::AlertError("Error Starting Data Source",msg);
        return;
      }
      this->is_collecting = true;
      //show wait box
      WidgetUtils::waitBoxShow("Collecting Data");
      WidgetUtils::waitBoxSetCancelAvailable(true);
      WidgetUtils::waitBoxSetProgress(0);
      int num_samples = 0;
      int display_step = 64;
      this->collect_timer.restart();
      double percent;
      while(this->is_collecting)
        {
          gtk_main_iteration();
          if(WidgetUtils::waitBoxCancelled())
            {
              gtk_main_iteration();
              try {
                model->dataStop();
              } catch(exception & e){}

              this->is_collecting = false;
            }
          else
            {
              try {
                this->training_data.append(model->dataReadAll());
              } catch(exception & e) {}
              int new_samples = training_data.size2();
              double elapsed = collect_timer.elapsed() / 1000.0;
              if(new_samples - num_samples > display_step)
                {
                  num_samples = new_samples;
                  percent = elapsed/num_seconds;
                  if(elapsed >= num_seconds)
                    {
                      this->is_collecting = false;
                      this->training_collected = true;
                      model->dataStop();
                      this->CB_showComponents(NULL,this);
                    }
                  else
                    {
                      stringstream msg;

                      msg << int(elapsed) << " of " << num_seconds << " seconds\n";
                      msg <<  num_samples << " samples";

                      WidgetUtils::waitBoxSetText(msg.str().c_str());
                      WidgetUtils::waitBoxSetProgress(percent);
                    }
                }
            }
        }
      WidgetUtils::waitBoxHide();
    }
}

void TabFilter::stopCollect()
{
  if(this->is_collecting)
    {

      this->is_collecting = false;
    }
}

//----------------------------------------------------------------------
// CALLBACKS


void TabFilter::CB_selectComponents(GtkWidget *w, gpointer data)
{
 TabFilter * filter = (TabFilter *) data;
  if(!filter->updating_view)
    {
      string input = string(gtk_entry_get_text(GTK_ENTRY(w)));
      filter->getView()->getModel()->filterSetSelectedComponentsString(input);

      filter->updateView();
    }
}


void TabFilter::CB_showComponents(GtkWidget *w, gpointer data)
{
  TabFilter * tab = (TabFilter*) data;
  CEBLModel * model = tab->getView()->getModel();
  if(!tab->training_collected || tab->is_collecting)
    {
      string msg = "Failed to extract components. Collect data first!";
      WidgetUtils::AlertError("Failed to extract components. ", msg);
      return;
    }


  try
    {
      WidgetUtils::waitBoxShow("Extracting components...",false);
      EEGData components = model->filterGetComponents(tab->training_data);
      tab->createPlot(components.size1());
      tab->plot->setNumDisplaySamples(components.numSamples());
      tab->plot->Plot(components);
      WidgetUtils::waitBoxHide();
    }
  catch(PluginException e)
    {
      WidgetUtils::waitBoxHide();
      string msg = "Failed to extract components. \n(" + string(e.what()) + ")";
      WidgetUtils::AlertError("Failed to extract components. ", msg);
    }
  catch(exception &e)
    {
      WidgetUtils::waitBoxHide();
      string msg = "Failed to extract components. \n(" + string(e.what()) + ")";
      WidgetUtils::AlertError("Failed to extract components. ", msg);
    }
}


void TabFilter::CB_createFilter(GtkWidget *w, gpointer data)
{
  TabFilter * tab = (TabFilter*) data;
  try
    {
      tab->getView()->getModel()->filterTrain(tab->training_data);
    }
  catch(exception &e)
    {
      string msg = "Failed to make filter. \n(" + string(e.what()) + ")";
      WidgetUtils::AlertError("Failed to make filter. ", msg);
    }
  tab->updateView();
}

void TabFilter::CB_changeFilter(GtkWidget *w, gpointer data)
{
  TabFilter * tab = (TabFilter*) data;
  //as long as the update view is not currently running,
  //set the model's selected filter to the selected filter
  //in combo box
  if(tab->num_filters > 0 && !tab->updating_view)
    {
      string filter = 
        gtk_combo_box_get_active_text(GTK_COMBO_BOX(tab->combo_filters));
      try
        {
          if(filter != tab->selected_filter)
            {
              // set the selected filter in the model
              tab->getView()->getModel()->filterSetSelected(filter);
            }
        }
      catch(...)
        {
          cerr << "Error selecting filter.\n";
        }
      tab->updateView();
    }
}

void TabFilter::CB_changeNumLags(GtkWidget *w, gpointer data)
{
  TabFilter * tab = (TabFilter*) data;
  if(!tab->updating_view)
    {
      int n = int(gtk_spin_button_get_value(GTK_SPIN_BUTTON(tab->spin_lags)));
      tab->getView()->getModel()->filterSetNumLags(n);
    }
}

void TabFilter::CB_changeNumSeconds(GtkWidget *w, gpointer data)
{
  TabFilter * tab = (TabFilter*) data;
  if(!tab->updating_view)
    {
      tab->num_seconds = int(gtk_spin_button_get_value(GTK_SPIN_BUTTON(tab->spin_seconds)));
    }
}


void TabFilter::CB_collectData(GtkWidget *w, gpointer data)
{
  TabFilter * tab = (TabFilter*) data;
  if(!tab->is_collecting)
    tab->startCollect();
}


