/*!
 * TabMonitor.cpp
 * \author Jeshua Bratman
 *
 */

#include "TabMonitor.hpp"
#include "DataSourceCombo.hpp"
#include "WidgetUtils.hpp"
#include "EEGMonitor.hpp"

//----------------------------------------------------------------------
// Constructors / Destructor

TabMonitor::~TabMonitor()
{
  if(plot!=NULL)
    delete plot;
}


//----------------------------------------------------------------------
// Create the GUI


void TabMonitor::CreateGUI()
{
  //add title
  GtkWidget *title = gtk_label_new("");
  gtk_label_set_markup(GTK_LABEL(title),"<big>Monitor</big>");
  TabAdd(title);
  TabAdd(gtk_hseparator_new());



  //set some member variables
  this->collecting_data = false;
  this->continue_plotting = false;
  this->passive_plot = false;

  //boxes
  GtkWidget * btn_box1 = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(btn_box1),GTK_BUTTONBOX_START);
  GtkWidget * btn_box2 = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(btn_box2),GTK_BUTTONBOX_START);
  GtkWidget *label_box = gtk_hbox_new(false,0);

  //create buttons
  btn_start = gtk_button_new_with_label("Start Monitoring");
  gtk_widget_set_size_request(btn_start,30,20);
  g_signal_connect(G_OBJECT(btn_start),
        	   "clicked",
        	   G_CALLBACK(CB_StartMonitor),
        	   (gpointer) this);

  btn_stop = gtk_button_new_with_label("Stop");
  gtk_widget_set_sensitive(btn_stop,false);
  g_signal_connect(G_OBJECT(btn_stop),
        	   "clicked",
        	   G_CALLBACK(CB_StopMonitor),
        	   (gpointer) this);

  btn_start_collecting = gtk_button_new_with_label("Start Collecting");
  g_signal_connect(G_OBJECT(btn_start_collecting),
        	   "clicked",
        	   G_CALLBACK(CB_StartCollect),
        	   (gpointer) this);


  check_filter = gtk_check_button_new_with_label("Enable Filter");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_filter),false);
  g_signal_connect(G_OBJECT(check_filter),
                   "toggled",
                   G_CALLBACK(CB_toggleFilter),
                   (gpointer) this);


  GtkWidget *samples_label = gtk_label_new("Samples Collected: ");
  label_collected_samples = gtk_label_new("0");


  gtk_box_pack_start(GTK_BOX(btn_box1),btn_start,false,false,2);
  gtk_box_pack_start(GTK_BOX(btn_box1),btn_start_collecting,false,false,2);
  gtk_box_pack_start(GTK_BOX(btn_box1),gtk_label_new("Data Source: "), false, false, 2);
  gtk_box_pack_start(GTK_BOX(btn_box1),getView()->getDataSource()->getCombo(),false,false,2);
  gtk_box_pack_start(GTK_BOX(btn_box2),btn_stop,false,false,2);
  gtk_box_pack_start(GTK_BOX(btn_box2),check_filter,false,false,2);
  gtk_box_pack_start(GTK_BOX(label_box),samples_label,false,false,2);
  gtk_box_pack_start(GTK_BOX(label_box),label_collected_samples,false,false,2);

  //pack button box into tab container
  GtkWidget * btn_boxes = gtk_vbox_new(false,0);
  gtk_box_pack_start(GTK_BOX(btn_boxes), btn_box1, false, false, 0);
  gtk_box_pack_start(GTK_BOX(btn_boxes), btn_box2, false, false, 0);
  TabFrameAdd(btn_boxes);

  TabAdd(label_box);

  //plot
  plot = NULL;
}



//----------------------------------------------------------------------
// EVENT GUI UPDATES

//! update the view from model
void TabMonitor::updateView()
{
  this->updating_view = true;
  // change button sensitivity based on stat of collection
  if(this->continue_plotting)
    {
      if(this->collecting_data)
        {
          gtk_widget_set_sensitive(btn_start,false);
          gtk_widget_set_sensitive(btn_stop,true);
          gtk_widget_set_sensitive(btn_start_collecting,false);
        }
      else
        {
          gtk_widget_set_sensitive(btn_start,false);
          gtk_widget_set_sensitive(btn_stop,true);
          gtk_widget_set_sensitive(btn_start_collecting,false);
        }
    }
  else
    {
      gtk_widget_set_sensitive(btn_start,true);
      gtk_widget_set_sensitive(btn_stop,false);
      gtk_widget_set_sensitive(btn_start_collecting,true);
    }

  int num_samples = getView()->getModel()->dataGetStoreNumSamples();
  gtk_label_set_text(GTK_LABEL(label_collected_samples),
        	     TextUtils::IntToString(num_samples).c_str());

  bool filter = getView()->getModel()->processGetFilterEnabled();
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_filter),filter);

  this->updating_view = false;
}


//update the model from the widgets
void TabMonitor::updateModel()
{

}

//! called when this tab is hidden
void TabMonitor::onHide()
{

  if(continue_plotting && plot->isDetached())
    {
      this->passive_plot = true;
      getView()->getModel()->dataSetStoreFlag(true);
      this->collecting_data = false;
      this->collecting_data = false;
      getView()->getModel()->dataStop();
      getView()->getStatusBar()->remove(status_id);
      updateView();
    }
  else
    {
      stopPlotting();
    }
}


//----------------------------------------------------------------------
// PLOTTING

//! Create the plot canvas
void TabMonitor::createPlot()
{
  if(plot==NULL)
    {
      plot = new EEGMonitor(this,this->getView()->getModel()->channelsGetNumEnabledChannels());
      plot->setLabels(this->getView()->getModel()->channelsGetEnabledNames());
      plot->reInitPlot(this->getView()->getModel()->channelsGetNumEnabledChannels());
      TabAdd(*plot, true, true, 0);
    }
  else
    {
      plot->setLabels(this->getView()->getModel()->channelsGetEnabledNames());
      plot->reInitPlot(this->getView()->getModel()->channelsGetNumEnabledChannels());

    }
}

bool TabMonitor::startDataSource()
{
  try
    {
      getView()->getModel()->dataStart();
    }
  catch(exception &e)
    {
      string msg = "There was an error starting the data source. \n("+string(e.what())+")";
      WidgetUtils::AlertError("Error Starting Data Source", msg);
      return false;
    }
  if(!getView()->getModel()->dataIsStarted())
    {
      WidgetUtils::AlertError("Error Starting Data Source", "There was an error starting the data source. Check device configuration.");
      return false;
    }
  else
    {
      return true;
    }
}

//! start model reading data and plotting data here
void TabMonitor::startPlotting()
{
  getView()->getModel()->dataClearStoredData();
  getView()->getModel()->dataSetStoreFlag(true);

  continue_plotting = true;

  //create the eeg plot
  this->createPlot();

  //push status
  if(this->collecting_data)
    this->status_id = getView()->getStatusBar()->push("Collecting Data");
  else
    this->status_id = getView()->getStatusBar()->push("Plotting Data");

  //start the timeout
  g_timeout_add(100, timedPlot, this);
  
  //update the widgets from model
  updateView();
}

gint TabMonitor::timedPlot(gpointer parent)
{
  TabMonitor * tab = (TabMonitor*)parent;

  //if window has been reattached, end timed plot
  if(tab->passive_plot && !tab->plot->isDetached())
    {
      tab->stopPlotting();
      return false;
    }
  //if continue plotting has been unset, end timed plot
  if(!tab->continuePlotting())
    {
      return false;
    }
  try
    {
      //read data from source. it will be put into the data buffer.
      if(!tab->passive_plot)
        tab->getView()->getModel()->dataReadAll();
      //! update data collection display
      int num_samples = tab->getView()->getModel()->dataGetStoreNumSamples();
      gtk_label_set_text(GTK_LABEL(tab->label_collected_samples),
                         TextUtils::IntToString(num_samples).c_str());

      //get the data buffer to plot
      EEGData data = tab->getView()->getModel()->dataGetStoredData();
      if(!tab->collecting_data)
        {
          data = tab->getView()->getModel()->processData(data);
        }
      ublas::matrix<double> d = (data.getMatrix());

      //! plot
      if(tab->plot!=NULL)
        {
          tab->plot->Plot(data);
        }
      else
        {
          tab->stopPlotting();
        }
      if(!tab->collecting_data)
        {
          tab->getView()->getModel()->dataClearStoredData();
        }
    }
  catch(DataExceptionUnderflow e)
    {
      cout << e.what() << "\n";
    }
  catch(DataException e)
    {
      cerr << e.what() << "\n";
      tab->stopPlotting();
    }
  catch(PluginException e)
    {
      string msg = "Failed to process data. \n(" + string(e.what()) + ")";
      WidgetUtils::AlertError("Error Processing Data",msg);
      tab->stopPlotting();
    }
  catch(exception e)
    {
      string msg = "Failed to process data. \n(" + string(e.what()) + ")";
      WidgetUtils::AlertError("Error Processing Data",msg);
      tab->stopPlotting();
    }
  return true;
}

//! stop plotting
void TabMonitor::stopPlotting()
{
  if(!this->passive_plot)
    stopDataSource();

  this->continue_plotting = false;
  this->collecting_data = false;
  this->passive_plot = false;
  getView()->getModel()->dataStop();
  getView()->getStatusBar()->remove(status_id);
  updateView();
}

//! stop the data source
void TabMonitor::stopDataSource()
{
  getView()->getModel()->dataSetStoreFlag(false);
  getView()->getModel()->dataStop();
}


//----------------------------------------------------------------------
// CALLBACKS


void TabMonitor::CB_StartMonitor(GtkWidget *w, gpointer data)
{
  TabMonitor * tab = (TabMonitor*)data;
  if(!tab->passive_plot)
    {
      if(tab->startDataSource())
        tab->startPlotting();
    }
  else
    {
      tab->startPlotting();
    }
}
void TabMonitor::CB_StopMonitor(GtkWidget *w, gpointer data)
{
  TabMonitor * tab = (TabMonitor*)data;
  if(!tab->passive_plot)
    tab->stopDataSource();
  tab->stopPlotting();
}
void TabMonitor::CB_StartCollect(GtkWidget *w, gpointer data)
{
  TabMonitor * tab = (TabMonitor*)data;
  tab->collecting_data = true;
  tab->CB_StartMonitor(w,data);
}


void TabMonitor::CB_toggleFilter(GtkWidget *w, gpointer data)
{
  TabMonitor *tab = (TabMonitor*)(data);
  //return if callback is suppressed
  if(!tab->updating_view)
    {
      bool checked = 
        gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tab->check_filter));
      tab->getView()->getModel()->processSetFilterEnabled(checked);
    }
}
