/*!
 * TabRealTimeClassification.hpp
 * \author Jeshua Bratman
 *
 * Tab for classifying real-time data and controlling interface.
 */


#include "TabRealTimeClassification.hpp"
#include "InterfaceCombo.hpp"
#include "InterfaceConfigurationWindow.hpp"
#include "interfaces/KeyboardPie.hpp"
#include "interfaces/RobotPie.hpp"
#include "DataSourceCombo.hpp"
#include "cppR.hpp"

//----------------------------------------------------------------------
// Constructors / Destructor

TabRealTimeClassification::~TabRealTimeClassification()
{
  delete this->interface_config;
}


//----------------------------------------------------------------------
// Create the GUI


void TabRealTimeClassification::CreateGUI()
{
  //add title
  GtkWidget *title = gtk_label_new("");
  gtk_label_set_markup(GTK_LABEL(title),"<big>Real Time Classification</big>");
  TabAdd(title);
  TabAdd(gtk_hseparator_new());



  //-------------------------------------------------

  this->interface_box = gtk_vbox_new(false, 0);
  this->updating_view = false;
  this->interface_config = new InterfaceConfigurationWindow(this->view);

  //----------------------------------------
  //create buttons

  btn_train_classifier = gtk_button_new_with_label("Train Classifier");
  g_signal_connect(G_OBJECT(btn_train_classifier),
      "clicked",
      G_CALLBACK(CB_trainClassifier),
      (gpointer) this);

  btn_start = gtk_button_new_with_label("Start Classifying");
  g_signal_connect(G_OBJECT(btn_start),
      "clicked",
      G_CALLBACK(CB_start),
      (gpointer) this);

  btn_stop = gtk_button_new_with_label("Stop Classifying");
  gtk_widget_set_sensitive(btn_stop,false);
  g_signal_connect(G_OBJECT(btn_stop),
      "clicked",
      G_CALLBACK(CB_stop),
      (gpointer) this);

  //----------------------------------------
  //create boxes
  GtkWidget * controls_hbox1 = gtk_hbox_new(false,0);
  GtkWidget * controls_hbox2 = gtk_hbox_new(false,0);
  GtkWidget * btn_box1 = gtk_hbutton_box_new();
  GtkWidget * btn_box2 = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(btn_box1),GTK_BUTTONBOX_START);
  gtk_button_box_set_layout(GTK_BUTTON_BOX(btn_box2),GTK_BUTTONBOX_START);


  //----------------------------------------
  //pack everything into boxes
  gtk_box_pack_start(GTK_BOX(btn_box1),btn_train_classifier,false,false,2);
  gtk_box_pack_start(GTK_BOX(controls_hbox1),btn_box1,false,false,2);
  source_combo = getView()->getDataSource()->getCombo();
  gtk_box_pack_end(GTK_BOX(controls_hbox1),source_combo,false,true,2);
  gtk_box_pack_end(GTK_BOX(controls_hbox1),gtk_label_new("Data Source: "),false,false,2);

  gtk_box_pack_start(GTK_BOX(btn_box2),btn_start,false,false,2);
  gtk_box_pack_start(GTK_BOX(btn_box2),btn_stop,false,false,2);
  gtk_box_pack_start(GTK_BOX(controls_hbox2),btn_box2,false,false,2);
  //----------------------------------------
  //add everything to tab
  controls_box = gtk_vbox_new(false,0);
  gtk_box_pack_start(GTK_BOX(controls_box), controls_hbox1, true, true, 2);
  gtk_box_pack_start(GTK_BOX(controls_box), controls_hbox2, true, true, 2);


  // Training/Using Interfaces

  //interface combo box
  this->interface_combo = this->view->getInterfaceCombo()->getCombo();
  gtk_box_pack_end(GTK_BOX(controls_hbox2),this->interface_combo,false,false,0);
  this->btn_interface_cfg = this->interface_config->getButton();
  gtk_box_pack_end(GTK_BOX(controls_hbox2),btn_interface_cfg,false,false,0);
  gtk_widget_show_all(controls_hbox2);

  this->interface = NULL;

  gtk_widget_set_size_request(controls_box,700,-1);
  TabFrameAdd(controls_box, "Classification Config");
  TabAdd(interface_box,true,true,0);
}



//update view from model
void TabRealTimeClassification::updateView()
{
  this->updating_view = true;


  //----------------------------------------
  //update interface
  //get a pointer to the selected interface
  EEGInterface* new_interface = this->view->getInterfaceCombo()->getInterface();

  //if the interface has changed, remove old interface and pack in new one
  if(new_interface != this->interface)
    {
    if(this->interface != NULL)
      {
      this->interface->hide();
      gtk_container_remove(GTK_CONTAINER(this->interface_box),this->interface->getContainer());
      }

    this->interface = new_interface;
    if(this->interface != NULL)
      {
      gtk_box_pack_start(GTK_BOX(this->interface_box),this->interface->getContainer(),true,true,0);
      interface->setBGColor(this->getView()->getBGRED(),
          this->getView()->getBGGREEN(),
          this->getView()->getBGBLUE());
      this->interface->show();
      this->interface->setUseMode();
      }
    }

  this->updating_view = false;
  updateInterface();
}
//----------------------------------------
//update the model from the widgets
void TabRealTimeClassification::updateModel()
{

}

//----------------------------------------

void TabRealTimeClassification::updateInterface()
{
  if(!updating_view)
    {
    int n = this->getView()->getModel()->trainingGetNumClasses();
    interface->setNumClasses(n);
    }
}

//----------------------------------------

gint TabRealTimeClassification::timedUpdateInterface(gpointer parent)
{
  TabRealTimeClassification * tab = (TabRealTimeClassification*)parent;
  CEBLModel * model = tab->getView()->getModel();

  //ERROR CHECKING
  bool stopped = !model->realtimeIsClassifying();
  if(stopped)
    {
    tab->CB_stop(NULL,tab);
    tab->getView()->updateInfoBar();
    return false;
    }
  else //classifying not stopped
    {
    if(tab->interface==NULL)
      {
      tab->CB_stop(NULL,tab);
      tab->getView()->updateInfoBar();
      return false;
      }
    else
      {
      int selected_class = model->realtimeGetSelectedClass();
      if(selected_class >= 0)
        {
        tab->interface->selectClass(selected_class);
        model->realtimeClearSelectedClass();
        }
      else
        {
        std::vector<double> std_props =
            model->realtimeGetClassProportions();

        //nan entries, set to even split
        if(!(std_props[0] >= 0 || std_props[0] < 0)){
          for(unsigned int i = 0; i<std_props.size();i++){
            std_props[i]=1.0/std_props.size();
          }
        }

        ublas::vector<double> proportions =
            cppR::asUblasVector(std_props);
        if(proportions.size() > 0)
          {
          tab->interface->setClassProportions(std_props);
          }
        }
      }
    }
  tab->getView()->updateInfoBar();
  return true;
}
//----------------------------------------
// Enable and disable controls
void TabRealTimeClassification::disableControls()
{
  gtk_widget_set_sensitive(btn_train_classifier,false);
  gtk_widget_set_sensitive(btn_stop,true);
  gtk_widget_set_sensitive(btn_start,false);
  gtk_widget_set_sensitive(interface_combo,false);
  gtk_widget_set_sensitive(source_combo,false);
  gtk_widget_set_sensitive(btn_interface_cfg,false);
}

void TabRealTimeClassification::enableControls()
{
  gtk_widget_set_sensitive(btn_train_classifier,true);
  gtk_widget_set_sensitive(btn_stop,false);
  gtk_widget_set_sensitive(btn_start,true);
  gtk_widget_set_sensitive(interface_combo,true);
  gtk_widget_set_sensitive(source_combo,true);
  gtk_widget_set_sensitive(btn_interface_cfg,true);
}

//----------------------------------------
// when the tab becomes hidden, this function is run

void TabRealTimeClassification::onHide()
{
  this->CB_stop(NULL,this);
  if(this->interface != NULL)
    {
    this->interface->hide();
    gtk_container_remove(GTK_CONTAINER(this->interface_box),this->interface->getContainer());
    }
  this->interface = NULL;
  this->enableControls();
  this->view->updateInfoBar();
}


//----------------------------------------------------------------------
// CALLBACKS

void TabRealTimeClassification::CB_trainClassifier(GtkWidget *w, gpointer data)
{
  TabRealTimeClassification * tab = (TabRealTimeClassification*)data;
  CEBLModel * model = tab->view->getModel();
  if(!model->trainingDataIsLoaded())
    {
    WidgetUtils::AlertError("No Training Data", "There is no training data available. Either load data in training tab or perform a training session.");
    }
  //otherwise data is loaded so we can train the classifier
  else
    {
    WidgetUtils::waitBoxShow("Training classifier");
    WidgetUtils::waitBoxSetCancelAvailable(true);
    model->realtimeTrainClassifierThreaded();
    while(model->realtimeIsTrainingClassifier())
      {
      gtk_main_iteration();
      if(WidgetUtils::waitBoxCancelled())
        {
        WidgetUtils::waitBoxSetText("Attempting to stop training.\n(This may take awhile.)");
        WidgetUtils::waitBoxSetCancelAvailable(false);
        gtk_main_iteration();
        model->realtimeTrainClassifierHalt();
        }
      }
    WidgetUtils::waitBoxHide();
    }
  tab->getView()->updateInfoBar();
}


void TabRealTimeClassification::CB_start(GtkWidget *w, gpointer data)
{
  TabRealTimeClassification * tab = (TabRealTimeClassification*)data;
  CEBLModel * model = tab->view->getModel();

  int nClasses = model->trainingGetNumClasses();
  int tClasses = model->classifierGetTrainedClasses();

  /* lags not currently implemented
  int nLags = model->featureGetParams()["lags"].getInt();
  int tLags = model->classifierGetTrainedLags();
  */
  if(!model->realtimeIsReady())
    {
    WidgetUtils::AlertError("Cannot Start Classifciation", "Real-time classification not ready. Make sure you have trained the selected classifier.");
    }
  else if(nClasses > tClasses)
    {
    std::ostringstream warning;
    warning << "Real-time classification not ready. Classifier is trained for " << tClasses<< " classes, " << nClasses << " classes selected.";
    WidgetUtils::AlertError("Cannot Start Classifciation", warning.str() );
    }
  /* lags not currently implemented
  else if (nLags != tLags)
    {
    std::ostringstream warning;
    warning << "Real-time classification not ready. Classifier is trained for " << tLags << " lags, " << nLags << " lags selected.";
    WidgetUtils::AlertError("Cannot Start Classifciation", warning.str());
    }
  */
  //otherwise classifier is trained on current settings
  else
    {
    if(nClasses<tClasses)
      {
      cout<<"Classifier trained for " << tClasses << ". Only the first " << nClasses <<" will be used in classification."<<endl;
      }

    try
    {
      model->realtimeStartClassifying();
    }
    catch(exception & e)
    {
      string msg = "There was an error starting the classification process.\n(" + string(e.what()) + "). Make sure you have installed the most recent classifier plugins.";
      WidgetUtils::AlertError("Error Starting Classification Process", msg);
      return;
    }
    tab->disableControls();
    g_timeout_add(150, timedUpdateInterface, tab);
    }
}

void TabRealTimeClassification::CB_stop(GtkWidget *w, gpointer data)
{
  TabRealTimeClassification * tab = (TabRealTimeClassification*)data;
  tab->enableControls();
  tab->view->getModel()->realtimeStopClassifying();
  if(tab->interface != NULL)
    tab->interface->selectTrainingClass(-1);

}


