/*!
 * TabTraining.cpp
 * \author Jeshua Bratman
 *
 */

#include "TabTraining.hpp"
#include "InterfaceCombo.hpp"
#include "InterfaceConfigurationWindow.hpp"
#include "interfaces/KeyboardPie.hpp"
#include "interfaces/RobotPie.hpp"
#include "DataSourceCombo.hpp"

//----------------------------------------------------------------------
// Constructors / Destructor

TabTraining::~TabTraining()
{
  delete this->interface_config;
}


//----------------------------------------------------------------------
// Create the GUI


void TabTraining::CreateGUI()
{
  //add title
  GtkWidget *title = gtk_label_new("");
  gtk_label_set_markup(GTK_LABEL(title),view->getString("TrainingTitle"));
  TabAdd(title);
  TabAdd(gtk_hseparator_new());

  //-------------------------------------------------

  this->interface_box = gtk_vbox_new(false, 0);
  this->updating_view = false;
  this->training_stopped_manually = false;
  this->interface_config = new InterfaceConfigurationWindow(this->view);

  //----------------------------------------
  //create buttons


  btn_start = gtk_button_new_with_label(view->getString("TrainingStr1"));
  g_signal_connect(G_OBJECT(btn_start),
                   "clicked",
                   G_CALLBACK(CB_startTraining),
                   (gpointer) this);

  btn_stop = gtk_button_new_with_label(view->getString("TrainingStr2"));
  gtk_widget_set_sensitive(btn_stop,false);
  g_signal_connect(G_OBJECT(btn_stop),
                   "clicked",
                   G_CALLBACK(CB_stopTraining),
                   (gpointer) this);

  spin_classes = gtk_spin_button_new_with_range(2,50,1);
  gtk_widget_set_size_request(spin_classes,40,25);

  g_signal_connect(G_OBJECT(spin_classes),
                   "value-changed",
                   G_CALLBACK(CB_changeNumClasses),
                   (gpointer) this);

  spin_sequences = gtk_spin_button_new_with_range(1,50,1);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_sequences),3);
  gtk_widget_set_size_request(spin_sequences,40,25);
  g_signal_connect(G_OBJECT(spin_sequences),
                   "value-changed",
                   G_CALLBACK(CB_changeNumSequences),
                   (gpointer) this);

  spin_sequence_length = gtk_spin_button_new_with_range(1,50,1);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_sequence_length),3);
  gtk_widget_set_size_request(spin_sequence_length,40,25);
  g_signal_connect(G_OBJECT(spin_sequence_length),
                   "value-changed",
                   G_CALLBACK(CB_changeSequenceLength),
                   (gpointer) this);

  spin_pause_length = gtk_spin_button_new_with_range(1,50,1);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_pause_length),3);
  gtk_widget_set_size_request(spin_pause_length,40,25);
  g_signal_connect(G_OBJECT(spin_pause_length),
                   "value-changed",
                   G_CALLBACK(CB_changePauseLength),
                   (gpointer) this);

  btn_load = gtk_button_new_with_label(view->getString("TrainingStr3"));
  g_signal_connect(G_OBJECT(btn_load),
                   "clicked",
                   G_CALLBACK(CB_loadData),
                   (gpointer) this);

  btn_save = gtk_button_new_with_label(view->getString("TrainingStr4"));
  g_signal_connect(G_OBJECT(btn_save),
                   "clicked",
                   G_CALLBACK(CB_saveData),
                   (gpointer) this);


  info_frame1 = gtk_viewport_new(NULL,NULL);
  //  gtk_container_border_width(GTK_CONTAINER(info_frame),1);
  label_info = gtk_label_new("");

  info_frame2 = gtk_viewport_new(NULL,NULL);
  //  gtk_container_border_width(GTK_CONTAINER(info_frame),1);
  label_data = gtk_label_new("");

  //----------------------------------------
  //create boxes
  controls_hbox1 = gtk_hbox_new(false,0);
  controls_hbox2 = gtk_hbox_new(false,0);
  controls_hbox3 = gtk_hbox_new(false,0);
  controls_hbox4 = gtk_hbox_new(false,0);
  GtkWidget * btn_box1 = gtk_hbutton_box_new();
  GtkWidget * btn_box2 = gtk_hbutton_box_new();
  GtkWidget * btn_box3 = gtk_hbutton_box_new();
  GtkWidget * btn_box4 = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(btn_box1),GTK_BUTTONBOX_START);
  gtk_button_box_set_layout(GTK_BUTTON_BOX(btn_box2),GTK_BUTTONBOX_START);
  gtk_button_box_set_layout(GTK_BUTTON_BOX(btn_box3),GTK_BUTTONBOX_START);
  gtk_button_box_set_layout(GTK_BUTTON_BOX(btn_box4),GTK_BUTTONBOX_START);



  //----------------------------------------
  //pack everything into boxes
  gtk_box_pack_start(GTK_BOX(btn_box1),btn_start,false,false,2);
  gtk_box_pack_start(GTK_BOX(btn_box1),btn_stop,false,false,2);
  gtk_box_pack_start(GTK_BOX(controls_hbox1),btn_box1,false,false,2);
  source_combo = getView()->getDataSource()->getCombo();
  gtk_box_pack_end(GTK_BOX(controls_hbox1),source_combo,false,true,2);
  gtk_box_pack_end(GTK_BOX(controls_hbox1),
                   gtk_label_new(view->getString("TrainingStr5")),
                   false,false,2);

  gtk_box_pack_start(GTK_BOX(controls_hbox2),spin_classes,false,false,2);
  gtk_box_pack_start(GTK_BOX(controls_hbox2),
                     gtk_label_new(view->getString("TrainingStr6")),
                     false,false,2);
  gtk_box_pack_start(GTK_BOX(controls_hbox2),spin_sequences,false,false,2);
  gtk_box_pack_start(GTK_BOX(controls_hbox2),
                     gtk_label_new(view->getString("TrainingStr7")),
                     false,false,2);
  gtk_box_pack_start(GTK_BOX(controls_hbox2),
                     spin_sequence_length,false,false,2);
  gtk_box_pack_start(GTK_BOX(controls_hbox2),
                     gtk_label_new(view->getString("TrainingStr8")),
                     false,false,2);
  gtk_box_pack_start(GTK_BOX(controls_hbox2),
                     spin_pause_length,false,false,2);
  gtk_box_pack_start(GTK_BOX(controls_hbox2),
                     gtk_label_new(view->getString("TrainingStr9")),
                     false,false,2);
  gtk_container_add(GTK_CONTAINER(info_frame1),label_info);

  gtk_box_pack_start(GTK_BOX(btn_box3),btn_load,false,false,2);
  gtk_box_pack_start(GTK_BOX(btn_box3),btn_save,false,false,2);
  gtk_box_pack_start(GTK_BOX(controls_hbox3),btn_box3,false,false,2);
  gtk_container_add(GTK_CONTAINER(info_frame2),label_data);


  //----------------------------------------
  // classification feedback controls
  check_classification_feedback = 
    gtk_check_button_new_with_label("Classification Feedback");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_classification_feedback),
                               false);
  gtk_box_pack_start(GTK_BOX(btn_box4),check_classification_feedback,
                     false,false,2);
  g_signal_connect(G_OBJECT(check_classification_feedback),
                   "toggled",
                   G_CALLBACK(CB_toggleClassificationFeedback),
                   (gpointer) this);

  gtk_box_pack_start(GTK_BOX(controls_hbox4),btn_box4,false,false,2);

  //----------------------------------------
  //add everything to tab
  controls_box = gtk_vbox_new(false,0);
  gtk_box_pack_start(GTK_BOX(controls_box), controls_hbox1, true, true, 2);
  gtk_box_pack_start(GTK_BOX(controls_box), controls_hbox2, true, true, 2);
  gtk_box_pack_start(GTK_BOX(controls_box), info_frame1, true, true, 4);
  gtk_box_pack_start(GTK_BOX(controls_box), controls_hbox3, true, true, 2);
  gtk_box_pack_start(GTK_BOX(controls_box), info_frame2, true, true, 4);
  gtk_box_pack_start(GTK_BOX(controls_box), controls_hbox4, true, true, 2);


  //--------------------------------------------------
  // Training/Using Interfaces
  // because training interfaces are totally view-side,
  // the selection of the training interface will be handled
  // independently from the model

  //interface combo box
  this->interface_combo = this->view->getInterfaceCombo()->getCombo();
  gtk_box_pack_end(GTK_BOX(controls_hbox3),this->interface_combo,false,false,0);
  this->btn_interface_cfg = this->interface_config->getButton();
  gtk_box_pack_end(GTK_BOX(controls_hbox3),btn_interface_cfg,false,false,0);
  gtk_widget_show_all(controls_hbox3);

  this->interface = NULL;


  gtk_widget_set_size_request(controls_box,700,-1);
  TabFrameAdd(controls_box, view->getString("TrainingStr10"));
  TabAdd(interface_box,true,true,0);

}



//update view from model
void TabTraining::updateView()
{
  this->updating_view = true;
  CEBLModel * model = this->getView()->getModel();
  //----------------------------------------
  //update training options
  int num_classes = model->trainingGetNumClasses();
  int num_sequences = model->trainingGetNumSequences();
  int sequence_length = model->trainingGetSequenceLength();
  int pause_length = model->trainingGetPauseLength();
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_classes),num_classes);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_sequences),num_sequences);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_sequence_length),
                            sequence_length);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_pause_length),pause_length);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_classification_feedback),
                               model->trainingFeedbackEnabled());

  //----------------------------------------
  // update data label
  bool data_loaded = model->trainingDataIsLoaded();
  string text = "";
  if(data_loaded)
    {
      EEGTrainingData data = model->trainingGetData();
      int num_classes = data.numClasses();
      int num_sequences = data.numSequences();
      int num_samples = data.numSamples();
      if(model->trainingIsDataFileLoaded())
        {
          //indicate filename of saved data
          text = view->getString("TrainingStr29") + model->trainingGetDataFilename() + ", ";
        }
      else
        {
          //indicate unsaved data
          text = view->getString("TrainingStr30");
        }
      text += view->getString("TrainingStr31") 
        + TextUtils::IntToString(num_classes)
        + view->getString("TrainingStr32") 
        + TextUtils::IntToString(num_sequences)
        + view->getString("TrainingStr33")
        + TextUtils::IntToString(num_samples);

    }
  else
    {
      text = view->getString("TrainingStr34");
    }

  text = "<small>" + text + "</small>";
  gtk_label_set_markup(GTK_LABEL(this->label_data),
                       text.c_str());


  //----------------------------------------
  //update interface
  //get a pointer to the selected interface
  EEGInterface* new_interface = this->view->getInterfaceCombo()->getInterface();


  //if the interface has changed, remove old interface and pack in new one
  if(new_interface != this->interface)
    {
      if(this->interface != NULL)
        {
          this->interface->hide();
          gtk_container_remove(GTK_CONTAINER(this->interface_box),
                               this->interface->getContainer());
        }

      this->interface = new_interface;
      if(this->interface != NULL)
        {
          gtk_box_pack_start(GTK_BOX(this->interface_box),
                             this->interface->getContainer(),true,true,0);
          interface->setBGColor(this->getView()->getBGRED(),
                                this->getView()->getBGGREEN(),
                                this->getView()->getBGBLUE());
          this->interface->show();
        }
    }

  this->updating_view = false;
  updateInterface();
  updateTextInfo();

}

//----------------------------------------

void TabTraining::updateInterface()
{
  if(!updating_view)
    {
      int n = int(gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin_classes)));
      interface->setNumClasses(n);
    }
}


//----------------------------------------

void TabTraining::updateTextInfo()
{
  int num_classes = this->getView()->getModel()->trainingGetNumClasses();
  int num_sequences = this->getView()->getModel()->trainingGetNumSequences();
  int sequence_length = this->getView()->getModel()->trainingGetSequenceLength();
  int pause_length = this->getView()->getModel()->trainingGetPauseLength();

  int total_data_time = num_classes * num_sequences * sequence_length;
  int total_time = total_data_time + pause_length * num_sequences * num_classes;
  stringstream ss;
  int minutes = total_data_time/60;
  int seconds = total_data_time % 60;
  int minutes2 = total_time/60;
  int seconds2 = total_time % 60;

  if(total_time > 60)
    {
      ss << view->getString("TrainingStr11") << minutes2 << view->getString("TrainingStr12") << seconds2 << view->getString("TrainingStr13")
         << minutes << view->getString("TrainingStr14") << seconds << view->getString("TrainingStr15");
    }
  else
    {
      ss << view->getString("TrainingStr16") << total_time << view->getString("TrainingStr17")
         << total_data_time << view->getString("TrainingStr18");
    }
  gtk_label_set_markup(GTK_LABEL(label_info),ss.str().c_str());
  gtk_widget_set_size_request(label_info,300,-1);
}


//----------------------------------------

gint TabTraining::timedUpdateInterface(gpointer parent)
{
  TabTraining * tab = (TabTraining*)parent;
  CEBLModel * model = tab->getView()->getModel();
  CEBLViewGTK * view = tab->getView();

  //ERROR CHECKING
  bool failed = model->trainingFailed();
  if(failed)
    {
      if(!tab->training_stopped_manually)
        WidgetUtils::AlertError(view->getString("TrainingStr19"),model->trainingGetFailureMessage());
      if(tab->interface!=NULL)
        tab->interface->selectTrainingClass(-1);
      tab->enableControls();
      tab->updateView();
      tab->interface->setTrainMode();
      return false;
    }

  //update interface
  bool paused = model->trainingIsPaused();
  bool training_classifier = model->trainingIsTrainingClassifier();
  bool classification_feedback = model->trainingFeedbackEnabled();
  if(!training_classifier)
    {
      WidgetUtils::waitBoxHide();
    }
  
  if(training_classifier)
    {
      WidgetUtils::waitBoxShow("Training classifier");
      if(!WidgetUtils::waitBoxCancelled())
        {
          WidgetUtils::waitBoxSetCancelAvailable(true);
        }
      else
        {
          WidgetUtils::waitBoxSetText("Attempting to stop training.\n(This may take awhile.)");
          WidgetUtils::waitBoxSetCancelAvailable(false);
          model->realtimeTrainClassifierHalt();
        }
    }
  else if(paused)
    {
      tab->interface->setTrainMode();
      int cls = model->trainingGetTrainingClass();
      int seq = model->trainingGetTrainingSequence();

      tab->interface->selectTrainingClass(cls);

      tab->view->getStatusBar()->remove(tab->status_id);
      tab->status_id = tab->view->getStatusBar()->push(string(view->getString("TrainingStr20"))
                                                       + TextUtils::IntToString(cls) +
                                                       view->getString("TrainingStr21")
                                                       + TextUtils::IntToString(seq) + ".");
    }
  else
    {
      tab->interface->selectTrainingClass(-1);

      //if we are providing feedback, draw the bars
      if(classification_feedback)
        {
          tab->interface->setUseMode();
          std::vector<double> std_props = 
            model->trainingGetClassProportions();
          ublas::vector<double> proportions = 
            cppR::asUblasVector(std_props);
          if(proportions.size() > 0)
            {
              tab->interface->setClassProportions(std_props);
            }
        }
    }

  if(tab->interface==NULL)
    {
      tab->enableControls();
      return false;
    }

  //end training successfully
  if(!model->trainingIsActive())
    {
      WidgetUtils::Alert(view->getString("TrainingStr22"),view->getString("TrainingStr23"), GTK_MESSAGE_INFO);
      if(tab->interface!=NULL)
        tab->interface->selectTrainingClass(-1);
      tab->enableControls();
      model->trainingStop();
      tab->updateView();
      tab->getView()->updateInfoBar();
      return false;
    }
  else
    {
      return true;
    }
}

//----------------------------------------
// Enable and disable controls
void TabTraining::disableControls()
{
  gtk_widget_set_sensitive(btn_stop,true);
  gtk_widget_set_sensitive(btn_start,false);
  gtk_widget_set_sensitive(source_combo,false);
  gtk_widget_hide(controls_hbox2);
  gtk_widget_hide(controls_hbox3);
  gtk_widget_hide(controls_hbox4);
  gtk_widget_hide(info_frame2);
}

void TabTraining::enableControls()
{
  gtk_widget_set_sensitive(btn_stop,false);
  gtk_widget_set_sensitive(btn_start,true);
  gtk_widget_set_sensitive(source_combo,true);
  gtk_widget_show(controls_hbox2);
  gtk_widget_show(controls_hbox3);
  gtk_widget_show(controls_hbox4);
  gtk_widget_show(info_frame2);
  view->getStatusBar()->remove(status_id);
}

//----------------------------------------

//update the model from the widgets
void TabTraining::updateModel()
{

}

// when the tab becomes hidden, this function is run
void TabTraining::onHide()
{
  this->CB_stopTraining(NULL,this);
  if(this->interface != NULL)
    {
      this->interface->hide();
      gtk_container_remove(GTK_CONTAINER(this->interface_box),this->interface->getContainer());
    }
  this->interface = NULL;
  this->enableControls();
  view->updateInfoBar();
  CEBL::Param param = CEBL::Param("classes","classes", view->getModel()->trainingGetNumClasses());
  view->getModel()->classifierReset(param);
}


//----------------------------------------------------------------------
// CALLBACKS

void TabTraining::CB_startTraining(GtkWidget *w, gpointer data)
{
  TabTraining * tab = (TabTraining*)data;
  CEBLViewGTK * view = tab->getView();
  try
    {
      tab->getView()->getModel()->trainingStart();
    }
  catch(exception & e)
    {
      string msg = view->getString("TrainingStr24") + string(e.what()) + ")";
      WidgetUtils::AlertError(view->getString("TrainingStr25"), msg);
      return;
    }
  tab->training_stopped_manually = false;
  tab->disableControls();
  g_timeout_add(50, timedUpdateInterface, tab);
}

void TabTraining::CB_stopTraining(GtkWidget *w, gpointer data)
{
  TabTraining * tab = (TabTraining*)data;
  tab->getView()->getModel()->trainingStop();
  if(tab->interface != NULL)
    tab->interface->selectTrainingClass(-1);
  tab->training_stopped_manually = true;
}

void TabTraining::CB_changeNumClasses(GtkWidget *w, gpointer data)
{
  TabTraining * tab = (TabTraining*)data;
  if(!tab->updating_view)
    {
      int n = int(gtk_spin_button_get_value(GTK_SPIN_BUTTON(tab->spin_classes)));
      tab->getView()->getModel()->trainingSetNumClasses(n);
      tab->updateInterface();
      tab->updateTextInfo();
      tab->getView()->updateInfoBar();
    }
}
void TabTraining::CB_changeNumSequences(GtkWidget *w, gpointer data)
{
  TabTraining * tab = (TabTraining*)data;
  if(!tab->updating_view)
    {
      int n = int(gtk_spin_button_get_value(GTK_SPIN_BUTTON(tab->spin_sequences)));
      tab->getView()->getModel()->trainingSetNumSequences(n);
      tab->updateTextInfo();
     }
}
void TabTraining::CB_changeSequenceLength(GtkWidget *w, gpointer data)
{
  TabTraining * tab = (TabTraining*)data;
  if(!tab->updating_view)
    {
      int n = int(gtk_spin_button_get_value(GTK_SPIN_BUTTON(tab->spin_sequence_length)));
      tab->getView()->getModel()->trainingSetSequenceLength(n);
      tab->updateTextInfo();
    }
}


void TabTraining::CB_changePauseLength(GtkWidget *w, gpointer data)
{
  TabTraining * tab = (TabTraining*)data;
  if(!tab->updating_view)
    {
      int n = int(gtk_spin_button_get_value(GTK_SPIN_BUTTON(tab->spin_pause_length)));
      tab->getView()->getModel()->trainingSetPauseLength(n);
      tab->updateTextInfo();
    }
}



void TabTraining::CB_loadData(GtkWidget *w, gpointer data)
{
  TabTraining * tab = (TabTraining*)data;
  CEBLViewGTK * view = tab->getView();

  if(tab->getView()->getModel()->trainingDataIsLoaded())
    {
        
      if(!WidgetUtils::Confirm("Replace Data?", "You already have training data loaded. If you load training data, any unsaved training data will be lost."))
        return;
    }
  string filename = WidgetUtils::selectLoadFile();
  if(filename != "")
    {
      view->getModel()->trainingLoadData(filename);
    }
  tab->updateView();
}

void TabTraining::CB_saveData(GtkWidget *w, gpointer data)
{
  TabTraining * tab = (TabTraining*)data;
  CEBLViewGTK * view = tab->getView();
  if(tab->getView()->getModel()->trainingDataIsLoaded())
    {
      string filename = WidgetUtils::selectSaveFile("tar.bz2");
      if(filename != "")
        {
          try
            {
              tab->getView()->getModel()->trainingSaveData(filename);
            }
          catch(exception &e)
            {
              string msg = "Failed to save training data: " + string(e.what());
              WidgetUtils::AlertError("Failed to save data.",msg);
            }
        }
    }
  else
    {
      WidgetUtils::AlertError(view->getString("TrainingStr26"),view->getString("TrainingStr27"));
    }

}




//update all the channel values in the model from gui
void TabTraining::CB_toggleClassificationFeedback(GtkWidget *w, gpointer data)
{
  TabTraining *tab = (TabTraining*)(data);
  //return if callback is suppressed
  if(!tab->updating_view)
    {
      bool checked = 
        gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tab->check_classification_feedback));
      tab->getView()->getModel()->trainingSetFeedbackEnabled(checked);
    }
}
