/*!
 * WidgetUtils.cpp
 * \author Jeshua Bratman
 *
 * Contains various widget functions such as alert and confirm boxes.
 */

#include "WidgetUtils.hpp"
#include <string>
#include <iostream>
#include "../model/FileUtils.hpp"
#include "../model/Timer.hpp"
//boost includes
#define BOOST_FILESYSTEM_VERSION 2
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/path.hpp>


using std::cout;
using std::string;
namespace fs = boost::filesystem;

//reference static members
GtkWidget * WidgetUtils::main_window;
bool WidgetUtils::wait_box_open = false;
bool WidgetUtils::wait_box_cancelled = false;
GtkWidget * WidgetUtils::wait_box_btn_cancel = NULL;
GtkWidget * WidgetUtils::wait_box_label_text = NULL;
double WidgetUtils::wait_box_progress = -1.0;

//static methods
void WidgetUtils::Alert(const char *title, const char *text, GtkMessageType type)
{
  cout << text << "\n";
  GtkWidget *alert_dialog;
  alert_dialog = gtk_message_dialog_new(GTK_WINDOW(main_window),
                                        GTK_DIALOG_DESTROY_WITH_PARENT,
                                        type,
                                        GTK_BUTTONS_CLOSE,
                                        "%s",
                                        text);
  gtk_window_set_title(GTK_WINDOW(alert_dialog),title);
  gtk_dialog_run(GTK_DIALOG (alert_dialog));
  gtk_widget_destroy(alert_dialog);
}
void WidgetUtils::Alert(const string title, const string text, GtkMessageType type)
{
  WidgetUtils::Alert(title.c_str(),text.c_str(),type);
}

void WidgetUtils::AlertWarning(const char *title, const char *text)
{
  WidgetUtils::Alert(title,text,GTK_MESSAGE_INFO);
}
void WidgetUtils::AlertWarning(const string title, const string text)
{
  WidgetUtils::AlertWarning(title.c_str(),text.c_str());
}
void WidgetUtils::AlertError(const char *title, const char *text)
{
  WidgetUtils::Alert(title,text,GTK_MESSAGE_ERROR);
}
void WidgetUtils::AlertError(const string title, const string text)
{
  WidgetUtils::AlertError(title.c_str(),text.c_str());
}

bool WidgetUtils::Confirm(const char *title, const char *text, GtkMessageType type)
{
  GtkWidget *confirm_dialog;
  confirm_dialog = gtk_message_dialog_new(GTK_WINDOW(main_window),
                                        GTK_DIALOG_DESTROY_WITH_PARENT,
                                        type,
                                        GTK_BUTTONS_YES_NO,
                                        "%s",
                                        text);
  gtk_window_set_title(GTK_WINDOW(confirm_dialog),title);
  gint response = gtk_dialog_run(GTK_DIALOG (confirm_dialog));
  gtk_widget_destroy(confirm_dialog);
  return response==GTK_RESPONSE_YES;
}

bool WidgetUtils::Confirm(const char *title, const char *text)
{
  return WidgetUtils::Confirm(title,text,GTK_MESSAGE_QUESTION);
}
bool WidgetUtils::Confirm(const string title, const string text)
{
  return WidgetUtils::Confirm(title.c_str(),text.c_str());
}
bool WidgetUtils::Confirm(const string title, const string text, GtkMessageType type)
{
  return WidgetUtils::Confirm(title.c_str(),text.c_str(), type);
}


//----------------------------------------------------------------------

string WidgetUtils::selectSaveFile(string extension)
{
  string ret_filename;
  GtkWidget *dialog;
  dialog = gtk_file_chooser_dialog_new ("Save As",
                                        GTK_WINDOW(WidgetUtils::getMainWindow()),
                                        GTK_FILE_CHOOSER_ACTION_SAVE,
                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                        GTK_STOCK_SAVE, GTK_RESPONSE_OK,
                                        NULL);
  gtk_widget_set_size_request(dialog,400,600);

  while (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      char *tmp;
      tmp = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      string filename = tmp;
      g_free(tmp);


      //add extension
      if(extension != "")
        {
          fs::path full_path;
          full_path = fs::system_complete(fs::path(filename, fs::native));
          string leaf = full_path.leaf();
          int dot = leaf.find(".");
          string ext = "";
          if(dot == -1)
            ext = "."+extension;
          filename += ext;
        }


      if(FileUtils::fileExists(filename))
        {
          //file already exists
          if(!WidgetUtils::Confirm("Overwrite file?","\"" + filename + "\" exists. Overwrite?"))
            {
              continue;
            }
        }
      ret_filename = filename;
      break;
    }
  gtk_widget_destroy (dialog);
  return ret_filename;
}


//----------------------------------------------------------------------

string WidgetUtils::selectLoadFile()
{
  string ret_filename;

  GtkWidget *dialog;
  dialog = gtk_file_chooser_dialog_new ("Load",
                                        GTK_WINDOW(WidgetUtils::getMainWindow()),
                                        GTK_FILE_CHOOSER_ACTION_OPEN,
                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                        GTK_STOCK_OPEN, GTK_RESPONSE_OK,
                                        NULL);
  gtk_widget_set_size_request(dialog,400,600);

  while (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      char *filename;

      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));

      ret_filename = filename;
      g_free (filename);
      break;
    }

  gtk_widget_destroy (dialog);

  return ret_filename;
}




//----------------------------------------------------------------------
// WaitBox

void WidgetUtils::waitBoxShow(const char * text, bool show_progress_bar)
{
  wait_box_progress = -1.0;
  WidgetUtils::waitBox(true, text, show_progress_bar);
}

void WidgetUtils::waitBoxHide()
{
  WidgetUtils::waitBox(false, NULL);
}

void WidgetUtils::waitBoxSetProgress(double progress)
{
  wait_box_progress = progress;
}

bool WidgetUtils::waitBoxCancelled()
{
  return wait_box_cancelled;
}

void WidgetUtils::waitBoxSetText(const char *text)
{
  string str_text = "<small>"+string(text)+"</small>";
  if(GTK_IS_LABEL(wait_box_label_text))
    gtk_label_set_markup(GTK_LABEL(wait_box_label_text),str_text.c_str());
}

void WidgetUtils::waitBoxSetCancelAvailable(bool a)
{
  if(GTK_IS_BUTTON(wait_box_btn_cancel))
    {
      if(a)
        gtk_widget_show(wait_box_btn_cancel);
      else
        gtk_widget_hide(wait_box_btn_cancel);
    }
  wait_box_cancelled = false;
}
void WidgetUtils::CB_waitBoxCancel(GtkWidget *w, gpointer data)
{
  WidgetUtils::wait_box_cancelled = true;
}

gint WidgetUtils::pulseProgressBar(gpointer progress)
{
  bool is_progress = GTK_IS_PROGRESS_BAR(progress);
  if(is_progress)
    {
      if(wait_box_progress < 0)
        gtk_progress_bar_pulse(GTK_PROGRESS_BAR(progress));
      else
        gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progress), 
                                      wait_box_progress);
    }
  return is_progress;
}

void WidgetUtils::waitBox(bool show, const char * text, bool show_progress_bar)
{
  static GtkWidget *dialog;
  if(show && !wait_box_open)
    {
      wait_box_open = true;
      wait_box_cancelled = false;

      gtk_widget_set_sensitive(main_window,false);

      dialog = gtk_window_new(GTK_WINDOW_POPUP);
      gtk_window_set_title(GTK_WINDOW(dialog), "Wait");
      gtk_window_set_position(GTK_WINDOW(dialog),GTK_WIN_POS_CENTER_ON_PARENT);
      gtk_window_set_transient_for(GTK_WINDOW(dialog),GTK_WINDOW(main_window));
      gtk_window_set_destroy_with_parent(GTK_WINDOW(dialog), true);
      gtk_window_set_default_size(GTK_WINDOW(dialog),250,150);


      //create container and add text
      GtkWidget * container = gtk_vbox_new(false, 0);
      wait_box_label_text = gtk_label_new(text);
      waitBoxSetText(text);
      gtk_box_pack_start(GTK_BOX(container),
                         wait_box_label_text,true, true, 0);

      //add progress bar
      if(show_progress_bar)
        {
          GtkWidget * progress = gtk_progress_bar_new();
          gtk_box_pack_start(GTK_BOX(container),
                             progress, false, false, 10);
          gtk_widget_set_size_request(progress,200,30);
          g_timeout_add(150, pulseProgressBar, gpointer(progress));
        }
      //add cancel button
      wait_box_btn_cancel = gtk_button_new_with_label("Cancel");
      g_signal_connect(G_OBJECT(wait_box_btn_cancel),
                       "clicked",
                       G_CALLBACK(CB_waitBoxCancel),
                       NULL);

      GtkWidget * cancel_btn_box = gtk_hbutton_box_new();
      gtk_button_box_set_layout(GTK_BUTTON_BOX(cancel_btn_box),GTK_BUTTONBOX_CENTER);
      gtk_widget_set_size_request(wait_box_btn_cancel,60,25);
      gtk_box_pack_start(GTK_BOX(cancel_btn_box),
                         wait_box_btn_cancel,false, false, 5);
      gtk_box_pack_start(GTK_BOX(container),
                         cancel_btn_box,false, false, 5);
      gtk_widget_hide(wait_box_btn_cancel);

      //show widgets
      GtkWidget * container_frame = gtk_viewport_new(NULL,NULL);
      gtk_container_border_width(GTK_CONTAINER(container_frame),2);
      gtk_container_add(GTK_CONTAINER(container_frame), container);
      gtk_container_add(GTK_CONTAINER(dialog), container_frame);
      gtk_widget_show_all(dialog);
      gtk_widget_hide(wait_box_btn_cancel);

      //update gui
      while(gtk_events_pending())
        {
          gtk_main_iteration();
        }
    }
  else if(!show && wait_box_open)
    {
      gtk_widget_destroy(dialog);
      gtk_widget_set_sensitive(main_window,true);
      wait_box_open = false;
    }
}

