#include <X11/Xlibint.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <X11/keysymdef.h>
#include <X11/keysym.h>
#include <X11/extensions/XTest.h>
#include <sys/shm.h>
#include <time.h>

#include <stdio.h>
#include <string>
#include <iostream>
using std::cerr;
using std::string;
using std::cout;

#include "X11Controller.hpp"

X11Controller::X11Controller()
{
  display = XOpenDisplay(getenv("DISPLAY"));

  if(display == NULL)
    {
      cerr << "Failed to open display.\n";
    }
  srand(time(NULL));
}


X11Controller::~X11Controller()
{
  if(display != NULL)
    {
      XCloseDisplay(display);
    }
}


//----------------------------------------------------------------------
//FUNCTIONS

Window X11Controller::getWindowByNameRecursive(Window root, const char * search_name, int level)
{
  Window root_window, parent_window, result, *child_windows;
  unsigned int num_children = 0;
  char * name;
  //get list of windows
  int ret = XQueryTree(display,
                       root,
                       &root_window, &parent_window,
                       &child_windows, &num_children);
  if(ret)
    {
      for(unsigned int i=0; i<num_children; i++)
        {
          //check if this is the right window
          ret = XFetchName(display,child_windows[i],&name);
          if(ret)
            {
              string str1 = name;
              string str2 = search_name;
              if(str1 == str2)
                {
                  result = child_windows[i];
                  XFree(child_windows);
                  return result;
                }
              XFree(name);
            }
          // no, this isn't the right window, maybe one of its children are
          result = getWindowByNameRecursive(child_windows[i],
                                            search_name, level + 1);
          if(result > 0)
            {
              XFree(child_windows);
              return result;
            }
        }//end of outer window children loop
    }//end of if(ret)
  if(child_windows != NULL ) XFree(child_windows);
  return 0;
}

Window X11Controller::getWindowByName(const char *search_name)
{
  Window ret =  getWindowByNameRecursive(XDefaultRootWindow(display), search_name, 0);
  return ret;
} 
  
bool X11Controller::getCursorXYAbsolute(int *x, int *y)
{
  Window root_return, child_return;
  int win_x_return, win_y_return;
  unsigned int mask_return;
  bool ret = XQueryPointer(display, XDefaultRootWindow(display), 
                           &root_return, &child_return,
                           x,y, 
                           &win_x_return, &win_y_return,
                           &mask_return);
  return ret;
}

bool X11Controller::getWindowXYAbsolute(Window window, int *x, int *y)
{
  XWindowAttributes win_attributes;
  Window not_needed;
  if(XGetWindowAttributes(display, window, &win_attributes))
    {
      return XTranslateCoordinates (display, window, win_attributes.root, 
                                    -win_attributes.border_width,
                                    -win_attributes.border_width,
                                    x, y, &not_needed);
    }
  else
    {
      return false;
    }
}

bool X11Controller::getWindowSize(Window window, int *width, int *height)
{
  XWindowAttributes win_attributes;
  if(XGetWindowAttributes(display, window, &win_attributes))
    {
      *width = win_attributes.width - win_attributes.border_width;
      *height = win_attributes.height - win_attributes.border_width;
      return true;
    }
  else
    {
      return false;
    }
}

bool X11Controller::moveCursorIntoWindow(Window window)
{
  int cursor_x, cursor_y, window_x, window_y, diffx, diffy;
  getCursorXYAbsolute(&cursor_x,&cursor_y);
  getWindowXYAbsolute(window,&window_x,&window_y);
  diffx = window_x - cursor_x;
  diffy = window_y - cursor_y;
  XWarpPointer(display, None, None, 0, 0, 0, 0, diffx, diffy);
  bool ret = XWarpPointer(display, None, None, 0, 0, 0, 0, diffx, diffy);
  XFlush(display);
  return ret;
}


bool X11Controller::moveCursorAbsoluteInWindow(Window window, int x, int y)
{
  int cursor_x, cursor_y, diffx, diffy;
  getCursorXYAbsolute(&cursor_x,&cursor_y);
  diffx = cursor_x + x;
  diffy = cursor_y + y;

  bool ret = XWarpPointer(display, None, window, 0, 0, 0, 0, x, y);
  XFlush(display);
  return ret;
}

bool X11Controller::moveCursorRelative(int x, int y)
{
  bool ret = XWarpPointer(display, None, None, 0, 0, 0, 0, x, y);
  XFlush(display);
  return ret;
}

//----------------------------------------------------------------------
// Clicking

bool X11Controller::buttonPress(unsigned int button)
{
  bool ret = XTestFakeButtonEvent(display, button, true, 10);
  XFlush(display);
  return ret;
}

bool X11Controller::buttonRelease(unsigned int button)
{
  bool ret = XTestFakeButtonEvent(display, button, false, 10);
  XFlush(display);
  return ret;
}

bool X11Controller::leftClick()
{
  bool r1 = buttonPress(1);
  pause();
  bool r2 = buttonRelease(1);
  return r1 && r2;
}

bool X11Controller::rightClick()
{
  bool r1 = buttonPress(3);
  pause();
  bool r2 = buttonRelease(3);
  return r1 && r2;
}

bool X11Controller::middleClick()
{
  bool r1 = buttonPress(2);
  pause();
  bool r2 = buttonRelease(2);
  return r1 && r2;
}

bool X11Controller::doubleClick()
{
  bool r1 = leftClick();
  pause();
  bool r2 = leftClick();
  return r1 && r2;
}

//----------------------------------------------------------------------
// Key Events
KeyCode X11Controller::getKeyCodeFromString(const char * key)
{
  KeySym ks = XStringToKeysym(key);
  KeyCode kc = XKeysymToKeycode(display, ks);
  return kc;
}

bool X11Controller::keyPress(const char * key)
{
  KeyCode kc = getKeyCodeFromString(key);
  if(kc == 0)
    return false;
  else
    {
      bool ret = XTestFakeKeyEvent(display, kc, true, 10);
      XFlush(display);
      return ret;
    }
}

bool X11Controller::keyRelease(const char * key)
{
  KeyCode kc = getKeyCodeFromString(key);
  if(kc == 0)
    return false;
  else
    {
      bool ret = XTestFakeKeyEvent(display, kc, false, 10);
      XFlush(display);
      return ret;
    }
}

bool X11Controller::keySend(const char * key)
{
  bool r1 = keyPress(key);
  pause();
  bool r2 = keyRelease(key);
  return r1 && r2;
}

bool X11Controller::stringSend(const char * str)
{
  bool ret = true;
  char send[32];
  for(unsigned i = 0; i < strlen(str); i++)
    {
      char c = str[i];
      switch(c)
        {
        case ' ':
          strcpy(send,"space");
          break;
        case '\t':
          strcpy(send,"tab");
          break;
        case '\n':
          strcpy(send,"Return");
          break;
        default:
          send[0] = c;
          send[1] = '\0';
        }
      ret = ret && keySend(send);
    }
  return ret;
}

//----------------------------------------------------------------------
// cleaning up functions

void X11Controller::closeDisplay()
{
  if(display != NULL)
    {
      XCloseDisplay(display);
      display = NULL;
    }
}


//----------------------------------------------------------------------
// utility functions
int X11Controller::msleep(unsigned long milisec)   
{   
  struct timespec req={0};   
  time_t sec=(int)(milisec/1000);   
  milisec=milisec-(sec*1000);   
  req.tv_sec=sec;   
  req.tv_nsec=milisec*1000000L;   
  while(nanosleep(&req,&req)==-1)   
    continue;   
  return 1;   
}

int X11Controller::randrange(int low, int high)
{
  return (rand() % (high - low)) + low;
}

void X11Controller::pause()
{  
  msleep(10);
}
