/* PieMenu.cpp
 * \author Jeshua Bratman
 *
 * Pie menu widget for GTK. Used by CEBL pie interfaces.
 */

#include "PieMenu.hpp"

#include <math.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkmain.h>
#include <cstring>
#include <iostream>
#include <string>
#include <vector>
using namespace std;

G_DEFINE_TYPE(PieMenu, pie_menu, GTK_TYPE_DRAWING_AREA);
static gboolean pie_menu_expose (GtkWidget *pie, GdkEventExpose *event);
static void pie_menu_dispose (GObject *self);

typedef struct _PieMenuPrivate PieMenuPrivate;
struct _PieMenuPrivate
 {
   int selected;
   int lit;
   int nSegments;
   bool bars_visible;

   bool train_mode;

   double pulse_value;
   int pulse_direction;

   std::vector<std::string> labels;
   std::vector<std::string> secondary_labels;
   std::vector<double> bar_percentages;
   unsigned int bg_red, bg_green, bg_blue;
 };

#define PIE_MENU_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), PIE_MENU_TYPE, PieMenuPrivate))

static void pie_menu_class_init (PieMenuClass *cls)
{
  GObjectClass *obj_class;
  GtkWidgetClass *widget_class;

  obj_class = G_OBJECT_CLASS (cls);
  widget_class = GTK_WIDGET_CLASS (cls);

  widget_class->expose_event = pie_menu_expose;
  obj_class->dispose = pie_menu_dispose;

  g_type_class_add_private (obj_class, sizeof (PieMenuPrivate));
}



GtkWidget *pie_menu_new ()
{

  return ((GtkWidget*)(g_object_new(PIE_MENU_TYPE,NULL)));
}



/***************************************************
 **Drawing
 ***************************************************/
//draw pie
static void pie_draw (GtkWidget *pie, cairo_t *cr)
{
  if(pie==NULL)
    return;



  double width = pie->allocation.width;
  double height = pie->allocation.height;
  double center_x = width/2;
  double center_y = height/2;
  double radius_normal = MIN (width / 2,height / 2) * .9;
  //double CURSOR_RADIUS = 7;

  //------------
  //get private variables
  PieMenuPrivate *vars;
  vars = PIE_MENU_GET_PRIVATE (pie);

  //-------------
  //pie slice color progression
  double color_red = .5;
  double color_green = .5;
  double color_blue = 1;
  //it is green's turn to change
  double color_turn = 3;
  double color_step;
  switch(vars->nSegments)
    {
    case 2:
      color_step = .5;
      break;
    case 3:
      color_step = .4;
      break;
    case 4:
    case 5:
    case 6:
    case 7:
    case 8:
    case 9:
      color_step = .2;
      break;
    default:
      color_step = .2 - (.01*(((vars->nSegments-1))%10));
      break;
    }
  //-------------

  double inner_radius_normal = radius_normal/1.7;
  double inner_radius = inner_radius_normal;
  double radius = radius_normal;
  double radius_selected = radius_normal*1.1;
  double inner_radius_selected  = radius_normal/2.5;

  int i;


  int nSegments = vars->nSegments;
  int selected = vars->selected;

  double bar_width = inner_radius/(15+nSegments);
  double poly_radius = inner_radius/10;

  double theta = 0;
  cairo_set_source_rgb(cr,
        	       (double)vars->bg_red/65535,
        	       (double)vars->bg_green/65535,
        	       (double)vars->bg_blue/65535);

  cairo_move_to(cr,0,0);
  cairo_line_to(cr,width,0);
  cairo_line_to(cr,width,height);
  cairo_line_to(cr,0,height);
  cairo_line_to(cr,0,0);
  cairo_fill_preserve (cr);
  cairo_stroke(cr);

  for(i=1;i<=nSegments;i++)
    {
      double next_theta = 2*M_PI * (double)i/nSegments;
      double mid_theta = (next_theta + theta)/2.0;
      //set color
      if(vars->lit == i-1)
        {
          cairo_set_source_rgba(cr,color_red * 2, color_green*2, color_blue*1,1);

        }
      else
        {
          cairo_set_source_rgba(cr,color_red, color_green, color_blue, 1);
        }
      cairo_set_line_width(cr,1);

      if(i == selected+1)
        {
          radius = radius_selected;
          inner_radius = inner_radius_selected;
        }
      cairo_move_to(cr,center_x+inner_radius*cos(theta),center_y+inner_radius*sin(theta));
      cairo_line_to(cr,center_x + radius * cos(theta),center_y + radius*sin(theta));
      cairo_line_to(cr,center_x + radius * cos(next_theta),center_y + radius*sin(next_theta));
      cairo_line_to(cr,center_x+inner_radius*cos(next_theta),center_y+inner_radius*sin(next_theta));

      //cairo_line_to(cr,center_x + radius * cos(theta-step),center_y + radius*sin(theta-step));
      cairo_fill_preserve (cr);
      cairo_stroke(cr);

      //now put curved end on
      cairo_arc (cr, center_x, center_y, radius, theta, next_theta);
      cairo_fill_preserve (cr);
      cairo_stroke (cr);




      //----------------
      //put circle inside

      cairo_set_source_rgb(cr,
        		   (double)vars->bg_red/65535,
        		   (double)vars->bg_green/65535,
        		   (double)vars->bg_blue/65535);


      cairo_move_to(cr,center_x,center_y);
      cairo_line_to(cr,center_x + inner_radius * cos(theta),     center_y + inner_radius*sin(theta));
      cairo_line_to(cr,center_x + inner_radius * cos(next_theta),center_y + inner_radius*sin(next_theta));
      cairo_line_to(cr,center_x,center_y);
      cairo_fill_preserve (cr);
      cairo_stroke(cr);


      cairo_arc (cr, center_x, center_y, inner_radius, theta, next_theta);
      cairo_fill_preserve (cr);
      cairo_stroke (cr);

      //----------------------------------------
      //bars

      if(vars->bars_visible)
        {
          double bar_height;
          //if section is lit, extend bar to reach it
          if(vars->lit==(i-1))
            {
              bar_height= inner_radius;

            }
          else
            bar_height= inner_radius *  vars->bar_percentages[i-1];

          if(bar_height != 0)
            {

              double costheta = cos(mid_theta);
              double sintheta = sin(mid_theta);

              double origin1x = center_x - bar_width * sintheta;
              double origin1y = center_y + bar_width * costheta;
              double origin2x = center_x + bar_width * sintheta;
              double origin2y = center_y - bar_width * costheta;

              cairo_set_source_rgba(cr,color_red, color_green, color_blue, 1);

              cairo_set_line_width(cr,1);
              cairo_move_to(cr
        		    , origin1x
        		    , origin1y);
              cairo_line_to(cr
        		    , origin1x + bar_height * costheta
        		    , origin1y + bar_height * sintheta);
              cairo_line_to(cr
        		    , origin2x + bar_height * costheta
        		    , origin2y + bar_height * sintheta);
              cairo_line_to(cr
        		    , origin2x
        		    , origin2y);
              cairo_line_to(cr
        		    , origin1x
        		    , origin1y);

              cairo_fill_preserve (cr);
              cairo_stroke(cr);
            }
        }

      //------------------------------
      //update for next iteration
      radius = radius_normal;
      inner_radius = inner_radius_normal;
      theta = next_theta;
      //-------------------------------
      //color progression
      //red
      if(color_turn == 1)
        {
          color_red = color_red - color_step;
          color_green = color_green + color_step;
          if(color_green >= .99)
            {
              color_turn = 2;
            }
        }
      //green
      else if(color_turn == 2)
        {

          color_green = color_green - color_step;
          color_blue = color_blue + color_step;
          if(color_blue >= .99)
            {
              color_turn = 3;
            }
        }
      //blue
      else
        {
          color_blue = color_blue - color_step;
          color_red = color_red + color_step;
          if(color_red >= .99)
            {
              color_turn = 1;
            }
        }
    }


  //----------------------------------------
  //draw a polygon in the center

  //only try to draw a triangle or bigger if not in train mode
  if(nSegments >= 3 && !vars->train_mode)
    {
      bool moved = false;
      theta = 0.0;
      double x,y;
      cairo_set_source_rgba(cr,0,0,0,1);
      for(i=0;i<nSegments;i++)
        {
          x = center_x + cos(theta) * poly_radius;
          y = center_y + sin(theta) * poly_radius;
          if(!moved)
            {
              cairo_move_to(cr,x,y);
              moved = true;
            }
          else
            {
              cairo_line_to(cr,x,y);
            }
          theta += 2.0 * M_PI / nSegments;
        }
      theta = 0.0;
      x = center_x + cos(theta) * poly_radius;
      y = center_y + sin(theta) * poly_radius;
      cairo_line_to(cr,x,y);
      cairo_fill_preserve (cr);
      cairo_stroke(cr);
    }
  //-------------------------------------------------------
  //text
  char buffer[255];
  for(i=0;i<nSegments;i++)
    {

      theta = 2*M_PI * (double)i/nSegments;
      double next_theta = 2*M_PI * (double)(i+1)/(nSegments);
      double mid_theta = (next_theta + theta)/2.0;

      //draw label
      double text_x = center_x + (radius*.7) * cos(mid_theta);
      double text_y = center_y + (radius*.7) * sin(mid_theta);


      //create label
      cairo_text_extents_t ext;
      double text_start_x, text_start_y;
      if(vars->labels.size() <= unsigned(i))
        {
          sprintf(buffer,"Class %d",i);
        }
      else if(vars->secondary_labels.size() <=unsigned(i) || vars->train_mode)
        {
          strncpy(buffer, vars->labels[i].c_str(), 254);
        }
      else
        {
          string l = vars->labels[i] + " [" + vars->secondary_labels[i] + "]";
          strncpy(buffer, l.c_str(), 254);
        }
      //determine a good size for the font
      int font_size =  int(18 / (vars->nSegments) + width/300);
      int length_factor = 10-(strlen(buffer)/5);
      if(length_factor >= 0)
        font_size+= length_factor;
      cairo_set_font_size(cr,font_size);

      cairo_text_extents(cr, buffer, &ext);
      text_start_x = text_x-ext.width/2;
      text_start_y = text_y;
      double padding = 10;
      double tl_x = text_start_x-padding;
      double tl_y = text_start_y+padding;
      double bl_x = tl_x;
      double bl_y = tl_y - ext.height - padding*2;
      double tr_x = tl_x + ext.width + padding*2;
      double tr_y = tl_y;
      double br_x = tr_x;
      double br_y = bl_y;
      cairo_new_path (cr);
      cairo_move_to(cr,tl_x, tl_y);
      cairo_line_to(cr,bl_x, bl_y);
      cairo_line_to(cr,br_x, br_y);
      cairo_line_to(cr,tr_x, tr_y);
      cairo_line_to(cr,tl_x, tl_y);
      cairo_set_source_rgba(cr,1,1,1,.7);
      cairo_fill_preserve (cr);
      cairo_set_source_rgb(cr,0,0,0);
      cairo_stroke(cr);

      cairo_set_source_rgba(cr,0,0,0, 1);
      cairo_move_to(cr,text_start_x, text_start_y);
      cairo_show_text(cr,buffer);
    }
  //-------------------------------------------------------
  //cursor
  /*double cursor_x,cursor_y;
  if(vars->cursor_visible)
    {
      //printf("!!cursor_x = %f\n",vars->cursor_x);
      cursor_x = vars->cursor_x;//cos(vars->cursor_theta)*(vars->cursor_dist * radius_normal)+center_x;
      cursor_y = vars->cursor_y; //sin(vars->cursor_theta)*(vars->cursor_dist * radius_normal)+center_y;
      //printf("\nx = %f, y = %f",cursor_x,cursor_y);
      cairo_move_to(cr,cursor_x+CURSOR_RADIUS, cursor_y);
      cairo_arc(cr,cursor_x, cursor_y, CURSOR_RADIUS, 0, 2 * M_PI);
      cairo_set_source_rgba(cr, .3,.9,.5,.7);
      cairo_fill_preserve (cr);

      //cairo_move_to(cr,cursor_x+CURSOR_RADIUS,cursor_y);
      cairo_set_source_rgb (cr, 0, 0, 0);
      cairo_stroke (cr);
      }*/


  vars->lit = -1; //unlight segment
}

//expose event
static gboolean pie_menu_expose (GtkWidget *pie, GdkEventExpose *event)
{
  g_return_val_if_fail (pie != NULL, false);
  g_return_val_if_fail (PIE_IS_PIE_MENU(pie), false);



  cairo_t *cr;
  //get cairo_t
  cr = gdk_cairo_create (pie->window);
  //set clip region
  cairo_rectangle (cr,
        	   event->area.x, event->area.y,
        	   event->area.width, event->area.height);
  cairo_clip (cr);
  //draw
  pie_draw (pie, cr);
  cairo_destroy (cr);
  return FALSE;
}

/***************************************************
 **Animation/Redrawing
 ***************************************************/
 static void pie_menu_redraw_canvas(PieMenu *pie)
 {



   g_return_if_fail (pie != NULL);

   g_return_if_fail (PIE_IS_PIE_MENU(pie));

   GtkWidget *widget;
   GdkRegion *region;

   widget = GTK_WIDGET (pie);

   if (!widget->window) return;

   region = gdk_drawable_get_clip_region (widget->window);
   gdk_window_invalidate_region (widget->window, region, TRUE);
   gdk_window_process_updates (widget->window, TRUE);

   gdk_region_destroy (region);
 }


static gboolean pie_menu_update(gpointer data)
{

  g_return_val_if_fail (data != NULL, false);
  if(!PIE_IS_PIE_MENU(data))
    {
      return false;
    }
  PieMenu *pie;
  pie = PIE_MENU(data);
  if(pie==NULL)
    return FALSE;
  pie_menu_redraw_canvas (pie);
  return TRUE; /* keep running this event */
}

static void pie_menu_final(gpointer g_class, gpointer class_data)
{
}
static void pie_menu_dispose (GObject *self)
{
  g_return_if_fail (self != NULL);
  g_return_if_fail (PIE_IS_PIE_MENU(self));

  PieMenu *pie = PIE_MENU(self);
  PieMenuPrivate *vars;
  vars = PIE_MENU_GET_PRIVATE (pie);
}


static void pie_menu_init (PieMenu *pie)
{
  if(pie==NULL)
    return;

  //GtkWidget *widget = GTK_WIDGET (pie);

  //initialize all variables for pie
  PieMenuPrivate * vars = PIE_MENU_GET_PRIVATE (pie);
  vars->nSegments = 3;
  vars->selected = -1;
  //vars->cursor_x = 0;
  //vars->cursor_y = 0.0;
  //vars->cursor_visible = 0;
  vars->bars_visible = false;

  vars->pulse_direction = 0;
  vars->pulse_value = 0.0;
  vars->lit = -1;

  vars->labels.resize(0);
  vars->bar_percentages.resize(vars->nSegments);
  for(int i=0;i<vars->nSegments;i++)
    vars->bar_percentages[i] = 0;

  //update pie with these new values
  pie_menu_update(pie);

  //add a timer to update widget every 200 milliseconds
  g_timeout_add(200, pie_menu_update, pie);
}
/***************************************************
 **mutators
 ***************************************************/

void pie_set_segments(GtkWidget *pie, int segments)
{
 if(pie==NULL)
    return;
  //------------
  //get private variables
  PieMenuPrivate *vars;
  vars = PIE_MENU_GET_PRIVATE (pie);
  vars->nSegments = segments;
  vars->bar_percentages.resize(segments);
  for(int i=0;i<vars->nSegments;i++)
    vars->bar_percentages[i] = 0;

}

void pie_set_bg(GtkWidget *pie, unsigned int red, unsigned int green, unsigned int blue)
{
 if(pie==NULL)
    return;
  //------------
  //get private variables
  PieMenuPrivate *vars;
  vars = PIE_MENU_GET_PRIVATE (pie);
  vars->bg_red = red;
  vars->bg_green = green;
  vars->bg_blue = blue;
}

void pie_set_selected(GtkWidget *pie, int segment)
{
  //  cout << "setting selected to " << segment << "\n";
  if(pie==NULL)
    return;
  //------------
  //get private variables
  PieMenuPrivate *vars;
  vars = PIE_MENU_GET_PRIVATE (pie);
  if(vars->selected != segment)
    {
      vars->selected = segment;
      pie_menu_update(pie);
    }
}



void pie_set_bars_visible(GtkWidget *pie, bool val)
{
  if(pie==NULL)
    return;
  //------------
  //get private variables
  PieMenuPrivate *vars;
  vars = PIE_MENU_GET_PRIVATE (pie);
  vars->bars_visible=val;
  if(!val)
    for(int i=0;i<vars->nSegments;i++)
      vars->bar_percentages[i] = 0;
}


void pie_set_labels(GtkWidget *pie, std::vector<std::string> labels)
{
  if(pie==NULL)
    return;
  PieMenuPrivate *vars;
  vars = PIE_MENU_GET_PRIVATE (pie);
  vars->labels = labels;
}

void pie_set_secondary_labels(GtkWidget *pie, std::vector<std::string> labels)
{
  if(pie==NULL)
    return;
  PieMenuPrivate *vars;
  vars = PIE_MENU_GET_PRIVATE (pie);
  vars->secondary_labels = labels;
}


void pie_set_train_mode(GtkWidget *pie)
{
  if(pie==NULL)
    return;
  PieMenuPrivate *vars;
  vars = PIE_MENU_GET_PRIVATE(pie);

  if(!vars->train_mode)
    {
      vars->train_mode = true;
      pie_menu_update(pie);
    }
}

void pie_set_use_mode(GtkWidget *pie)
{
  if(pie==NULL)
    return;
  PieMenuPrivate *vars;
  vars = PIE_MENU_GET_PRIVATE(pie);
  if(vars->train_mode)
    {
      vars->train_mode = false;
      pie_menu_update(pie);
    }
}



void pie_select_class(GtkWidget *pie, int cls)
{
  if(pie==NULL)
    {
      cerr << "ERROR pie is NULL\n";
      return;
    }
  PieMenuPrivate *vars;
  vars = PIE_MENU_GET_PRIVATE (pie);

  for(int i=0;i<vars->nSegments;i++)
    vars->bar_percentages[i] = 0;
  vars->lit = cls;
}

void pie_set_class_proportions(GtkWidget *pie, std::vector<double> proportions)
{
  if(pie==NULL)
    {
      cerr << "ERROR pie is NULL\n";
      return;
    }


  PieMenuPrivate *vars;
  vars = PIE_MENU_GET_PRIVATE (pie);

  if(unsigned(vars->nSegments) > proportions.size())
    {
      cerr << "ERROR vector is of wrong size in pie_set_class_proportions\n";
      return;
    }

  for(int i=0;i<vars->nSegments;i++)
    {
      vars->bar_percentages[i] = proportions[i];
    }
}
