#include <fstream>
#include <sstream>
#include <boost/regex.hpp>
#include "../CEBLModel.hpp"
#include "ChannelsConfig.hpp"
#include "Preferences.hpp"
using namespace std;

//----------------------------------------------------------------------
// Constructors / Destructors

ChannelsConfig::ChannelsConfig(CEBLModel * model)
{
  this->model = model;
  max_num_channels = 24;
  channels_names.resize(max_num_channels);
  channels_enabled.resize(max_num_channels);
  channels_reference.resize(max_num_channels);
  config_filename = model->getPreferences()->getDefaultChannelsFilename();
  try
    {
      loadFile(config_filename);
    }
  catch(...)
    {
      cerr << "Error loading default channels file. Enabling 8 channels.\n";
      for(int i=0;i<8;i++)
        this->setElectrodeEnabled(i,true);
    }
}

ChannelsConfig::~ChannelsConfig()
{

}


//----------------------------------------------------------------------
//GETTERS



string ChannelsConfig::getCurrentFilename()
{
  return config_filename;
}

string ChannelsConfig::getElectrodeName(int electrode)
{
  return channels_names[electrode];
}

bool ChannelsConfig::getElectrodeEnabled(int electrode)
{
  return channels_enabled[electrode];
}

bool ChannelsConfig::getElectrodeReference(int electrode)
{
  return channels_reference[electrode];
}

int ChannelsConfig::getMaxNumChannels()
{
  return max_num_channels;
}

int ChannelsConfig::getNumEnabled()
{
  int count = 0;
  for(unsigned int i = 0; i<this->channels_enabled.size();i++)
    {
      if(this->channels_enabled[i])
        count++;
    }
  return count;
}

std::vector<string> ChannelsConfig::getEnabledNames()
{
  std::vector<string> ret;
  for(unsigned int i = 0; i<this->channels_enabled.size();i++)
    {
      if(this->channels_enabled[i])
        ret.push_back(getElectrodeName(i));
    }
  return ret;
}

string ChannelsConfig::getConfigurationString()
{
  stringstream ss;
  this->saveConfigToStream(ss);
  return ss.str();
}




//----------------------------------------------------------------------
//SETTERS




void ChannelsConfig::setElectrodeName(int electrode, string name)
{
  channels_names[electrode] = name;
}

void ChannelsConfig::setElectrodeReference(int electrode, bool enabled)
{
  channels_reference[electrode] = enabled;
}

void ChannelsConfig::setElectrodeEnabled(int electrode, bool enabled)
{
  channels_enabled[electrode] = enabled;
}

void ChannelsConfig::setConfigurationFromString(string config)
{
  stringstream ss;
  ss << config;
  this->parseConfigFromStream(ss);
}




//---------------------------------------------------
// LOADING




void ChannelsConfig::loadFile(string filename)
{

  //open the file
  ifstream ifs(filename.c_str());
  if(!ifs.is_open())
    {
      throw FileException("Failed to open \"" + filename + "\".");
    }

  if(parseConfigFromStream(ifs))
    this->config_filename = filename;
  else
    {
      throw FileException("Failed to parse channels config file  \""
                          + filename
                          + "\".");
    }

}

bool ChannelsConfig::parseConfigFromStream(std::istream &ifs)
{
  bool success = true;
  vector<string> new_names;
  vector<bool> new_enabled;
  vector<bool> new_reference;
  new_names.resize(channels_names.size());
  new_enabled.resize(channels_enabled.size());
  new_reference.resize(channels_reference.size());

  boost::regex comment_regexp("^[[:space:]]*#.*");
  boost::regex valid_line_regexp("^[[:space:]]*\\d+[[:space:]]+[\\w|-]+");
  char buffer[256];
  string line;
  int jack = 0;
  string name("");
  char ref;
  char enable;
  int valid_line_count = 0;

  try
    {
      while(!ifs.eof())
        {
          //get next line
          ifs.getline(buffer,256);
          line = buffer;
          //check if this line is a comment
          if(regex_search(line,comment_regexp))
            {
              continue;
            }
          if(!regex_search(line,valid_line_regexp))
            {
              continue;
            }
          else
            valid_line_count++;
          stringstream ss_parser(line);
          //enable exceptions for failure
          ss_parser.exceptions(ifstream::failbit);
          //-------------------------
          try{
            ss_parser >> jack;
          }catch(ifstream::failure e){
            continue;
          }
          //-------------------------
          try{
            ss_parser >> name;
          }catch(ifstream::failure e){
            continue;
          }
          //-------------------------
          try{
            ss_parser >> ref;
          }catch(ifstream::failure e){
            continue;
          }
          //-------------------------
          try{
            ss_parser >> enable;
          }catch(ifstream::failure e){
            continue;
          }
          //-------------------------
          //now we have a whole line
          if(jack >0 && jack <= max_num_channels)
            {
              new_enabled[jack-1] = enable=='y';
              new_reference[jack-1] = ref=='y' || ref=='r';
              if(name!="-")
                new_names[jack-1] = string(name);
              else
                new_names[jack-1] = "";
            }

        }
    } catch(...)
    {
      success = false;
    }
  if(valid_line_count == 0)
    success = false;

  //set new values if successful
  if(success)
    {
      channels_names = new_names;
      channels_enabled = new_enabled;
      channels_reference = new_reference;
    }
  return success;
}




//--------------------------------------------------
// SAVING




void ChannelsConfig::saveFile(string filename)
{
  string message;
  ofstream ofs(filename.c_str());
  if(!ofs.is_open())
    {
      throw FileException("Cannot open file for reading.");
    }

  if(!saveConfigToStream(ofs))
    {
      throw FileException("Failed to write file.");
    }
}

//------------------------------------------------------------------------------
bool ChannelsConfig::saveConfigToStream(std::ostream &ofs)
{
  bool success = true;
  try
    {
      ofs << "#\tname\tref\tenable" << endl;
      for(int i=0; i < max_num_channels; i++)
        {
          ofs << i+1 << '\t'
              <<  (channels_names[i] == "" ? "-" : channels_names[i]) << '\t'
              << (channels_reference[i] ? 'y' : '-') << '\t'
              << (channels_enabled[i] ? 'y' : 'n') << endl;
        }
    }
  catch(...)
    {
      success = false;
    }
  return success;
}
