/*
 * CEBL : CSU EEG Brain-Computer Interface Lab
 *
 * Author: Jeshua Bratman - jeshuabratman@gmail.com
 *
 * This file is part of CEBL.
 *
 * CEBL is free software; you can redistribute it and/or modify it.
 * We only ask that if you use our code that you cite the source in
 * your project or publication.
 *
 * EEG Group (www.cs.colostate.edu/eeg)
 * Department of Computer Science
 * Colorado State University
 *
 */

/*! \class CEBL::Classifier
 *  \author Jeshua Bratman
 *
 * Provides interface for classifier plugins.
 *
 */


#ifndef CLASSIFIER_H
#define CLASSIFIER_H

#include <vector>


#include "../cppR/cppR.hpp"
#include "Plugin.hpp"
#include "EEGTrainingData.hpp"
#include "../Param.hpp"



namespace CEBL
{
  class Classifier : public Plugin
  {
  protected:
    bool trained;
    bool compute_probs;
    std::vector<std::vector<double> > probabilities;
    //number of classes the classifier is using for classification
    int using_classes;
    int using_lags;
    //number of classes the classifier is trained on
    int trained_classes;
    int trained_lags;

  public:
    Classifier(){ trained = false;compute_probs = true;using_classes=using_lags=trained_classes=trained_lags=0; }
    virtual ~Classifier(){}
    bool isTrained() { return trained; }

    /*
     * called when training data changes
     * use_classes is number of classes set to be used in classification
     */
    void reset(CEBL::Param param)
    {
      int use_classes = param.getInt();
      //save number of classes to use for classification
      if(use_classes>=1)
        using_classes = use_classes;

      /* lags is currently unused
       * uncomment if use of lags reimplemented

      int use_lags = (params["lags"]).getInt();
      if(use_lags>=0)
        using_lags = use_lags;

      */
    }

    /*! Gets flag determining whether classifier should compute
      probabilities or not.

      \return
     */
    virtual bool getProbabilitiesFlag(){ return compute_probs; }

    /*! Gets flag determining whether classifier should compute
      probabilities or not
      \param flag
     */
    virtual void setProbabilitiesFlag(bool flag){ compute_probs = flag; }



    /*! Gets probabilities for each class from latest classification.
      If your classifier cannot produce probabilities, just set probabilities
      to a vector where the winning class gets 100% and others get 0%

      \return 
    */
    virtual std::vector<std::vector<double> > getProbabilities()
    {
      return probabilities;
    }

    int getTrainedClasses(){ return trained_classes; }

    int getTrainedLags(){ return trained_lags; }


    /*! Updates classifier with training data set.
      By default just trains the classifier.
     */
    virtual void update(const EEGTrainingData &data)
    {
      train(data);
    }


    //-----------------------------------------------------------------
    //Functions to override below

    /*! Train classifier over training data set 
     */
    virtual void train(const EEGTrainingData &) = 0;

    /*! Use classifier on data and return class for each sample. 

      \return 
    */
    virtual ublas::vector<int> use(const ublas::matrix<double> &) = 0;

  };

}


#endif
