#include "ClassifiersConfig.hpp"
#include "../CEBLModel.hpp"

//----------------------------------------------------------------------
// CONSTRUCTORS / DESTRUCTORS

ClassifiersConfig::ClassifiersConfig(CEBLModel *model)
{
  this->model = model;
  this->plugin_loader = new PluginLoader<Classifier>;
  this->selected_classifier = "";
  std::vector<string> paths = model->preferencesGetPaths();
  for(unsigned int i=0; i<paths.size();i++)
    {
    string path = paths[i]+"plugins/classifiers/";
    try
    {
      plugin_loader->loadDir(path.c_str());
    }catch(FileException & e)
    {
      //cerr << e.what() << "\n";
      //if there were exceptions, just continue
      continue;
    }
    }
}

ClassifiersConfig::~ClassifiersConfig()
{
  delete plugin_loader;
}

//----------------------------------------------------------------------
//GETTING OPERATIONS

//special sort to put QDA on top
bool classifierSort(string i, string j)
{
  if(i == "QDA")
    return true;
  else
    return (i<j);
}

std::vector<string> ClassifiersConfig::getNameList()
{
  std::vector<string> names = plugin_loader->getNames();
  sort(names.begin(),names.end(),classifierSort);
  return names;
}

std::vector<string> ClassifiersConfig::getPathList()
{
  return plugin_loader->getPaths();
}

bool ClassifiersConfig::isTrained(string classifier)
{
  if(classifier == "")
    classifier = this->selected_classifier;
  try
  {
    if(plugin_loader->getPlugin(classifier) != NULL)
      return plugin_loader->getPlugin(classifier)->isTrained();
  }
  catch(...)
  {
    throw PluginException("Failed to get trained status from classifier: "
        + classifier);
  }
  return false;
}

string ClassifiersConfig::getSelected()
{
  return this->selected_classifier;
}

std::map<std::string, CEBL::Param> ClassifiersConfig::getParams(string classifier)
{
  try
  {
    return plugin_loader->getPlugin(classifier)->getParamsList();
  }
  catch(...)
  {
    throw PluginException("Failed to get parameter list for classifier: "
        + classifier);
  }
}

bool ClassifiersConfig::getUseProbs()
{
  string classifier = selected_classifier;
  return plugin_loader->getPlugin(classifier)->getProbabilitiesFlag();
}

std::vector<std::vector<double> > ClassifiersConfig::getLastProbs()
{
  string classifier = selected_classifier;
  return plugin_loader->getPlugin(classifier)->getProbabilities();
}

int ClassifiersConfig::getTrainedClasses()
{
  string classifier = selected_classifier;
  return plugin_loader->getPlugin(classifier)->getTrainedClasses();
}

int ClassifiersConfig::getTrainedLags()
{
  string classifier = selected_classifier;
  return plugin_loader->getPlugin(classifier)->getTrainedLags();
}

//----------------------------------------------------------------------
//SETTING OPERATIONS

void ClassifiersConfig::reset(CEBL::Param param, string classifier)
{
  if(classifier == "")
    classifier = this->selected_classifier;
  try
  {
    if(plugin_loader->getPlugin(classifier)!=NULL)
      plugin_loader->getPlugin(classifier)->reset(param);
  }
  catch(...)
  {
    throw PluginException("Failed to reset classifier: " + classifier);
  }
}

void ClassifiersConfig::setSelected(string classifier)
{
  this->selected_classifier = classifier;
  if(plugin_loader->getPlugin(classifier) == NULL)
    {
    cerr << "ERROR: trying to select classifier " << classifier
        << " which doesn't seem to exist.\n";
    }
}

void ClassifiersConfig::train(EEGTrainingData &training_data, string classifier)
{
  if(classifier == "")
    classifier = this->selected_classifier;
  try
  {
    this->is_training = true;
    this->currently_training_classifier = classifier;
    plugin_loader->getPlugin(classifier)->train(training_data);
  }
  catch(...)
  {
    this->is_training = false;
    throw PluginException("Failed to train classifier: " + classifier);
  }
}

void ClassifiersConfig::haltTrain()
{
  if(is_training)
    {
    try
    {
      plugin_loader->getPlugin(currently_training_classifier)->halt();
    }
    catch(...)
    {
      cerr << "error halting training.\n";
    }
    }
  this->is_training = false;
}

void ClassifiersConfig::setParams(std::map<std::string, CEBL::Param> params,
    string classifier)
{
  //if the parameter is blank, use the selected classifier instead
  if(classifier == "")
    classifier = this->selected_classifier;

  //if the selected classifier is also nothing, then don't attempt to set params
  if(classifier == "")
    return;

  try
  {
    plugin_loader->getPlugin(classifier)->setParamsList(params);
  }
  catch(...)
  {
    throw PluginException("Failed to set parameter list for classifier: "
        + classifier);
  }
}

void ClassifiersConfig::setUseProbs(bool flag)
{
  string classifier = selected_classifier;
  plugin_loader->getPlugin(classifier)->setProbabilitiesFlag(flag);
}

//----------------------------------------------------------------------
// USE CLASSIFIER

ublas::vector<int> ClassifiersConfig::use(EEGData &data)
{
  string classifier = selected_classifier;
  if(!isTrained())
    {
    throw ClassificationException("Classifier is not trained.");
    }
  else
    {
    ublas::matrix<double> d = data;
    return plugin_loader->getPlugin(classifier)->use(d);
    }
}


