
#include <fstream>
#include <ctime>
#include <boost/regex.hpp>

#define BOOST_FILESYSTEM_VERSION 2
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/lexical_cast.hpp>

using namespace std;
namespace fs = boost::filesystem;

#include "../Exceptions.hpp"
#include "DataIO.hpp"
#include "EEGTrainingData.hpp"



namespace DataIO
{
  //----------------------------------------------------------------------

  void removeTempDir(string temp_dir)
  {
    fs::path full_path;
    full_path = fs::system_complete(fs::path(temp_dir,
                                             fs::native));
    try
      {
        if(fs::exists(full_path))
          {
            fs::remove_all(full_path);
          }
      }
    catch(...)
      {
        cerr << "Could not delete temp directory...\n";
      }
  }



  //----------------------------------------------------------------------


  string createTempDir()
  {
    //TODO: make more compatible
    string TEMP = "/tmp";
    char buffer[] = "/tmp/CEBL-XXXXXX";
    mkdtemp(buffer);
    string temp_dir = buffer;

    fs::path full_path;

    //set full path to the temp directory
    full_path = fs::system_complete(fs::path(TEMP, fs::native));

    //see if temp directory exists
    if(fs::exists(full_path))
      {
        full_path = fs::system_complete(fs::path(temp_dir.c_str(), fs::native));
        //temp dir does not exist
        if(!fs::exists(full_path))
          {
            try
              {
                fs::create_directory(full_path);
              }
            catch(...)
              {
                cerr << "Failed to create temporary directory.\n";
                throw FileException("DataIO: Failed to create temp directory.");
              }
          }
        return temp_dir;
      }
    else
      {
        throw FileException("DataIO: Failed to create temp directory.");
      }
  }



  //----------------------------------------------------------------------




  void saveTrainingDataToFile(const EEGTrainingData &tdata, string filename)
  {
    EEGTrainingData temp;
    int lags = 0;
    std::vector<int> components;
    ublas::matrix<double> filter;
    saveTrainingSessionToFile(tdata,filename,temp,lags,components,filter);
  }

  //----------------------------------------------------------------------

  //! saves a filtered and unfiltered training data session to a file
  void saveTrainingSessionToFile(const EEGTrainingData &unfiltered_data,
                                 string filename,
                                 const EEGTrainingData &filtered_data,
                                 int filter_lags,
                                 std::vector<int> filter_removed_components,
                                 ublas::matrix<double> filter_matrix)
  {

    //decide if filtered data is good
    bool filtered = filtered_data.numClasses() == unfiltered_data.numClasses();


    string current_temp_dir = createTempDir();


    //now create data files
    for(int cls = 0; cls<unfiltered_data.numClasses(); cls++)
      {
        for(int seq=0; seq<unfiltered_data.numSequences(cls); seq++)
          {
            stringstream fname;
            fname <<  current_temp_dir << "/class-" << cls << "_seq-" << seq;
            unfiltered_data.getConst(cls,seq).saveToFile(fname.str().c_str());
          }
      }


    //now create data files for filtered data
    string filter_matrix_filename = current_temp_dir + "/filter";
    if(filtered)
      {
        for(int cls = 0; cls<filtered_data.numClasses(); cls++)
          {
            for(int seq=0; seq<filtered_data.numSequences(cls); seq++)
              {
                stringstream fname;
                fname <<  current_temp_dir << "/filtered_class-" 
                      << cls << "_seq-" << seq;
                filtered_data.getConst(cls,seq).saveToFile(fname.str().c_str());
              }
          }

        //write filter matrix
        EEGData filt = filter_matrix;
        filt.saveToFile(filter_matrix_filename.c_str());
      }

    //----------------------------------------------------------------------
    // create data description file

    stringstream fname;
    fname <<  current_temp_dir << "/Description.txt";
    ofstream desc_file;
    desc_file.open(fname.str().c_str());

    //save the date and time
    time_t rawtime;
    struct tm * current_tm;
    time(&rawtime);
    current_tm = localtime(&rawtime);
    desc_file <<  "# Date: " << asctime(current_tm);


    desc_file << "# Session\n";
    desc_file << "num_classes = " << unfiltered_data.numClasses() << "\n";
    desc_file << "num_sequences = " << unfiltered_data.numSequences() << "\n";
    std::vector<int> ord = unfiltered_data.getSequenceOrder();
    if(ord.size() > 0)
      {
        desc_file << "class_sequence_ordering = ";
        for(unsigned i=0;i<ord.size();i++)
          {
            desc_file << ord[i];
            if(i != ord.size()-1)
              desc_file << ", ";
          }
        desc_file << "\n";
      }


    desc_file << "# Class Labels\n";
    std::vector<string> labels = unfiltered_data.getClassLabels();
    if(labels.size() > 0)
      {
        for(int i=0; i < unfiltered_data.numClasses(); i++)
          {
            desc_file << "class" << i << " = \"" <<  labels.at(i) << "\"\n";
          }
      }

    std::vector<string> channel_names = unfiltered_data.getChannelNames();
    desc_file << "# Channel Names\n";
    if(channel_names.size() > 0)
      {
        for(unsigned i=0; i < channel_names.size(); i++)
          {
            desc_file  << "channel" << i << " = \"" 
                       << channel_names.at(i) << "\"\n";
          }
      }

    desc_file << "# Data Files Size - CHANNELS x SAMPLES\n";
    for(int cls = 0; cls<unfiltered_data.numClasses(); cls++)
      {
        for(int seq=0; seq<unfiltered_data.numSequences(cls); seq++)
          {
            desc_file  << "class-" << cls << "_seq-" << seq << " = "
                       << unfiltered_data.getConst(cls,seq).numChannels() 
                       << "x"
                       << unfiltered_data.getConst(cls,seq).numSamples() 
                       << "\n";
          }
      }

    desc_file  << "# Filter\n";
    desc_file  << "filtered_data_included = " << filtered << "\n";
    if(filtered)
      {
        desc_file  << "filter_matrix_file = filter\n";
        desc_file  << "filter_matrix_size = " << filter_matrix.size1() 
                   << "x" << filter_matrix.size2() << "\n";
        desc_file  << "filter_num_lags = " << filter_lags << "\n";
        desc_file  << "filter_removed_components = ";
        for(unsigned i=0;i<filter_removed_components.size();i++)
          {
            desc_file << filter_removed_components[i];
            if(i != filter_removed_components.size()-1)
              desc_file << ", ";
          }
        desc_file << "\n";
      }

    desc_file.close();


    //----------------------------------------------------------------------

    //now tar files and move to specified location
    string command = "cd " + current_temp_dir +"; tar -cjf " 
      + static_cast<string>(filename) +  " ./*";
    cout << command << endl;
    system(command.c_str());

    //remove the temp dir
    removeTempDir(current_temp_dir);
  }

  //----------------------------------------------------------------------

  //load from file
  EEGTrainingData loadTrainingDataFromFile(string filename)
  {
    EEGTrainingData tdata;

    string current_temp_dir = createTempDir();
    fs::path full_path;
    //check that we can open the file
    string file = filename;
    full_path = fs::system_complete(fs::path(file,fs::native));
    if(!fs::exists(full_path))
      {
        cerr << "File does not exist.\n";
        throw FileException("DataIO: File does not exist.");
      }
    //now tar files and move to specified location
    string command = "cd " + current_temp_dir + "; cp "+file+" " 
      + current_temp_dir +"; tar -xjf *.bz2;rm *.bz2";
    cout << command << endl;
    system(command.c_str());

    //now load the data files
    full_path = fs::system_complete(fs::path(current_temp_dir,
                                             fs::native));
    //loop through directory
    fs::directory_iterator end_iter;
    for (fs::directory_iterator dir_itr( full_path );
         dir_itr != end_iter;
         ++dir_itr )
      {

        //cout << "----------------------------------------\n";
        //cout << tdata << "\n";
        try
          {
            //make sure file is correctly named
            string file = dir_itr->leaf();

            boost::regex filename_regexp("^class-(\\d*)_seq-(\\d*)");
            //check if this file is a data file
            if(!regex_search(file,filename_regexp))
              {
                continue;
              }

            cout << "reading " << file << "\n";
            //get class and sequence number from fileename
            boost::cmatch what;
            regex_match(file.c_str(), what, filename_regexp);

            int c = 0;
            int s = 0;
            try
              {
                c = boost::lexical_cast<int>(what[1]);
                s = boost::lexical_cast<int>(what[2]);
              } 
            catch(...)
              {
                cerr << "failed to load class " << c << " seq " << s << "\n";
                continue; 
              }
            //create a new data object
            EEGData new_data;
            new_data.loadFromFile(current_temp_dir+"/"+file);
            tdata.set(c,s,new_data);
            //cout << "Loaded class " << c << ", sequence "  << s << "\n";

          }
        catch(const std::exception & ex)
          {
            cerr << "Exception: " << ex.what() << "\n";
          }
      }
    cout << tdata << "\n";
    return tdata;
  }
}

