#include "DataSource.hpp"
#include "../CEBLModel.hpp"
#include "EEGDataStream.hpp"
#include "RandomDataStream.hpp"
#include "FileDataStream.hpp"
#include "MindsetStream.hpp"

//----------------------------------------------------------------------
// CONSTRUCTORS / DESTRUCTORS

DataSource::DataSource(CEBLModel * model)
{
  this->model = model;

  source_names.push_back("Mindset");
  source_names.push_back("Random Numbers");
  source_names.push_back("Random EEG");
  source_names.push_back("File");
  selected_source = 0;
  active_stream_type = 0;
  data_store_flag = false;
  data_stream = NULL;
  data_buffer.clear();
}

DataSource::~DataSource()
{
  if(data_stream != NULL)
    delete data_stream;
}



//----------------------------------------------------------------------
// GETTING OPERATIONS


//! reads and returns n samples from data stream
//  also stores the results into buffer if data store
//  flag is set
EEGData DataSource::read(int samples)
{
  if(!sourceReady())
    throw DataException("Selected data source is not ready.");
  EEGData ret = data_stream->read(samples);

  // save into buffer
  if(this->data_store_flag)
    {
      this->data_buffer.append(ret);
    }
  return ret;
}

//! reads and returns all samples from data stream
//  also stores the results into buffer if data store
//  flag is set
EEGData DataSource::readAll()
{
  if(!sourceReady())
    throw DataException("Selected data source is not ready.");
  EEGData ret = data_stream->readAll();
  // save into buffer
  if(this->data_store_flag)
    {
      this->data_buffer.append(ret);
    }

  return ret;
}

int DataSource::samplesAvailable()
{
  if(!sourceReady())
    throw DataException("Selected data source is not ready.");
  return data_stream->getSamplesAvailable();
}

bool DataSource::sourceReady()
{
  if(data_stream == NULL || !data_stream->isStarted())
    return false;
  else
    return true;
}

bool DataSource::isStarted()
{
  if(data_stream != NULL)
    return data_stream->isStarted();
  else
    return false;
}

bool DataSource::getStoreFlag()
{
  return this->data_store_flag;
}

int DataSource::getStoreNumSamples()
{
  return this->data_buffer.numSamples();
}
EEGData DataSource::getStoredData()
{
  return this->data_buffer;
}

//----------------------------------------------------------------------
// SETTING OPERATIONS

void DataSource::clearStoredData()
{
  this->data_buffer.clear();
}

void DataSource::setStoreFlag(bool flag)
{
  this->data_store_flag = flag;
}

void DataSource::setSource(int source)
{
  this->selected_source = source;
}

void DataSource::setSource(string source)
{
  for(unsigned int i=0; i<source_names.size();i++)
    {
      if(source_names[i]==source)
        {
          selected_source = i;
          break;
        }
    }

}

void DataSource::clearSamples()
{
  if(data_stream != NULL)
    data_stream->readAll();
}

void DataSource::createDataStream()
{
  if(this->data_stream != NULL)
    {
      delete this->data_stream;
      this->data_stream = NULL;
    }

  switch(this->selected_source)
    {
    case 0:
      this->data_stream = new MindsetStream(this->model);
      break;
    case 1:
      this->data_stream = new RandomDataStream();
      break;
    case 2:
      this->data_stream = new RandomDataStream();
      break;
    case 3:
      this->data_stream = new FileDataStream(this->model);
      break;

    }
  this->active_stream_type = selected_source;
}

void DataSource::start()
{
  // create data stream if it has not been created or is of the wrong type
  if(data_stream == NULL || active_stream_type != selected_source)
    createDataStream();
  if(data_stream != NULL)
    data_stream->start();
  else
    cerr << "DataSource: Illegal source selected.\n";
}

void DataSource::stop()
{
  if(data_stream != NULL)
    {
      data_stream->stop();
    }
}
