/*
* CEBL : CSU EEG Brain-Computer Interface Lab
*
* Author: Jeshua Bratman - jeshuabratman@gmail.com
*
* This file is part of CEBL.
*
* CEBL is free software; you can redistribute it and/or modify it.
* We only ask that if you use our code that you cite the source in
* your project or publication.
*
* EEG Group (www.cs.colostate.edu/eeg)
* Department of Computer Science
* Colorado State University
* 
*/

/*! 
 * \class CEBL::Decision
 * \author Jeshua Bratman
 *
 * Provides interface for realtime decision making plugins.
 *
 */


#ifndef DECISION_H
#define DECISION_H

#include <vector>
#include "Plugin.hpp"


namespace CEBL
{

  class Decision : public Plugin
  {
  protected:
    int num_classes;
  public:
    virtual ~Decision() 
    {
      num_classes = 3;
    }


    /*! Initialize the decision process. 
      \param num_classes 
    */
    virtual void init(int num_classes)
    {
      this->num_classes = num_classes;
    };


    /*! Updates decision with classes for several samples.
      \param classes classes for several samples
    */
    virtual void updateWithClassification(ublas::vector<int> classes)
    {
      for(unsigned i=0; i<classes.size();i++)
        updateWithClassification(classes[i]);
    }

    //! perform update over several samples
    void updateWithProbabilities(std::vector<std::vector<double> > probs)
    {
      for(unsigned i=0; i<probs.size();i++)
        updateWithProbabilities(probs.at(i));
    }

    /*! Update for a single sample based on classification.
      By default, this will construct a list of probabilities with 100% for
      the class and 0% for all other classes and call updateWithProbabilities()
      \param cls classified class number
    */
    virtual void updateWithClassification(int cls)
    {
      std::vector<double> probs;
      for(int j=0;j<num_classes;j++)
        {
          if(j==cls)
            probs.push_back(1.0);
          else
            probs.push_back(0.0);
        }
      updateWithProbabilities(probs);
    }

    //------------------------------------------------------------
    // PURE VIRTUAL FUNCTIONS BELOW


    /*! Update decision process based on probabilities from a classifiction
      if we don't have probs available, just use updateWithClassification.

      \param probs 
    */
    virtual void updateWithProbabilities(std::vector<double> probs) = 0;


    /*! Return percents for each class. 
      
      \return 
    */
    virtual std::vector<double> decideClasses() = 0;


  };
}



#endif
