/*!
 * EEGData.cpp
 * \author Jeshua Bratman
 *
 * Class designed to encapsulate the management of a data
 * matrix used specifically for EEG.
 */


#include "Exceptions.hpp"
#include "EEGData.hpp"
#include "cppR/cppR.hpp"
#include <sstream>
#include <iostream>
#include <boost/lexical_cast.hpp>
#include <boost/numeric/ublas/io.hpp>
using namespace std;
using namespace boost;

//CONSTRUCTORS:
EEGData::EEGData()
{
}

EEGData::EEGData(const ublas::matrix<double> &data)
{
  this->data = data;
}
EEGData::EEGData(const EEGData &data)
{
  this->data = data.data;
}
EEGData::~EEGData()
{
}


//binds two matrices together by adding new matrix after the last column of the old
void EEGData::append(const EEGData &new_data)
{
  if(new_data.size1() <= 0 || new_data.size2() <= 0)
    return;

  //int orig_size1 = this->data.size1();
  int orig_size2 = this->data.size2();

  int new_size1 = new_data.size1();
  int new_size2 = orig_size2 + new_data.size2();

  try
    {
      this->data.resize(new_size1, new_size2);
    }
  catch(...)
    {
      MatrixException ex("Error resizing eeg data matrix in order to append data.");
      throw(ex);
    }

  for(int row = 0; row < new_data.size1(); row++)
    {
      for(int col = 0; col< new_data.size2(); col++)
        {
          this->data(row, col+orig_size2) = new_data.data(row,col);
        }
    }

}

//remove all data from matrix
void EEGData::clear()
{
  data.resize(0,0,false);
}

/**********************************************
 *Output
 */
void EEGData::saveToFile(const char *filename,
                         std::ios_base::openmode mode) const
{
  std::ofstream ofs;
  ofs.open(filename, mode);
  if(ofs.fail())
    {
      cerr << "Error opening file: " << filename << " for writing.\n";
      return;
    }
  else
    {
      saveToFile(ofs);
    }
}


void EEGData::saveToFile(ofstream &ofs) const
{
  try
    {
      for(unsigned int row=0; row<data.size1();row++)
        {
          for(unsigned int col=0; col<data.size2(); col++)
          {
            ofs << data(row,col) << " ";
          }
          ofs << "\n";
        }
    }
  catch(...)
    {
      cerr << "Error writing EEGData to file.\n";
    }
}

/**********************************************
 *Input
 */
void EEGData::loadFromFile(const char *filename)
{
  ifstream ifs;
  ifs.open(filename);
  if(ifs.fail())
    {
      throw FileException("Failed to open eeg data file.");
    }
  else
    {
      loadFromFile(ifs);
    }
}


void EEGData::loadFromFile(ifstream &ifs)
{
  try
    {
      int row,col;
      row = 0;col=0;
      int max_col = 0;
      data.resize(1,1);
      //loop through file
      while(!ifs.eof())
        {
          string line;
          getline(ifs,line);
          stringstream ssline(line);
          col = 0;
          while(!ssline.eof())
            {
              string value;
              ssline >> value;
              try {
        	double v = lexical_cast<double>(value);
        	if(data.size2() <= unsigned(col))
        	  {
        	    data.resize(data.size1(),data.size2()*2);
        	  }
        	if(data.size1() <= unsigned(row))
        	  {
        	    data.resize(data.size1()*2,data.size2());
        	  }
        	data(row,col) = v;
        	col++;
        	if(col > max_col)
        	  max_col = col;
              } catch(...)
        	{
        	  continue;
        	}
            }
          row++;
        }
      //cout << "now resizing to " << (row-1) << ", " << (max_col) << "\n";
      data.resize(row-1,max_col);
      //cout << "Loaded: " << (*this) << "\n";
    }
  catch(...)
    {
      throw FileException("Failed to open eeg data file.");
    }
}


//SELECTION METHODS

void EEGData::splitData(EEGData *data1,
        		EEGData *data2,
        		int amount, DataSelectionMethod selection) const
{

  int new_size2 = int(data.size2() * (double(amount)/100.0));
  data1->data.resize(data.size1(),new_size2);
  data2->data.resize(data.size1(),new_size2);

  if(selection == DATA_SELECT_START)
    {
      //selecting from start
      //cout << "selecting from start\n";
      cppR::submatrixAssign(data1->data,
        		    0,data1->data.size1(),
        		    0,new_size2-1,
        		    data,
        		    0,data.size1(),
        		    0,new_size2-1);
      cppR::submatrixAssign(data2->data,
        		    0,data2->data.size1(),
        		    0,new_size2-1,
        		    data,
        		    0,data.size1(),
        		    new_size2,data.size2()-1);
    }
  else if(selection == DATA_SELECT_END)
    {
      //selecting from end
      //cout << "selecting from end\n";
      cppR::submatrixAssign(data2->data,
        		    0,data2->data.size1(),
        		    0,new_size2-1,
        		    data,
        		    0,data.size1(),
        		    0,new_size2-1);
      cppR::submatrixAssign(data1->data,
        		    0,data1->data.size1(),
        		    0,new_size2-1,
        		    data,
        		    0,data.size1(),
        		    new_size2,data.size2()-1);
    }
  else if(selection == DATA_SELECT_MIDDLE)
    {
      //selecting from middle
      //cout << "selecting from middle\n";
      int start_middle, end_middle;
      start_middle = int(.25 * data.size2());
      end_middle = int(.75 * data.size2());
      cppR::submatrixAssign(data2->data,
        		    0,data2->data.size1(),
        		    0,new_size2-1,
        		    data,
        		    0,data.size1(),
        		    0,start_middle-1);
      cppR::submatrixAssign(data1->data,
        		    0,data1->data.size1(),
        		    0,new_size2-1,
        		    data,
        		    0,data.size1(),
        		    start_middle,end_middle-1);
      cppR::submatrixAssign(data2->data,
        		    0,data2->data.size1(),
        		    0,new_size2-1,
        		    data,
        		    0,data.size1(),
        		    end_middle,data.size2()-1);

    }
  else
    {
      //selecting random samples

    }

}

//----------------------------------------------------------------------
//OPERATORS

EEGData& EEGData::operator=(const EEGData& d)
{
  if(this != &d)
    {
      this->data = d.data;
    }
  return *this;
}

void EEGData::print()
{
  std::cout << data;
}


/**********************************************
 *Friends
 */
EEGData operator+(const EEGData &d1, const EEGData &d2)
{
  EEGData d3 = EEGData(d1);
  d3.append(d2);
  return d3;
}

ostream & operator<<(ostream &os, const EEGData &d)
{
  os << "EEGData: " << ublas::matrix<double>(d).size1()
     << " features X " << ublas::matrix<double>(d).size2() << " samples";
  return os;
}
