/*
* CEBL : CSU EEG Brain-Computer Interface Lab
*
* Author: Jeshua Bratman - jeshuabratman@gmail.com
*
* This file is part of CEBL.
*
* CEBL is free software; you can redistribute it and/or modify it.
* We only ask that if you use our code that you cite the source in
* your project or publication.
*
* EEG Group (www.cs.colostate.edu/eeg)
* Department of Computer Science
* Colorado State University
* 
*/

/*!
 * EEGData.hpp
 * \author Jeshua Bratman
 *
 * Class designed to encapsulate the management of a data
 * matrix used specifically for EEG.
 */





#ifndef EEGDATA_H
#define EEGDATA_H

#include "Serialization.hpp"
#include <boost/numeric/ublas/matrix.hpp>
#include <boost/numeric/ublas/vector.hpp>
#include <iostream>
using namespace std;

//! Enum to specify where samples should be selected from.
enum DataSelectionMethod { DATA_SELECT_START,
        		   DATA_SELECT_END,
        		   DATA_SELECT_MIDDLE,
        		   DATA_SELECT_RANDOM };

//define ublas namespace to be shorter
namespace ublas = boost::numeric::ublas;

class EEGData
{
 private:
  //----------------------------------------
  //serialization
  friend class boost::serialization::access;
  template<class Archive>
  void serialize(Archive & ar, const unsigned int version)
  {
    ar & data;
  }

  //! Underlying data container.
  ublas::matrix<double> data;

 public:
  EEGData();
  EEGData(const ublas::matrix<double>&);
  EEGData(const EEGData &data);
  ~EEGData();

  //! append samples by column.
  void append(const EEGData &data);

  //! remove all data.
  void clear();

  //Operators
  operator ublas::matrix<double>() const { return data; };
  EEGData& operator=(const EEGData& data);

  ublas::matrix<double> & getMatrix() { return data; }

  //determine size of data matrix
  //! Determine number of rows or channels.
  int size1() const { return data.size1(); }
  //! Determine number of columns or samples.
  int size2() const { return data.size2(); }

  //! other names for size1 and size 2
  int nrow() const { return size1(); }
  int ncol() const { return size2(); }


  //! Samples are number of columns
  int numSamples() const { return data.size2(); }
  //! Channels are number of rows
  int numChannels() const { return data.size1(); }

  //File Output
  void saveToFile(const char *filename, std::ios_base::openmode mode = std::ios_base::app) const;
  void saveToFile(std::ofstream &ofs) const;

  //File Input
  void loadFromFile(const std::string str) { loadFromFile(str.c_str()); };
  void loadFromFile(const char *filename);
  void loadFromFile(std::ifstream &ifs);

  //Selection Methods
  void splitData(EEGData *, EEGData *, int, DataSelectionMethod) const;

  //printing
  void print();

};
//non-member operators
EEGData operator+(const EEGData &, const EEGData &);
std::ostream& operator<<(std::ostream &os, const EEGData &d);

#endif
