#include "EEGTrainingData.hpp"
#include "cppR/cppR.hpp"
#include <sstream>
#include <exception>

using namespace cppR;
using namespace ublas;
using namespace std;
//constructor
EEGTrainingData::EEGTrainingData()
{
  data.resize(0);
  filtered = false;
}

//destructor
EEGTrainingData::~EEGTrainingData()
{
}

//reserves space for c classes and s sequences
void EEGTrainingData::reserve(unsigned int cls, unsigned int seq)
{
  if(cls > data.size())
    {
      data.resize(cls);
    }

  for(unsigned int c=0;c<data.size();c++)
    if(seq > data.at(c).size())
      {
        data.at(c).resize(seq);
      }
}

//sets data at class c and sequence s
void EEGTrainingData::set(unsigned int cls, unsigned int seq, EEGData &new_data)
{
  if(cls >= unsigned(numClasses()) || seq >= unsigned(numSequences(cls)))
    {
      reserve(cls+1,seq+1);
    }
  this->data[cls][seq] = new_data;
}

//appends data at class c and sequence s
void EEGTrainingData::append(unsigned int cls,
                             unsigned int seq, EEGData &new_data)
{
  if(cls >= unsigned(numClasses()) || seq >= unsigned(numSequences(cls)))
    {
      reserve(cls+1,seq+1);
      this->set(cls,seq,new_data);
    }
  if(this->numSamples(cls,seq)==0)
    {
      this->set(cls,seq,new_data);
    }
  else
    {
      data.at(cls).at(seq).append(new_data);
    }
}
//returns data at location
EEGData& EEGTrainingData::get(unsigned int cls, unsigned int seq)
{
  return data[cls][seq];
}

//returns data at location
const EEGData & EEGTrainingData::getConst(unsigned int c, unsigned int s) const
{
  return data.at(c).at(s);
}

//getters
ublas::matrix<double> EEGTrainingData::collapse() const
{
  EEGData ret;
  for(unsigned int c=0; c<data.size(); c++)
    {
      for(unsigned int s=0; s<data[c].size();s++)
        {
          ret.append(data[c][s]);
        }
    }
  return ret;
}


ublas::vector<int>  EEGTrainingData::getClassVector() const
{
  ublas::vector<int> ret;
  ret.resize(data.size() * data.at(0).size());
  int count = 0;
  for(unsigned int c=0; c<data.size(); c++)
    {
      for(unsigned int s=0; s<data.at(c).size(); s++)
        {
          ret[count++] = c;
        }
    }
  return ret;
}

std::vector<int>  EEGTrainingData::getSequenceOrder() const
{
  return sequence_order;
}

ublas::vector<int>  EEGTrainingData::getTargets() const
{
  // this is likely not the fastest way to do this
  // but it works fine for now

  // find total number of samples in all classes
  unsigned size = 0;
  // for each class
  for (unsigned cls = 0; cls < data.size(); ++cls)
  {
    // for each sequence
    for (unsigned seq = 0; seq < data[cls].size(); ++seq)
    {
      // tally samples
      size += data[cls][seq].size2();
    }
  }

  // vector holding class labels for every sample
  ublas::vector<int> targets(size);

  // current index into our target vector
  unsigned samp_index = 0;

  // for each class
  for (unsigned cls = 0; cls < data.size(); ++cls)
  {
    // for each sequence
    for (unsigned seq = 0; seq < data[cls].size(); ++seq)
    {
      // for each sample
      for (int samp = 0; samp < data[cls][seq].size2(); ++samp)
      {
        // set target value
        targets[samp_index] = cls;
        ++samp_index;
      }
    }
  }

  return targets;
}

int EEGTrainingData::numClasses() const
{
  return data.size();
}

int  EEGTrainingData::numSequences() const
{
  if(data.size() == 0)
    return 0;
  else
    return data.at(0).size();
}

int  EEGTrainingData::numSequences(unsigned int cls) const
{
  if(cls >= unsigned(numClasses()))
    return 0;
  else
    return data.at(cls).size();
}

int EEGTrainingData::samplesInClass(unsigned int cls) const
{
  int num = 0;
  for(unsigned int i=0; i < data.at(cls).size(); i++)
    {
      num += ublas::matrix<double>(data.at(cls).at(i)).size2();
    }
  return num;
}

int EEGTrainingData::numSamples(unsigned int cls,unsigned int seq) const
{
  return this->data.at(cls).at(seq).numSamples();
}

int EEGTrainingData::numSamples() const
{
  int num = 0;
  for(unsigned int i=0; i < data.size(); i++)
    {
      num += samplesInClass(i);
    }
  return num;
}

int EEGTrainingData::numFeatures() const
{
  if(data.size() == 0 || data.at(0).size() == 0)
    return 0;
  else
    return data.at(0).at(0).size1();
}

//*************************************************************/
// Info about data

//! get class labels
std::vector<string> EEGTrainingData::getClassLabels() const
{
  return class_labels;
}

//! get channel names
std::vector<string> EEGTrainingData::getChannelNames() const
{
  return channel_names;
}

//! get filtered flag
bool EEGTrainingData::getFiltered() const
{
  return filtered;
}

//! set class labels
void EEGTrainingData::setClassLabels(std::vector<string> labels)
{
  this->class_labels = labels;
}

//! set channel names
void EEGTrainingData::setChannelNames(std::vector<string> names)
{
  this->channel_names = names;
}

//! set filtered flag
void EEGTrainingData::setFiltered(bool filtered)
{
  this->filtered = filtered;
}

//*************************************************************/

ostream & operator<<(ostream &os, const EEGTrainingData &d)
{
  os << "EEG Training Data: " << d.numClasses() << " classes "
     << d.numSequences() << " sequences, " << d.numSamples() << " samples total.";

  return os;
}
