/*
* CEBL : CSU EEG Brain-Computer Interface Lab
*
* Author: Jeshua Bratman - jeshuabratman@gmail.com
*
* This file is part of CEBL.
*
* CEBL is free software; you can redistribute it and/or modify it.
* We only ask that if you use our code that you cite the source in
* your project or publication.
*
* EEG Group (www.cs.colostate.edu/eeg)
* Department of Computer Science
* Colorado State University
* 
*/

/*! EEGTrainingData.hpp
 * \author Jeshua Bratman
 *
 * EEGTrainingData is a class designed to encapsulate the management
 *  of data arranged into sequences and classes.
 */

#ifndef EEGTRAININGDATA_H
#define EEGTRAININGDATA_H

#include "EEGData.hpp"
#include "Serialization.hpp"

#include <string>
#include <map>
#include <vector>
#include <string>

using std::vector;
using std::string;
using std::map;


class EEGTrainingData
{
 private:

  //----------------------------------------
  //serialization
  friend class boost::serialization::access;
  template<class Archive>
  void serialize(Archive & ar, const unsigned int version)
  {
    ar & data;
    ar & class_vector;
    ar & sequence_order;
    ar & filtered;
    ar & class_labels;
    ar & channel_names;
  }

  //----------------------------------------
  std::vector< std::vector <EEGData> > data;
  ublas::vector<int> class_vector;

  //! the order of sequences performed - a vector of class numbers
  std::vector<int> sequence_order;

  //info about data
  std::vector<string> class_labels;
  std::vector<string> channel_names;
  bool filtered;

 public:
  EEGTrainingData();
  ~EEGTrainingData();
  //! sets data at class and sequence
  void set(unsigned int cls, unsigned int seq, EEGData &data);

  //! adds data to the end of already set data at class and sequence
  void append(unsigned int cls, unsigned int seq, EEGData &data);

  //! returns the data at class and sequence
  EEGData& get(unsigned int cls, unsigned int seq);

  //! get a constant reference to data located at class and sequence
  const EEGData & getConst(unsigned int cls, unsigned int seq) const;

  //! reserve space for certain number of classes and sequence
  void reserve(unsigned int cls, unsigned int seq);

  //! clear's all data
  void clear() { data.clear(); }

  //! collapse sequences and classes into a single matrix of data.
  ublas::matrix<double> collapse() const;

  //! returns a vector representing the order of classes
  ublas::vector<int> getClassVector() const;

  //! returns a vector representing the order of sequences recorded
  std::vector<int> getSequenceOrder() const;

  //! returns a vector of classes for each sample in the collapsed matrix
  ublas::vector<int> getTargets() const;

  //! returns the number of classes.
  int numClasses() const;

  //! returns the number of sequences in first class.
  int numSequences() const;

  //! returns the number of sequences in class.
  int numSequences(unsigned int cls) const;

  //! returns the number of samples in a specific class.
  int samplesInClass(unsigned int cls) const;

  //! returns the number of samples total in a class
  int numSamples(unsigned int cls) const { return samplesInClass(cls); }

  //! returns the number of samples int specific sequence
  int numSamples(unsigned int cls, unsigned int sequence) const;

  //! returns the number of samples total.
  int numSamples() const;

  //! returns the number of features total.
  int numFeatures() const;

  //! get class labels
  std::vector<string> getClassLabels() const;

  //! get channel names
  std::vector<string> getChannelNames() const;

  //! get filtered flag
  bool getFiltered() const;

  //! set class labels
  void setClassLabels(std::vector<string>);

  //! set channel names
  void setChannelNames(std::vector<string>);

  //! set filtered flag
  void setFiltered(bool filtered);

  //! set sequence order vector
  void setSequenceOrder(std::vector<int> ord) { sequence_order = ord; }


  
};

std::ostream& operator<<(std::ostream &os, const EEGTrainingData &d);




#endif
