
/*! FeaturesConfig.cpp
 * \author Jeshua Bratman
 *
 * Finds, loads, and creates features from features shared libraries.
 * Also provides methods to train and use these features.
 */



#include "FeaturesConfig.hpp"
#include "../CEBLModel.hpp"
#include <algorithm>
using namespace std;


//----------------------------------------------------------------------
// CONSTRUCTORS / DESTRUCTORS


FeaturesConfig::FeaturesConfig(CEBLModel *model)
{
  this->model = model;
  this->plugin_loader = new PluginLoader<Feature>;
  this->selected_feature = "";
  std::vector<string> paths = model->preferencesGetPaths();
  for(unsigned int i=0; i<paths.size();i++)
    {
      string path = paths[i]+"plugins/features/";
      try
        {
          plugin_loader->loadDir(path.c_str());
        } catch(FileException & e)
        {
          //cerr << e.what() << "\n";
          //if there were exceptions, just continue
          continue;
        }
    }
  if (plugin_loader->getNames().size() > 0)
    {
      this->selected_feature = this->getNameList().at(0);
    }
}

FeaturesConfig::~FeaturesConfig()
{
  delete plugin_loader;
}

//----------------------------------------------------------------------
//GETTING OPERATIONS

//special sort to put None and Lag at the top
bool featuresSort(string i, string j)
{
  if(i == "None")
    return true;
  else if(j == "None")
    return false;
  else if(i == "Lag")
    return true;
  else if(j == "Lag")
    return false;
  else
    return (i<j);
}

std::vector<string> FeaturesConfig::getNameList()
{
  std::vector<string> names = plugin_loader->getNames();
  sort(names.begin(),names.end(),featuresSort);
  return names;
}

std::vector<string> FeaturesConfig::getPathList()
{
  return plugin_loader->getPaths();
}

bool FeaturesConfig::isTrained(string feature)
{
  if(feature == "")
    feature = this->selected_feature;
  try
    {
      if(plugin_loader->getPlugin(feature) != NULL)
        return plugin_loader->getPlugin(feature)->isTrained();
    }
  catch(...)
    {
      throw PluginException("Failed to get trained status from feature: " + feature);
    }

  return false;
}

string FeaturesConfig::getSelected()
{
  return this->selected_feature;
}

std::map<std::string, CEBL::Param> FeaturesConfig::getParams(string feature)
{
  if(feature == "")
    feature = selected_feature;

  try
    {
      return plugin_loader->getPlugin(feature)->getParamsList();
    }
  catch(...)
    {
      throw PluginException("Failed to get parameter list for feature: " + feature);
    }
}

//----------------------------------------------------------------------
  //SETTING OPERATIONS

void FeaturesConfig::setSelected(string feature)
{
  this->selected_feature = feature;
  if(plugin_loader->getPlugin(feature) == NULL)
    {
      cerr << "ERROR: trying to select feature " << feature 
           << " which doesn't seem to exist.\n " << flush;
    }

}



void FeaturesConfig::train(string feature)
{
  //if the parameter is blank, use the selected feature instead
  if(feature == "")
    feature = this->selected_feature;

  //if the selected feature is also nothing, then don't attempt to train it
  if(feature == "")
    return;
  try
    {
      plugin_loader->getPlugin(feature)->train();
    }
  catch(...)
    {
      throw PluginException("Failed to train: " + feature);
    }

}

void FeaturesConfig::reset(string feature)
{
  //if the parameter is blank, use the selected feature instead
  if(feature == "")
    feature = this->selected_feature;

  //if the selected feature is also nothing, then don't attempt to reset it
  if(feature == "")
    return;
  try
    {
      plugin_loader->getPlugin(feature)->reset();
    }
  catch(...)
    {
      throw PluginException("Failed to reset: " + feature);
    }
}

void FeaturesConfig::setParams(std::map<std::string, CEBL::Param> params, string feature)
{
  //if the parameter is blank, use the selected feature instead
  if(feature == "")
    feature = this->selected_feature;

  //if the selected feature is also nothing, then don't attempt to set params
  if(feature == "")
    return;

  try
    {
      plugin_loader->getPlugin(feature)->setParamsList(params);
    }
  catch(...)
    {
      throw PluginException("Failed to set parameter list for feature: " + feature);
    }
}


//----------------------------------------------------------------------
//EXTRACT FEATURES

EEGTrainingData FeaturesConfig::extract(EEGTrainingData &data)
{
  EEGTrainingData processed_data;
  EEGData temp;
  processed_data.reserve(data.numClasses(),data.numSequences());
  for(int cls = 0; cls < data.numClasses(); cls++)
    {
      for(int seq = 0; seq < data.numSequences(cls); seq++)
        {
          cout << "Featurizing class " << cls << ", seq" << seq << "\n";
          temp = data.get(cls,seq);
          temp = extract(temp);
          processed_data.set(cls,seq,temp);
        }
    }
  return processed_data;
}


EEGData FeaturesConfig::extract(EEGData &data)
{
  string feature = selected_feature;
  EEGData ret;

  if(!isTrained())
    {
      throw FeatureException("Feature is not trained.");
    }
  else
    {
      ublas::matrix<double> d = data;
      ret = EEGData(plugin_loader->getPlugin(feature)->use(d));
   }

  return ret;
}


void FeaturesConfig::halt()
{
  plugin_loader->getPlugin(selected_feature)->halt();
}
