/*! FiltrConfig.cpp
 * \author Jeshua Bratman
 *
 * Finds, loads, and creates filters from filter shared libraries.
 * Also provides methods to train and use these filters.
 */



#include "FilterConfig.hpp"
#include "../CEBLModel.hpp"
#include "../cppR/cppR.hpp"
#include "DataProcess.hpp"

//----------------------------------------------------------------------
// CONSTRUCTORS / DESTRUCTORS


FilterConfig::FilterConfig(CEBLModel *model)
{
  this->model = model;
  this->plugin_loader = new PluginLoader<Filter>;
  this->num_lags = 0;
  this->selected_components_string = "(not set)";
  this->selected_components_valid = false;
  this->num_components = 0;
  this->num_expected_channels = 0;
  this->num_expected_lags = 0;
  this->num_channels = 0;
  this->trained = false;

  std::vector<string> paths = model->preferencesGetPaths();
  for(unsigned int i=0; i<paths.size();i++)
    {
      string path = paths[i] + "plugins/filters/";
      try
        {
          plugin_loader->loadDir(path.c_str());
        } catch(FileException & e)
        {
          //cerr << e.what() << "\n";
          //if there were exceptions, just continue
          continue;
        }
    }
}

//------------------------------

FilterConfig::~FilterConfig()
{
  delete plugin_loader;
}

//----------------------------------------------------------------------
//GETTING OPERATIONS


//------------------------------

std::vector<string> FilterConfig::getNameList()
{
  return plugin_loader->getNames();
}

//------------------------------

std::vector<string> FilterConfig::getPathList()
{
  return plugin_loader->getPaths();
}

//------------------------------

bool FilterConfig::isTrained(string filter)
{
  if(selected_filter == "")
    {
      return false;
    }
  return plugin_loader->getPlugin(selected_filter)->isTrained() && this->trained;
}

//------------------------------

int FilterConfig::getNumLags()
{
  return num_lags;
}

//------------------------------

std::vector<int> FilterConfig::getSelectedComponents()
{
  return selected_components;
}

//------------------------------

EEGData FilterConfig::getComponents(EEGData data)
{
  this->num_channels = data.size1();
  EEGData ret;

  string filter = this->selected_filter;
  if(filter == "")
    {
      throw PluginException("Extracting filtering components: No filter selected.");
    }
  if(data.nrow() == 0 || data.ncol() == 0)
    {
      throw DataException("Extracting filtering components: No data recorded in EEG data buffer.");
    }

  //lag the data
  data = model->getDataProcess()->process(data,true,true,false);
  data = cppR::Lag(data.getMatrix(), num_lags);
  this->num_components = ublas::matrix<double>(data).size1();

  try
    {
      ret = plugin_loader->getPlugin(filter)->extract(data);
    }
  catch(...)
    {
      throw PluginException("Failed to extract components using filter: " + filter);
    }

  return ret;
}

//------------------------------

string FilterConfig::getSelectedComponentsString()
{
  return selected_components_string;
}

//------------------------------

bool FilterConfig::getSelectedComponentsValid()
{
  return selected_components_valid;
}


//------------------------------


EEGData FilterConfig::apply(EEGData data)
{
  if(!this->isTrained())
    {
      throw PluginException("Filter has not been trained.");
    }
  if(data.size1() != this->num_expected_channels)
    {
      throw PluginException("Filter cannot be applied because the number of channels does not match the filter.");
    }

  if(num_lags != this->num_expected_lags)
    {
      throw PluginException("Filter cannot be applied because the number of lags has changed.");
    }
  ublas::matrix<double> filtered = cppR::Lag(data.getMatrix(), num_lags);
  filtered = plugin_loader->getPlugin(this->selected_filter)->apply(filtered);
  //select just the first N rows
  ublas::matrix<double> ret =  cppR::submatrix(filtered,
                                               0, data.size1() - 1,
                                               0,0);
  return ret;
}

//------------------------------

//get the number of channels that the filter was trained on
int FilterConfig::getNumExpectedChannels()
{
  return num_expected_channels;
}


//------------------------------
ublas::matrix<double> FilterConfig::getFilterMatrix()
{
  if(!this->isTrained())
    {
      throw PluginException("Filter has not been trained.");
    }
  return plugin_loader->getPlugin(this->selected_filter)->getFilterMatrix();
}

//----------------------------------------------------------------------
//SETTING OPERATIONS


//------------------------------

void FilterConfig::setNumLags(int lags)
{
  num_lags = lags;
}

//------------------------------

void FilterConfig::setSelected(string filter)
{
  this->selected_filter = filter;
}

//------------------------------

void FilterConfig::train(EEGData training_data, string filter)
{
  this->trained = false;
  std::vector<int> selected_components = this->selected_components;
  for(unsigned int i=0; i<selected_components.size();i++)
    {
      selected_components[i]--;
    }

  if(selected_components[selected_components.size()-1] > num_components
    || selected_components[0] < 0)
    {
      throw PluginException("Training filter failed because selected components are out of range.");
    }

  training_data = model->getDataProcess()->process(training_data,true,true,false);
  this->num_expected_channels = training_data.size1();
  this->num_expected_lags = num_lags;
  training_data = cppR::Lag(training_data.getMatrix(), num_lags);

  if(filter == "")
    filter = this->selected_filter;

  plugin_loader->getPlugin(filter)->make(training_data, selected_components);
  this->trained = true;
}

//------------------------------

void FilterConfig::setSelectedComponentsString(string comp_string)
{
  const char * components = comp_string.c_str();
  std::vector<int> comp;
  int MAX_VALUE = num_channels * (this->num_lags+1);
  num_components = MAX_VALUE;
  int MIN_VALUE = 0;
  int state = 0;
  int location = 0;
  int buffer_loc = 0;
  bool cont = true; //is the string still valid?
  bool end = false; //have we reached end of string?
  char buffer[25] = "";
  char current;
  int colon_start = 0;

  while(cont && !end)
    {
      //what state are we in?
      switch(state)
        {
          //init state - after a comma or at the beginning of string
          //----------------------------------------
        case 0:
          current = components[location++];
          //ignore whitespace
          if(current == ' ')
            break;


          if(current >= '0' && current <= '9')
            {
              buffer[buffer_loc++] = current;
              //we are in integer reading state now
              state = 1;
            }
          else
            {
              cont = false;//failed

            }
          break;

          //----------------------------------------
          // reading integer
        case 1:
          //consume a character
          current = components[location++];
          if(current == ' ')
            break;

          //if current character is an integer add it to the buffer
          //and go back to integer state
          if(current >= '0' && current <= '9')
            {
              buffer[buffer_loc++] = current;
              //we are in integer reading state now
              state = 1;
            }
          else if(current == ',')
            {
              buffer[buffer_loc] = '\0';
              int val = atoi(buffer);
              if(val > MAX_VALUE || val < MIN_VALUE)
                {
                  cont = false;
                  break;
                }
              comp.push_back(val);
              buffer_loc = 0;
              state = 0;
            }
          else if(current == ':')
            {
              colon_start = atoi(buffer);
              if(colon_start > MAX_VALUE || colon_start < MIN_VALUE)
                {
                  cont = false;
                  break;
                }
              buffer_loc = 0;
              state = 2;
            }
          else if(current == '\0')
            {
              buffer[buffer_loc] = '\0';
              int val = atoi(buffer);
              if(val > MAX_VALUE || val < MIN_VALUE)
                {
                  cont = false;
                  break;
                }
              comp.push_back(val);
              buffer_loc = 0;
              end = true;
            }
          else
            {
              cont = false;//failed
            }
          break;
          //----------------------------------------
          //reading a ':'
        case 2:
          //consume a character
          current = components[location++];
          //ignore whitespace
          if(current == ' ')
            break;

          //must read an integer from here
          if(current >= '0' && current <= '9')
            {
              buffer[buffer_loc++] = current;
              //put us in the :integer state
              state = 3;
            }
          else
            {
              cont = false;//nothing other than an integer can follow a colon
            }
          break;
          //reading integer after colon
        case 3:
          //consume a character
          current = components[location++];
          //ignore whitespace
          if(current == ' ')
              break;

          //if current character is an integer add it to the buffer
          //and go back to integer state
          if(current >= '0' && current <= '9')
            {
              buffer[buffer_loc++] = current;
              //we are in integer reading state now
              state = 3;
            }
          else if(current == ',')
            {
              buffer[buffer_loc] = '\0';
              int val = atoi(buffer);
              if(val > MAX_VALUE || val < MIN_VALUE )
                {
                  cont = false;
                  break;
                }
              if(val > colon_start)
                for(int i=colon_start; i<= val; i++)
                  comp.push_back(i);
              else
                for(int i=val; i<= colon_start; i++)
                  comp.push_back(i);
              buffer_loc = 0;
              state = 0;
            }
          else if(current == '\0')
            {
              buffer[buffer_loc] = '\0';
              int val = atoi(buffer);
              if(val > MAX_VALUE || val < MIN_VALUE)
                {
                  cont = false;
                  break;
                }
              if(val > colon_start)
                for(int i=colon_start; i<= val; i++)
                  comp.push_back(i);
              else
                for(int i=val; i<= colon_start; i++)
                  comp.push_back(i);
              buffer_loc = 0;
              end = true;
            }
          else
            {
              cont = false;//failed
            }
          break;
        default:
          cont = false;
          break;
        }
    }

  //save the resulting list of integers

  if(cont)
    {

      //sort and pick unique components
      std::sort(comp.begin(), comp.end());
      std::vector<int>::iterator new_end =
        std::unique(comp.begin(), comp.end());
      comp.erase(new_end, comp.end());

      //create a string for the components
      stringstream s;
      this->selected_components.clear();
      for(unsigned int i=0; i < comp.size()-1; i++)
        {
          this->selected_components.push_back(comp[i]);
          s << comp[i] << ", ";
        }
      this->selected_components.push_back(comp[comp.size()-1]);
      s << comp[comp.size()-1];

      //check that there are an okay number of selected components
      bool comp_num_okay = false;
      if(this->selected_components.size() <= unsigned(MAX_VALUE)
         && this->selected_components.size() > 0)
        {
          if(this->selected_components[0] > 0
             && this->selected_components[0] <= MAX_VALUE
             && this->selected_components[this->selected_components.size()-1] > 0
             && this->selected_components[this->selected_components.size()-1] <= MAX_VALUE)
            {
              comp_num_okay = true;
            }
        }
      this->selected_components_valid = comp_num_okay;
      if(comp_num_okay)
        {
          this->selected_components_string = s.str();
        }
      else
        {
          this->selected_components_string = "(value out of range)";
        }
    }
  else
    {
      this->selected_components_valid = false;
      this->selected_components_string = "(error parsing input)";
    }
}
