/*!
 * Mindset24.hpp
 * \author Jeshua Bratman
 *
 * Driver to interface with EEG Amplifier
 */



#include "Mindset24.hpp"
#include "EEGData.hpp"
#include <string.h>
#include <sstream>
#include <iostream>
using namespace std;
/*****************************************************************************/
//constructors/destructors

Mindset24::Mindset24()
{
  Init();
}


void Mindset24::Init()
{
  debug = false;
  mindset_fp = -1;
  is_open = false;
  is_mindset = false;
}

Mindset24::~Mindset24()
{
  Close();
}


/*****************************************************************************/
//mindset file management

bool Mindset24::Open(const char * filename)
{
  if(filename == NULL)
    {
      return false;
    }
  strcpy(this->filename,filename);
  mindset_fp = open(filename,O_RDWR);
  is_open = mindset_fp >= 0;
  CheckInquiry();
  return is_open;
}

//close mindset
void Mindset24::Close()
{
  if(is_open)
    try{
      SendCommand(SETSAMPLERATE,0);
    } catch(...){}
  if(mindset_fp >= 0)
    close(mindset_fp);
}

//is the mindset ready?
bool Mindset24::Ready()
{
  //if mindset isn't opened it obviously isn't ready
  if(!is_open)
    return false;
  //if device isn't a mindset return false
  if(!is_mindset)
    return false;
  //return status from io_hdr
  bool status;
  try
    {
      status =  SendCommand(READY).io_hdr.masked_status==0;
    }
  catch(...)
    {
      status = false;
    }
  return status;
}

/*****************************************************************************/
//set mindset value

void Mindset24::SetSampleRate(SampleRate s)
{

  SendCommand(SETSAMPLERATE,s);

}

void Mindset24::SetBlockSize(BlockSize b)
{
  //int data_bytes_length = ActualBlockSize(b);
  //unsigned char *dataBytes = (unsigned char *)malloc(dataBytes_length);
  SendCommand(SETBLOCKSIZE, b);

}


/*****************************************************************************/
//getters:

SampleRate Mindset24::GetSampleRate()
{
  return GetStatus().sample_rate;
}

BlockSize Mindset24::GetBlockSize()
{
  return GetStatus().block_size;
}






/*****************************************************************************/
//Private methods

//get status of mindset
MindsetStatus Mindset24::GetStatus()
{
   MindsetReply mr = SendCommand(READSTATUS);
   unsigned char *reply = mr.reply;
   MindsetStatus status;

   status.sample_rate = (SampleRate) reply[0];
   status.block_size = (BlockSize) reply[5];

   //determine num data bytes
   static const int DeviceBufferBegin = 0x2000;
   static const int DeviceBufferEnd   = 0x9DFF;
   unsigned short shortHead;
   unsigned short shortTail;
   int numBytes;

   shortHead = reply[1] << 8 | reply[2];
   shortTail = reply[3] << 8 | reply[4];

   if (shortHead <= shortTail)
     numBytes = shortTail - shortHead;
   else
     numBytes = shortTail - DeviceBufferBegin + DeviceBufferEnd - shortHead;

   status.num_data_bytes = numBytes;

   return status;
}

void Mindset24::CheckInquiry()
{
  if(!is_open)
    return;
  MindsetReply mr = SendCommand(INQUIRE, 0, 0x24);
  unsigned char *dest = mr.reply;
  char product[31-15+1];
  memcpy(product,dest+16,16);product[16]='\0';


  delete[] mr.reply;
  delete[] mr.sense_buffer;

  is_mindset = strstr(product,"MINDSET") !=NULL;
}


std::string Mindset24::Inquiry()
{
  if(!is_open)
    return "Error - Mindset Not Opened";

  std::stringstream ss;
  MindsetReply mr = SendCommand(INQUIRE, 0, 0x24);

  unsigned char *dest = mr.reply;
  char vendor[15-7+1];
  char product[31-15+1];
  char firmware[35-31+1];
  memcpy(product,dest+16,16);product[16]='\0';
  memcpy(vendor,dest+8,8);vendor[8]='\0';
  memcpy(firmware,dest+32,4);firmware[4]='\0';

  ss << "Product ID:\t\t" << product  << std::endl;
  ss << "Vendor ID:\t\t" << vendor  << std::endl;
  ss << "Firmware version:\t" << firmware  << std::endl;
  ss << "Device type:\t\t" << int(dest[0]) << std::endl;
  ss << "Type Qualifier:\t\t" << int(dest[1]) << std::endl;
  ss << "ANSI Version:\t\t" << int(dest[2]) << std::endl;
  ss << "Additional Length:\t" << int(dest[4]) << std::endl;
  ss << "Firmware Year:\t\t" << int(dest[5]);



  delete[] mr.reply;
  delete[] mr.sense_buffer;

  return ss.str();
}

/*****************************************************************************/
//reading data

EEGData Mindset24::GetAllData()
{
  ublas::matrix<double> ret;
  int bytes_available;
  BlockSize bsize;
  int bytes_per_block;
  MindsetStatus status = GetStatus();

  bytes_available = status.num_data_bytes;
  bsize = status.block_size;
  bytes_per_block = ActualBlockSize(bsize);

  //TODO: WHY DOES BUFFER HABE 30000 SAMPLES IN IT SOMETIMES?
  /*check if there is less than one block to get
    or the buffer got corrupted*/
  if(bytes_available < bytes_per_block || bytes_available > 30000)
    {
      return ret;
    }

  //determine how many full blocks are available
  int output_len;
  if(bytes_per_block != 0)
    output_len = (int)(bytes_available/bytes_per_block) * bytes_per_block;
  else
    throw "Error reading from mindset.";

  if(debug)
    printf("DEBUG: inside GetAllData(), bytes_available = %d, bytes_per_block = %d, output_len = %d\n",bytes_available, bytes_per_block, output_len);

  MindsetReply mr = SendCommand(READDATA, output_len/bytes_per_block, output_len);


  //don't try to create the matrix if there was an error
  if(mr.error)
    return ret;

  ret = (CreateEEGMatrix(mr));
  delete[] mr.reply;
  delete[] mr.sense_buffer;

  return ret;
}

EEGData Mindset24::GetNextDataBlock()
{
  int bytes_available;
  BlockSize bsize;
  int bytes_per_block;

   MindsetStatus status = GetStatus();
   bytes_available = status.num_data_bytes;
   bsize = status.block_size;
   bytes_per_block = ActualBlockSize(bsize);

   /* Wait until enough bytes available to fill one block */
   while (bytes_available < bytes_per_block)
     {
       status = GetStatus();
       bytes_available = status.num_data_bytes;
       bsize = status.block_size;
       bytes_per_block = ActualBlockSize(bsize);
     }

   ublas::matrix<double> ret(0,0);
   MindsetReply mr = SendCommand(READDATA, 1, bytes_per_block);
   if(mr.error)
     return ret;

   ret =  CreateEEGMatrix(mr);
   delete[] mr.reply;
   delete[] mr.sense_buffer;

   return ret;
}

/*****************************************************************************/

int Mindset24::ActualBlockSize(BlockSize b)
{
  switch(b)
    {
    case BLOCKSIZE96: return 96;
    case BLOCKSIZE192: return 192;
    case BLOCKSIZE384: return 384;
    case BLOCKSIZE768: return 768;
    }
  return 0;
}



ublas::matrix<double> Mindset24::CreateEEGMatrix(const MindsetReply &mr)
{
  int numBytes = mr.reply_len;
  int nsamples = numBytes / 24 / 2;
  int chan = 0;
  int b = 0;
  int v = 0;
  int sample = 0;
  ublas::matrix<double> ret_matrix;
  if(debug)
    printf("DEBUG: inside CreateEEGMatrix, nsamples = %d, ret_matrix size1 = %d, reply_len = %d\n",
           nsamples,
           int(ret_matrix.size1()),
           int(mr.reply_len));

  //ret_matrix.reserve(24,nsamples);
  ret_matrix.resize(24,nsamples);
  for (sample = 0; sample < nsamples; sample++)
    for (chan = 0; chan < 24; chan++)
      {
        ret_matrix(chan, sample) = ((mr.reply[b] << 8 | mr.reply[b+1]) - 0x7000) / 363.63;
        b += 2;
        v++;
      }
  return(ret_matrix);
}







/*****************************************************************************/
//SendCommand
//sends a command to the mindset
MindsetReply Mindset24::SendCommand(MindsetCommand command, int value, int bytes)
{
  MindsetReply ret;
  bool debug = false;
  memset(&ret,0,sizeof(MindsetReply));
  ret.error = true;
  ret.reply_len = bytes;
  if(!is_open)
    throw("Failed to read from mindset.");

  sg_io_hdr_t io_hdr;

  static unsigned char commandBlock[6];
  memset(&commandBlock,0,6);

  //create generic scsi io header for ioctl
  ret.reply = new unsigned char[ret.reply_len];
  ret.sense_buffer = new unsigned char[SENSE_BUF_LEN];
  unsigned char *reply = ret.reply;
  unsigned char *sense_buffer = ret.sense_buffer;

  memset(&io_hdr,0,sizeof(sg_io_hdr_t));
  io_hdr.interface_id = 'S';
  io_hdr.mx_sb_len = SENSE_BUF_LEN;
  io_hdr.cmd_len = sizeof(commandBlock);
  io_hdr.dxfer_len = bytes;
  io_hdr.dxferp = reply;
  io_hdr.cmdp = commandBlock;
  io_hdr.sbp = sense_buffer;
  io_hdr.timeout = 5000;

  //build command
  switch(command)
    {
    case INQUIRE:
      commandBlock[0] = 0x12;
      commandBlock[4] = bytes;
      io_hdr.dxfer_direction = SG_DXFER_FROM_DEV;
      break;
    case READDATA:
      commandBlock[0] = 0xc0;
      commandBlock[1] = 0x04;
      commandBlock[2] = value;
      io_hdr.dxfer_direction = SG_DXFER_FROM_DEV;
      break;
    case READSTATUS:
      commandBlock[0] = 0xc0;
      commandBlock[1] = 0x06;
      commandBlock[4] = 0x1e;
      io_hdr.dxfer_direction = SG_DXFER_FROM_DEV;
      break;
    case READY:
      commandBlock[0] = 0x00;
      io_hdr.dxfer_direction = SG_DXFER_FROM_DEV;
      break;
    case SETSAMPLERATE:
      commandBlock[0] = 0xc0;
      commandBlock[1] = 0x03;
      commandBlock[2] = value; // Sample rate
      io_hdr.dxfer_direction = SG_DXFER_TO_DEV;
      break;
    case SETBLOCKSIZE:
      commandBlock[0] = 0xc0;
      commandBlock[1] = 0x05;
      commandBlock[2] = value; // Block size
      io_hdr.dxfer_direction = SG_DXFER_TO_DEV;
      break;
    default:
      printf("Mindset SCSI: Incorrect command %d sent to Mindset.\n",command);
    }

  //write command using ioctl
  if(ioctl(mindset_fp, SG_IO, &io_hdr) < 0)
    {
      throw("Mindset SCSI: ioctl error\n");
    }
  // now for the error processing
  if(debug){
    if ((io_hdr.info & SG_INFO_OK_MASK) != SG_INFO_OK)
      {
        printf("Mindset SCSI: io_hdr.info & SG_INFO_OK_MASK) != SG_INFO_OK\n");
        if (io_hdr.sb_len_wr > 0)
          {
            //ms_dump(sense_buffer,io_hdr.sb_len_wr,"Mindset SCSI sense data: ");
          }
        if (io_hdr.masked_status)
          printf("Mindset SCSI status=0x%x\n", io_hdr.status);
        if (io_hdr.host_status)
          printf("Mindset SCSI host_status=0x%x\n", io_hdr.host_status);
        if (io_hdr.driver_status)
          printf("Mindset driver_status=0x%x\n", io_hdr.driver_status);
        else {
          //ms_dump(reply,REPLY_STRING_LENGTH,"Mindset SCSI reply");
          printf("Mindst SCSI command duration=%u millisecs, resid=%d\n",
        	 io_hdr.duration, io_hdr.resid);
        }
      }
    }

  //save io_hdr into mindset reply
  ret.io_hdr = io_hdr;

  //return mindset reply
  ret.error = false;
  return ret;
}
