/*
* CEBL : CSU EEG Brain-Computer Interface Lab
*
* Author: Jeshua Bratman - jeshuabratman@gmail.com
*
* This file is part of CEBL.
*
* CEBL is free software; you can redistribute it and/or modify it.
* We only ask that if you use our code that you cite the source in
* your project or publication.
*
* EEG Group (www.cs.colostate.edu/eeg)
* Department of Computer Science
* Colorado State University
* 
*/

/*!
 * Mindset24.hpp
 * \author Jeshua Bratman
 *
 * Driver to interface with EEG Amplifier
 */

#ifndef MINDSET24_H
#define MINDSET24_H


#include <sys/ioctl.h>
#include <fcntl.h>
#include <scsi/sg.h>
#include <boost/numeric/ublas/matrix.hpp>
#include <boost/numeric/ublas/vector.hpp>
#include <string>
using std::string;

namespace ublas = boost::numeric::ublas;

class EEGData;

#define SENSE_BUF_LEN 32
#define REPLY_STRING_LENGTH 32
#define REPLY_STRING_MAX 4096

//! size of blocks recieved in bytes
enum BlockSize {
  BLOCKSIZE96=1,
  BLOCKSIZE192=2,
  BLOCKSIZE384=4,
  BLOCKSIZE768=8
};

//! sample rate -- samples/sec
enum SampleRate {
  SAMPLERATE0=0,
  SAMPLERATE1024=1,
  SAMPLERATE512=2,
  SAMPLERATE256=3,
  SAMPLERATE128=4,
  SAMPLERATE64=5
};

//! commands to send to mindset
enum MindsetCommand {
  INQUIRE,
  READDATA,
  READSTATUS,
  READY,
  SETSAMPLERATE,
  SETBLOCKSIZE
};

//! structure containing mindset's reply to command
struct MindsetReply
{
  unsigned char *reply;
  unsigned char *sense_buffer;
  sg_io_hdr_t io_hdr;
  bool error;
  int reply_len;
  int sense_buffer_len;
};

//! structure containing the status of the mindset
struct MindsetStatus
{
  SampleRate sample_rate;
  BlockSize block_size;
  int num_data_bytes;
};


//! driver class to interface with mindset
class Mindset24
{

private:
  MindsetReply SendCommand(MindsetCommand command, int value = 0, int bytes = REPLY_STRING_LENGTH);

  ublas::matrix<double> CreateEEGMatrix(const MindsetReply&);
  int ActualBlockSize(BlockSize b);
  void CheckInquiry();
  //private members
  bool is_open;
  bool is_mindset;
  int mindset_fp;
  char filename[255];
  bool debug;


public:

  Mindset24();
  void Init();
  ~Mindset24();

  //mindset functions
  bool Open(const char * filename);
  void Close();
  bool Ready();
  bool IsOpen() { return is_open; };
  SampleRate GetSampleRate();
  BlockSize GetBlockSize();
  void SetSampleRate(SampleRate);
  void SetBlockSize(BlockSize);
  std::string Inquiry();
  bool IsMindset() { return is_mindset; };
  MindsetStatus GetStatus();

  //data reading
  EEGData GetAllData();
  EEGData GetNextDataBlock();
};

#endif
