/*!
 * MindsetStream.hpp
 * \author Jeshua Bratman
 *
 * An EEG DataStream to generate random data.
 */


#include "../CEBLModel.hpp"
#include "DeviceConfig.hpp"
#include "Mindset24.hpp"
#include "MindsetStream.hpp"
#include <cppR/cppR.hpp>

#include <iostream>
using namespace std;
//----------------------------------------------------------------------
// CONSTRUCTORS / DESTRUCTORS

MindsetStream::MindsetStream(CEBLModel *model)
{
  this->model = model;
}
MindsetStream::~MindsetStream()
{}


//----------------------------------------------------------------------

void MindsetStream::updater()
{

  //! make sure the mindset is still ready to record
  if(!mindset.Ready())
    {
      throw DeviceException("Lost connection to mindset during recording.");
    }

  ublas::matrix<double> data;
  try
    {
      data = mindset.GetAllData();
    }
  catch(const char *m)
    {
      throw DeviceException("Exception caught when trying to read data from mindset: " + string(m));
    }
  {
    boost::mutex::scoped_lock lock(thread_lock);
    buffer.append(data);

  }

}

void MindsetStream::onStart()
{
  // make sure device is ready
  if(!model->getDeviceConfig()->isReady())
    {
      throw DeviceException("Mindset is not ready.");
    }
  // open up the device
  mindset.Open(model->getDeviceConfig()->getLocation().c_str());

  // start it
  int bs = model->getDeviceConfig()->getBlockSize();
  int sr = model->getDeviceConfig()->getSampleRate();
  BlockSize block_size;
  switch(bs)
    {
    case 1:
      block_size = BLOCKSIZE96;
      break;
    case 2:
      block_size = BLOCKSIZE192;
      break;
    case 3:
      block_size = BLOCKSIZE384;
      break;
    case 4:
      block_size = BLOCKSIZE768;
      break;
    default:
      block_size = BLOCKSIZE192;
    }
  SampleRate sample_rate;
  switch(sr)
    {
    case 0:
      sample_rate = SAMPLERATE0;
      break;
    case 1:
      sample_rate = SAMPLERATE1024;
      break;
    case 2:
      sample_rate = SAMPLERATE512;
      break;
    case 3:
      sample_rate = SAMPLERATE256;
      break;
    case 4:
      sample_rate = SAMPLERATE128;
      break;
    case 5:
      sample_rate = SAMPLERATE64;
      break;
    default:
      sample_rate = SAMPLERATE256;
    }


  mindset.SetBlockSize(block_size);
  mindset.SetSampleRate(sample_rate);
}

void MindsetStream::onStop()
{
  mindset.Close();
}




