/*
* CEBL : CSU EEG Brain-Computer Interface Lab
*
* Author: Jeshua Bratman - jeshuabratman@gmail.com
*
* This file is part of CEBL.
*
* CEBL is free software; you can redistribute it and/or modify it.
* We only ask that if you use our code that you cite the source in
* your project or publication.
*
* EEG Group (www.cs.colostate.edu/eeg)
* Department of Computer Science
* Colorado State University
* 
*/


/*! PluginLoader.hpp
 * \author Jeshua Bratman
 *
 * Class to find, load, and organize plugins.
 */

#ifndef PLUGINLOADER_H
#define PLUGINLOADER_H


#include "SharedLoader.hpp"
#include "Serialization.hpp"
#include "FileUtils.hpp"
#include "../Exceptions.hpp"


//std includes
#include <map>
#include <string>
#include <sstream>
#include <vector>

//boost includes
#define BOOST_FILESYSTEM_VERSION 2
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/path.hpp>


//namespaces
using namespace std;
namespace fs = boost::filesystem;


template <typename T>
class PluginLoader
{

private:
  //list of shared loaders organized by filename
  map<string, SharedLoader<T> > libs;
  //list of instantiated plugins organized by filename
  map<string, T* > plugins;
  //list of paths associated with plugin filenames
  map<string, string> paths;

public:
  PluginLoader(){};
  ~PluginLoader();

  //! load plugins from directories
  void loadDirs(std::vector<string> dirs) {
    for(int i=0; i<dirs.size(); i++)
      {
        loadDir(dirs[i]);
      }
  };
  //! load plugins from single directory
  void loadDir(string dir);

  //access loaded plugins
  map<string, SharedLoader<T> > & getPluginsMap() { return libs; }
  map<string, string > & getPathsMap() { return paths; }

  vector<SharedLoader<T> > & getPlugins();
  vector<string> getPaths();
  vector<string> getFileNames();
  vector<string> getNames();
  string getFilename(string plugin_name);

  string getPath(string lib);
  string getPathByFilename(string lib);

  //gets shared loader for specified lib
  SharedLoader<T> getLoader(string lib) { return libs[lib]; }

  //get copy of library for specified plugin name
  T* getPlugin(string plugin);
  //get copy of library for specified plugin filename
  T* getPluginByFilename(string plugin);

};

//DESTRUCTOR
template <typename T>
PluginLoader<T>::~PluginLoader()
{
    typedef typename std::map<string, T* >::const_iterator mit; //required for templated iterators
    for(mit it = plugins.begin(); it != plugins.end(); ++it)
      {
        if(it->second != NULL)
          {
            delete it->second;
          }
      }
}


//scans a directory for all shared libraries and loads them
template <typename T>
void PluginLoader<T>::loadDir(string directory)
{
  fs::path full_path;
  full_path = fs::system_complete(fs::path(directory, fs::native));


  //make sure directory exists
  if(!fs::exists(full_path))
    {
      throw FileException(string("Directory does not exist: ") + full_path.native_file_string());
    }
  //make sure it is a directory
  if(!fs::is_directory(full_path))
    {
      throw FileException(full_path.native_file_string() + string(" is not a directory. "));
    }


  //loop through directory
  fs::directory_iterator end_iter;
  for (fs::directory_iterator dir_itr( full_path );
       dir_itr != end_iter;
       ++dir_itr )
    {
      try
        {
          string file = dir_itr->leaf();

          if(file.length() > 3)
            {
              if(file.substr(file.length()-3,3) == ".so")
        	{
        	  string name = file.substr(0,file.length()-3);
        	  string full_name = full_path.native_file_string() + file;
        	  libs[name].LoadLibrary(full_name.c_str());
        	  paths[name] = directory;
        	  if(libs[name].Loaded())
        	    {
        	      cout << "Loaded plugin " << name << ".so" << endl;
        	      plugins[name] = libs[name].New();
        	    }
        	  else
        	    {
        	      //remove entry
        	      libs.erase(name);
        	    }
        	}
            }
        }
      catch(const std::exception & ex)
        {
          throw FileException("Failed to load shared libraries.");
        }
    }

}
//get vectors of
template <typename T>
vector<string> PluginLoader<T>::getPaths()
{
  map<string, string>::iterator it;
  vector<string> ret;
  for(it=paths.begin(); it != paths.end(); ++it)
    {

      ret.push_back(it->second);
    }

  return ret;

}

template <typename T>
vector<string> PluginLoader<T>::getFileNames()
{
  vector<string> ret;

  typedef typename std::map<string, SharedLoader<T> >::const_iterator mit; //required for templated iterators
  for(mit it = libs.begin(); it != libs.end(); ++it)
    {
      ret.push_back(it->first);
    }

  return ret;

}
//! get list of plugin names
template <typename T>
vector<string> PluginLoader<T>::getNames()
{
  vector<string> ret;

  typedef typename std::map<string, T* >::const_iterator mit; //required for templated iterators
  for(mit it = plugins.begin(); it != plugins.end(); ++it)
    {
      ret.push_back(it->second->getName());
    }

  return ret;

}



//! gets filename of plugin by its name
template <typename T>
string PluginLoader<T>::getFilename(string name)
{
  typedef typename std::map<string, T* >::const_iterator mit; //required for templated iterators
  for(mit it = plugins.begin(); it != plugins.end(); ++it)
    {
      if(it->second->getName() == name)
        {
          return it->first;
        }
    }
  return "";
}


//! get plugin by its filename
template <typename T>
T* PluginLoader<T>::getPluginByFilename(string name)
{
  if(plugins.count(name) > 0)
    {
      return plugins[name];
    }
  else
    {
      return NULL;
    }
}


//! gets plugin path by its filename
template <typename T>
string PluginLoader<T>::getPathByFilename(string name)
{
  if(paths.count(name) > 0)
    {
      return paths[name];
    }
  else
    {
      return "";
    }
}

//! gets plugin by its name
template <typename T>
T* PluginLoader<T>::getPlugin(string name)
{
  return getPluginByFilename(getFilename(name));
}

//! gets plugin path by its name
template <typename T>
string PluginLoader<T>::getPath(string name)
{
  return getPathByFilename(getFilename(name));
}

#endif

