/*! Preferences.hpp
 * \author Jeshua Bratman
 *
 * Loads and stores CEBL preferences.
 */

//local includes
#include "../CEBLIncludes.hpp"
#include "Preferences.hpp"
#include "../TextUtils.hpp"
#include "../Exceptions.hpp"

//std includes
#include <stdio.h>
#include <stdlib.h>
#include <fstream>
#include <exception>
#include <iostream>

//boost filesystem and program options includes
#define BOOST_FILESYSTEM_VERSION 2
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/path.hpp>


//namespaces
using namespace std;
namespace fs = boost::filesystem;

// print out vector
template<class T>
ostream& operator<<(ostream& os, const vector<T>& v)
{
  copy(v.begin(), v.end(), ostream_iterator<T>(os, " "));
  return os;
}

//Constructor
Preferences::Preferences()
{
  have_conf_dir = true;
  session_filename = "";
  default_channels_filename = string(HOMEDIR) + "/.cebl/conf/channels-19.conf";
  string_table_filename = string(HOMEDIR) + "/.cebl/conf/string-table-en.txt";
}

//Destructor
Preferences::~Preferences()
{

}

//process command line
bool Preferences::processCL(int ac, char ** av)
{
  session_filename = "";
  try {

    desc.add_options()
      ("help", "Print Help")
      ("version", "Print Version String")
      ;

    po::options_description hidden("Hidden options");
    hidden.add_options()
      ("session-file", po::value< vector<string> >(), "session file")
      ;

    //----------

    po::options_description visible("CEBL Options");
    visible.add(desc).add(hidden);

    po::positional_options_description p;
    p.add("session-file", -1);

    po::variables_map vm;
    po::store(po::command_line_parser(ac, av).
              options(visible).positional(p).run(), vm);
    po::notify(vm);


    if (vm.count("help"))
      {
        cout << desc << "\n";
        return false;
      }
    if (vm.count("version"))
      {
        cout << PACKAGE_STRING << endl;
        return false;
      }
    if (vm.count("session-file"))
        {
          session_filename = vm["session-file"].as< vector<string> >()[0];
        }

    // read all other options from the option list
    for(unsigned i=0;i<option_list.size();i++)
      {
        string name = option_list[i];
        set_options[name] = "None";
        if(vm.count(name))
          {
            set_options[name] = vm[name].as< vector<string> >()[0];
          }
      }
  }
  catch(exception& e) {
    cerr << "error: " << e.what() << "\n";
    return 1;
  }
  catch(...) {
    cerr << "Exception of unknown type!\n";
  }
  return true;
}



//Iniialize Preferences
bool Preferences::initCEBL()
{

  //set members from env
  char *pref_path = NULL;
  bool pref_path_specified = true;
  pref_path = getenv("CEBL_PREF_DIR");
  if(pref_path==NULL)
    pref_path_specified = false;


  if(!pref_path_specified)
    {
      string home_path = string(HOMEDIR);
      if(home_path=="")
        {
          throw FileException("Failed to open home directory.");
        }
      conf_dir = home_path+"/.cebl/";
    }
  else
    {
      conf_dir = pref_path;
    }

  //check if this is first-run
  //check for ini directory
  fs::path full_path;
  full_path = fs::system_complete(fs::path(conf_dir, fs::native));
  //make sure directory exists
  if(!fs::exists(full_path))
    {
      if(false)
        {
         have_conf_dir = false;
         return true;
        }
      //create config directory
      else
        {
          //try creating the directory
          try
            {
              fs::create_directory(full_path);
              fs::create_directory(fs::system_complete(fs::path(conf_dir+"/plugins", fs::native)));
              fs::create_directory(fs::system_complete(fs::path(conf_dir+"/plugins/classifiers", fs::native)));
              fs::create_directory(fs::system_complete(fs::path(conf_dir+"/plugins/filters", fs::native)));
              fs::create_directory(fs::system_complete(fs::path(conf_dir+"/plugins/features", fs::native)));

            }
          catch(...)

            {
              //Confirm("Could Not Create Directory","Could not create directory "+conf_dir+". Continue without being able to save peferences?",GTK_MESSAGE_ERROR)
              throw FileException("Could not create cebl configuration directory.");
            }
        }
    }
  //make sure it is a directory
  if(!fs::is_directory(full_path))
    {
      throw FileException(conf_dir+" is not a directory. Please remove the file "+conf_dir+".");
    }

  //--------------------------
  //directory exists for sure
  string default_path = static_cast<string>(DATAROOTDIR)+"/"+PACKAGE;
  // paths.push_back(default_path);

  paths.push_back(conf_dir);

  string conf_file = conf_dir+"/cebl.conf";
  full_path = fs::system_complete(fs::path(conf_file, fs::native));
  if(!fs::exists(full_path))
    {
      //need to create conf file
      try
        {

        }
      catch(...)
        {
          throw FileException("Failed to create initial configuration file.\n");
        }
    }
  else
    {
      try
        {
          po::options_description config("Configuration");
          config.add_options()
            ("path",
             po::value< vector<string> >()->composing(),
             "include path")
            ;

          po::options_description config_file_options;
          config_file_options.add(config);


          po::variables_map vm;

          ifstream ifs(conf_file.c_str());
          store(parse_config_file(ifs, config_file_options), vm);
          notify(vm);
          //save options
          if(vm.count("path"))
            {
              cout << "Path = " << vm["path"].as< string >();
            }
        }
      catch(exception& e)
        {
          cout << e.what() << "\n";

        }

    }

  return true;
}


string Preferences::getDefaultChannelsFilename()
{
  return this->default_channels_filename;
}


void Preferences::addOption(string name, string description, char short_name)
{
  string full_name = name;
  if(short_name != '\0')
    full_name = full_name+","+short_name;

  option_list.push_back(name);
  desc.add_options()
    (full_name.c_str(), po::value< vector<string> >(), description.c_str());
}

string Preferences::getOption(string name)
{
  return set_options[name];
}
