#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <iostream>
#include <sstream>
#include <boost/regex.hpp>
#include <boost/lexical_cast.hpp>

#include "RController.hpp"

using namespace std;

namespace CEBL
{
  RController::RController()
  {
    this->child_pid = -1;
    this->connected = false;
    this->read_pipe = -1;
    this->write_pipe = -1;
    this->cleared = false;
    this->temp_filename = "";
  }

  RController::~RController()
  {
    this->rProcessKill();
    if(temp_filename != "")
      {
        remove(temp_filename.c_str());
        remove(temp_dir.c_str());
      }
  }

  //----------------------------------------------------------------------

  bool RController::rProcessStart()
  {
    //pipe
    int child_read_pipe[2];
    int child_write_pipe[2];

    //try to create pipes
    if(pipe(child_read_pipe) < 0)
      {
        perror("pipe");
        return false;
      }
    if(pipe(child_write_pipe) < 0)
      {
        close(child_read_pipe[0]);
        close(child_read_pipe[1]);
        perror("pipe");
        return false;
      }

    //try to create child
    pid_t pid = fork();
    if(pid < 0)
      {
        cerr << "Failed to fork a child process.\n";
        perror("fork");
        return false;
      }

    //--------------------------------------------------
    //CHILD:
    else if(pid == 0)
      {
        close(child_read_pipe[1]);               //close writing side of pipe
        dup2(child_read_pipe[0], STDIN_FILENO);  //replace stdin
        close(child_read_pipe[0]);               //close old pipe read end

        close(child_write_pipe[0]);              //close reading side of pipe
        dup2(child_write_pipe[1], STDOUT_FILENO);//replace stdout
        close(child_write_pipe[1]);              //close old pipe read end

        const char * program = "R";
        execlp(program,
               program,
               "--no-save","--no-restore","--quiet",
               NULL);

        //if we get here, there was an error loading R
        cerr << "Failed to execute.\n";
        perror("execlp");
        exit(1);
      }


    //--------------------------------------------------
    //PARENT
    close(child_read_pipe[0]);  //close read end of child read pipe
    close(child_write_pipe[1]); //close write end of child write pipe
    this->read_pipe = child_write_pipe[0];
    this->write_pipe = child_read_pipe[1];
    this->child_pid = pid;
    this->connected = true;
    return true;
  }


  //----------------------------------------------------------------------


  void RController::rProcessKill()
  {
    kill(this->child_pid,9);
    this->connected = false;
  }

  //----------------------------------------------------------------------


  bool RController::start()
  {

    if(!rProcessStart())
      {
        cerr << "Failed to create R child process.\n";
        return false;
      }
    else
      {
        return true;
      }
  }

  void RController::stop()
  {
    rProcessKill();
  }

  //----------------------------------------------------------------------


  string RController::pipeReadAll()
  {
    string result = "";
    char buffer[1024];
    memset(buffer, 0, sizeof (buffer));

    // read result
    unsigned bytes = 0;
    int total_bytes = 0;
    while(true)
      {
        bytes = read(read_pipe, buffer, sizeof(buffer));
        total_bytes += bytes;
        //put the result into return string
        result += buffer;
        memset(buffer, 0, sizeof (buffer));
        
        if(bytes < sizeof(buffer))
          {
            break;
          }
      }
    this->cleared = true;
    return result;
  }

  void RController::pipeWriteCommand(string command)
  {
    command += "\n";
    const char * temp = command.c_str();
    write(write_pipe,temp,command.length());
    this->cleared = false;
  }

  void RController::sendCommand(string command, bool clear)
  {

    if(clear && !cleared)
      {
        sleep(1);
        pipeReadAll();
      }

    //replace all tokens
    // %f token
    if(regex_search(command,boost::regex("%f")))
      {
        createTempFilename();
        command = regex_replace(command,
                                boost::regex("%f"),
                                temp_filename);
      }

    pipeWriteCommand(command);
  }

  string RController::getResponse()
  {
    //read until we encounter an R prompt again
    boost::regex prompt_regexp("^> $");
    bool cont = true;
    string result,line,full_response;
    char buffer[1024];
    full_response = "";

    //continue reading from pipe until we reach a prompt
    while(cont)
      {

        result = pipeReadAll();
        full_response += result;

        // split result into lines
        stringstream temp;
        temp << result;
        // loop through lines to check if we have reached a prompt
        while(!temp.eof())
          {
            temp.getline(buffer,sizeof(buffer),'\n');
            line = buffer;
            if(regex_search(line,prompt_regexp))
              {
                cont = false; 
                break;
              }
          }
        sleep(1);
      }
      return full_response;
  }

  //----------------------------------------------------------------------
  void RController::createTempFilename()
  {
    if(temp_filename == "")
      {
        char dir[] = "/tmp/CEBLXXXXXX";
        mkdtemp(dir);
        this->temp_dir = dir;
        this->temp_filename = string(dir) + "/matrix";
      }
  }

  //----------------------------------------------------------------------
  // R Parsing

  string RController::trimResponse(string response)
  {
    boost::regex prompt_regexp("^> *");
    string ret = "";
    char buffer[1024];
    string line;
    stringstream temp;
    temp << response;
    // loop through lines adding them to new string
    int i = 0;
    while(!temp.eof())
      {
        temp.getline(buffer,sizeof(buffer),'\n');
        if(i > 0)
          {
            line = buffer;
            if(!regex_search(line,prompt_regexp))
              {
                ret += line + "\n";
              }
          }
        i++;
      }
    return ret;
  }

  ublas::matrix<double> RController::readMatrixFromTempFile()
  {
    return cppR::readTable<double>(this->temp_filename);
  }

  void RController::writeMatrixToTempFile(ublas::matrix<double> mat)
  {
    createTempFilename();
    cout << temp_filename << "\n";
    cppR::writeTable(mat,this->temp_filename);
  }

}
