/*
* CEBL : CSU EEG Brain-Computer Interface Lab
*
* Author: Jeshua Bratman - jeshuabratman@gmail.com
*
* This file is part of CEBL.
*
* CEBL is free software; you can redistribute it and/or modify it.
* We only ask that if you use our code that you cite the source in
* your project or publication.
*
* EEG Group (www.cs.colostate.edu/eeg)
* Department of Computer Science
* Colorado State University
* 
*/

#ifndef RCONTROLLER_H
#define RCONTROLLER_H
#include <cppR/cppR.hpp>
#include <string>
using std::string;

/*! \class RController
 * \author Jeshua Bratman
 *
 * Opens an R process and communicates via pipes.
 */

namespace CEBL
{
  class RController
  {
  private:
    int read_pipe;
    int write_pipe;
    int child_pid;
    bool connected;
    bool cleared;
    string temp_filename;
    string temp_dir;

    bool rProcessStart();
    void rProcessKill();
    string pipeReadAll();
    void pipeWriteCommand(string command);
    string trimResponse(string response);
    void createTempFilename();
   
  public:
    RController();
    ~RController();

    /*! Starts R process and connects pipes
      Returns success.
      
      \return success
    */
    bool start();

    /*! Attempt to close R process
     */
    void stop();


    /*! Check if we are connected to R
      
      \return 
    */
    bool isConnected()
    {
      return connected;
    }

    /*! Sends a command to R. This function will not block. Since we
      are using pipes, there's no way of knowing when R finishes processing
      a command.
      
      NOTES:command can contain a few token that will be replaced
            before sending to R:

            %f - generate a temporary filename

      \param command R command
    */
    void sendCommand(string command, bool clear = true);

    /*! Get a raw response from the R process. This function will block
      until R prompt has been encountered from standard output.
      
      \return 
    */
    string getResponse();
    

    /*! Reads matrix written to tempfile. Use %f token in
      sendCommand to send temporary filename to R.
      
      \return 
    */
    ublas::matrix<double> readMatrixFromTempFile();

    /*! Writes a matrix to a file which can later be
      read by R using %f token in sendCommand.
      \param mat 
    */
    void writeMatrixToTempFile(ublas::matrix<double> mat);



  };
}


#endif
