/*! Session.cpp
 * \author Jeshua Bratman
 *
 * Saving and loading CEBL model.
 */


#include <fstream>
#include <iostream>
#include <sstream>
#include <algorithm>
#include <boost/regex.hpp>
using namespace std;
using namespace boost;

#include "Session.hpp"
#include "../Exceptions.hpp"



//----------------------------------------------------------------------
// CONSTRUCTORS / DESTRUCTORS

//default constructor
Session::Session()
{
  saved_once = false;
  saved_filename = "";
  current_section = "default";
}
//destructor
Session::~Session()
{}

//copy constructor
Session::Session(const Session &s2)
{
  this->params = s2.params;
  this->saved_once = s2.saved_once;
  this->saved_filename = s2.saved_filename;
  this->current_section = s2.current_section;
}

//----------------------------------------------------------------------
// LOADING and SAVING


//! Load session from session file.
void Session::load(const char *filename)
{
  //open file
  ifstream is;
  is.open(filename);
  if(is.fail())
    {
      cerr << "Failed to open file " << filename << " for reading.\n";
      throw FileException("Failed to open file " + string(filename) + " for writing.\n");
    }

  //now load file
  string line;
  string current_section = "default";
  string buffer;
  stringstream channels_config;

  regex comment_regexp("^[[:space:]]*#.*");
  regex blank_regexp("^[[:space:]]*$");
  regex valid_section_regexp("^[[:space:]]*\\[(\\w*)\\][[:space:]]*");
  regex valid_param_regexp("^[[:space:]]*(\\w+)[[:space:]]*=[[:space:]]*(.+)[[:space:]]*");

  //loop through file
  while(!is.eof())
    {
      // get the next line
      getline(is, buffer);
      line = buffer;
      //check if this line is a comment
      if(regex_search(line,comment_regexp)
         ||
         regex_search(line,blank_regexp))
        {
          continue;
        }
      //otherwise extract the section and value
      bool section = regex_search(line,valid_section_regexp);
      bool param = regex_search(line,valid_param_regexp);

      //check if either search failed
      if(!section && !param)
        {
          cerr << "Session::LoadFile(): Bad line: " << line << endl;
          continue;
        }

      //SECTION
      if(section)
        {
          cmatch what;
          regex_match(line.c_str(), what, valid_section_regexp);
          current_section = what[1];
        }

      //PARAM
      if(param)
        {
          cmatch what;
          regex_match(line.c_str(), what, valid_param_regexp);
          string temp = what[2];
          params[current_section][what[1]] = decodeString(temp);
        }
    }

  saved_once = true;
  saved_filename = filename;
}

//! Save session to session file.
void Session::save(const char *filename)
{
  ofstream os;
  os.open(filename);
  if(os.fail())
    {
      cerr << "Failed to open file " << filename << " for writing.\n";
      throw FileException("Failed to open file " + string(filename) + " for writing.\n");
      return;
    }
  //send to ostream
  this->printToStream(os);

  //update members
  saved_once = true;
  saved_filename = filename;
}

//! Save to file already saved to once before.
void Session::save()
{
  if(saved_once)
    save(saved_filename.c_str());
}
//! Should the user specify where to save?
bool Session::shouldSaveAs()
{
  return !saved_once;
}

//! Send to ostream.
ostream & operator<<(ostream &os, Session &s1)
{
  s1.printToStream(os);
  return os;
}

//----------------------------------------------------------------------
// Accessors

bool Session::exists(string param, string section)
{
  if(section=="")
    section = current_section;

  if(params.count(section) > 0)
    if(params[section].count(param) > 0)
      return true;
  return false;
}




//----------------------------------------------------------------------
// Private Methods

void Session::printToStream(ostream &os)
{
  typedef map<string, string> inner;
  typedef map<string, map<string, string> > outer;
  outer::iterator oit = params.begin();

  inner::iterator iit;


  for(;oit != params.end();
      ++oit)
    {
      os << "[" << oit->first << "]\n";
      for(iit = oit->second.begin();
          iit != oit->second.end();
          ++iit)
        {
          string key = iit->first;
          string value = encodeString(iit->second);
          os << key << " = " << value << endl;
        }
    }
}


//----------------------------------------------------------------------
//decode and encode strings

string Session::encodeString(string str)
{
  std::replace(str.begin(), str.end(), '\n', char(2));
  std::replace(str.begin(), str.end(), '\t', char(3));
  std::replace(str.begin(), str.end(), ' ', char(4));
  return str;
}

string Session::decodeString(string str)
{
  std::replace(str.begin(), str.end(), char(2), '\n');
  std::replace(str.begin(), str.end(), char(3), '\t');
  std::replace(str.begin(), str.end(), char(4), ' ');
  return str;
}
