/*
* CEBL : CSU EEG Brain-Computer Interface Lab
*
* Author: Jeshua Bratman - jeshuabratman@gmail.com
*
* This file is part of CEBL.
*
* CEBL is free software; you can redistribute it and/or modify it.
* We only ask that if you use our code that you cite the source in
* your project or publication.
*
* EEG Group (www.cs.colostate.edu/eeg)
* Department of Computer Science
* Colorado State University
* 
*/

/*! Session.hpp
 * \author Jeshua Bratman
 *
 * Saving and loading CEBL model.
 */


#ifndef SESSION_H
#define SESSION_H

#include <string>
#include <sstream>
#include <map>
#include <iostream>
#include "Serialization.hpp"

using namespace std;
//forward declarations
class TabSet;


class Session
{
private:
  //internal storage of variables to save and load
  std::map<std::string, std::map<std::string, std::string> > params;
  //has this session already been saved
  bool saved_once;
  std::string saved_filename;
  std::string current_section;


  void printToStream(std::ostream &os);
  string decodeString(string str);
  string encodeString(string str);

public:
  Session();
  ~Session();
  Session(const Session &);

  //! Load session from session file.
  void load(const char *filename);
  //! Save session to session file.
  void save(const char *filename);

  //! Save session to current session file.
  void save();

  //! Should the user be prompted to save as
  bool shouldSaveAs();

  //! Add any type param to the session.
  template <typename T>
  void add(std::string param, const T& value, std::string section="")
  {
    if(section=="")
      section = current_section;
    std::stringstream os;
    boost::archive::text_oarchive oa(os);
    oa << value;

    params[section][param] = os.str();
  }
   //! Add any type param to the session.
  template <typename T>
  void add(const char *param, const T& value, std::string section="")
  {
    add(string(param),value,section);
  }

  //! easy access to add
  template <typename T>
  Session & operator()(std::string param, const T& value)
  {
    this->add(param, value);
    return *this;
  }

  //! easy access to add
  template <typename T>
  Session & operator()(const char *param,  const T& value)
  {
    this->add(param, value);
    return *this;
  }

  //! Clear contents.
  void clear()
  {
    params.clear();
  }

  //Accessors

  //! Set current section so it doesn't have to be specified every time.
  void setCurrentSection(std::string sc) { current_section = sc; }

  //! Check if a value exists.
  bool exists(std::string param, std::string section = "");

  //! get parameter
  template <typename T>
  T get(std::string param, std::string section = "")
  {
    if(section=="")
      section = current_section;

    std::stringstream is;
    is << params[section][param];
    boost::archive::text_iarchive ia(is);
    T temp;
    ia >> temp;
    return temp;
  }

  //! get into given variable
  template <typename T>
  void get(std::string param, T *v,  std::string section = "")
  {
    if(section=="")
      section = current_section;

    std::stringstream is;
    is << params[section][param];
    boost::archive::text_iarchive ia(is);
    ia >> (*v);
  }



  friend std::ostream & operator<<(std::ostream &os, Session &s1);
};

#endif
